package simonton.core;

import java.awt.*;
import java.awt.geom.*;
import java.awt.geom.Point2D.Double;
import java.util.Comparator;

import robocode.Bullet;
import simonton.guns.Gun;
import simonton.utils.*;

public class Wave {

	public Point2D.Double origin;
	public double firePower;
	public double maxEscape;
	public double speed;
	public double angle;
	public int fireTime;
	public int timeTillNext;

	public Wave(Point2D.Double origin, int fireTime, double angle) {

		this.angle = angle;
		this.origin = origin;
		this.fireTime = fireTime;
	}

	public Wave(Point2D.Double origin, Point2D.Double target, int fireTime) {

		this(origin, fireTime, Util.bearing(origin, target));
	}

	public Wave(Gun gun, int fireTime) {

		this(new Point2D.Double(gun.getX(), gun.getY()), fireTime, gun
				.getGunHeadingRadians());
		setBulletPower(gun.getNextBulletPower());
	}

	public Wave(Gun gun) {

		this(gun, (int) gun.getTime());
	}

	public void setBulletPower(double power) {

		firePower = power;
		speed = Util.getBulletSpeed(power);
		maxEscape = Util.getMaxEscape(speed);
		timeTillNext = (int) Math.ceil(Util.getGunHeat(power)
				/ Util.gunCoolingRate);
	}

	public int impactTime(Point2D.Double p) {

		return fireTime + (int) Math.ceil(origin.distance(p) / speed);
	}

	public boolean isHit(double angle, int time, Point2D.Double botLocation) {

		double travel = (time - fireTime) * speed;
		Point2D.Double lead = Util.project(origin, travel, angle);
		Point2D.Double trail = Util.project(origin, travel - speed, angle);
		return new Rectangle.Double(botLocation.x - 18, botLocation.y - 18, 36,
				36).intersectsLine(lead.x, lead.y, trail.x, trail.y);
	}

	public boolean isFactorHit(int factor, int gfZero, int binPadding,
			int time, Point2D.Double botLocation) {

		factor -= binPadding;
		double factorAngle = (factor - gfZero) * maxEscape / gfZero + angle;
		return isHit(factorAngle, time, botLocation);
	}

	/**
	 * @param p
	 * @param time
	 * @return 0 if its a hit, 1 if the bullet is still dangerous, -1 if the
	 *         bullet has passed.
	 */
	public int washCode(Point2D.Double p, int time) {
		double travel = (time - fireTime) * speed;
		double toGo = origin.distance(p) - travel;
		if (toGo > 26) {
			return 1;
		}
		if (toGo < -26 - speed) {
			return -1;
		}
		return 0;
	}

	public int getFactorIndex(double factorAngle, int gfZero, int binPadding) {
		return binPadding
				+ (int) ((1 + Util.normalize(factorAngle - angle) / maxEscape)
						* gfZero + .5);
	}

	public int getFactorIndex(Double location, int gfZero, int binPadding) {
		return getFactorIndex(Util.bearing(origin, location), gfZero,
				binPadding);
	}

	public boolean equals(Bullet b) {
		return Util.equal(speed, b.getVelocity())
				&& washCode(new Point2D.Double(b.getX(), b.getY()), Util.time) == 0;
	}

	public void onPaint(Graphics2D g, int time) {

		g.setColor(Color.BLUE);
		int r = (int) (speed * (time - fireTime));
		if (r < 0) {
			g.setColor(Color.RED);
			r = -r;
		}
		g.drawOval((int) origin.x - r, (int) origin.y - r, 2 * r, 2 * r);
		Point2D.Double lead = Util.project(origin, r, angle);
		Point2D.Double trail = Util.project(origin, r - speed, angle);
		g.drawLine((int) lead.x, (int) lead.y, (int) trail.x, (int) trail.y);
	}

	public static Wave getNextShot(Gun gun) {

		return new Wave(gun, (int) gun.getTime()
				+ (int) (Math.ceil(gun.getGunHeat() / gun.getGunCoolingRate())));
	}

	public static class TimeToImpactComparator implements Comparator<Wave> {
		private Point2D.Double impactPoint;
		
		public TimeToImpactComparator(Point2D.Double impactPoint) {
			this.impactPoint = impactPoint;
		}

		public int compare(Wave w1, Wave w2) {
			return w1.impactTime(impactPoint) - w2.impactTime(impactPoint);
		}
	}
}
