package djc;
import robocode.*;
import java.util.LinkedList;

/**
 * Target - Based on a class by Simon Parker
 *  Heading is in radians
 */
public class Target extends Object
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    /** Amount of time to consider transient data as valid */
    public static final double TRANSIENT_VALID_TIME = 4.0;
    public static final int FIRE_TIME = 25;
    public static final String INVALID_TARGET_LOCATION = "The target position is off field";
    public static final String TARGET_CANNOT_REACH_LOCATION = 
	"The target could not get to targetted position in the time allowed";
    /** Max number of positions to remember for dodge analyzer */
    public static final int MAX_HISTORY = 250;
    /** Max number of positions to remember */
    public static final int MAX_FUTURE = 20;

    /** For fireResponse - where was the target when I fired a shot */
    public static final int WHEN_FIRED = 0;
    /** For fireResponse - where was I aiming when I fired a shot */
    public static final int AIMED_AT = 1;
    /** For fireResponse - where was the target when the shot should have hit */
    public static final int ACTUAL_LOCATION = 2;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    public Coordinate position = new Coordinate();
    public Coordinate cAimHere = new Coordinate();
    public double lastTimePosition = -1;
    public double heading_rad = 0;
    public double bearing_rad = 0;
    public double energy = 100;
    public double lastTimeEnergy = -1;
    public String name = "";
    public int deathCount = 0;
    public double damageInflicted = 0; // damage inflicted by this target on me
    public double damageSustained = 0;
    public boolean isAlive = true;
    public boolean isCurrentTarget = false;
    public double lastTimeTargeted = -1;
    public double velocity = 0;
    public double avgVelocity = 0;
    public double angularVelocity = 0;
    public double avgAngVelocity = 0;
    public double lastBulletPower = 1.0;
    public double lastBulletHitTime = -1;
    public double lastBulletFromHeading = -1;
    public Coordinate lastBulletHitMeHere = new Coordinate();
    public double firedAt = 0;
    public double timesHit = 0;
    public double lastTimeFiredAt = -1;
    public static int outSurvived = 0;
    public GravPoint gp = new GravPoint();
    public TargetingStrategy currentTargetStrategy = null;
    public TargetingStrategyManager tsm = null;
    public String targetMode = "NONE";
    public static Position fireResponse[][] = new Position[MAX_HISTORY][3];
    public static String fireResponseTargetingStrat[] = new String[MAX_HISTORY];
    public int currentFirePos = 0;
    public int currentFireRespPos = -1;
    //public double expectedImpactTime = Double.MAX_VALUE;
    public StrategyBot self;
    public JiggleAnalyser ja = new JiggleAnalyser();
    public PatternAnalyser pa = new PatternAnalyser();
    public FireResponseAnalyser fa = null;

    public LinkedList completedBullets = new LinkedList();
    public static Coordinate predictedLocation[][] = 
	new Coordinate[MAX_FUTURE][TargetingStrategyManager.NUM_PREDICTIVE_METHODS];

    /**
     * Target Constructor
     *
     * @param targetName - String for the name
     */
    public Target(String targetName, StrategyBot s)
    {
	name = targetName;
	self = s;
	if(tsm == null) tsm = new TargetingStrategyManager(this);
	if(fa == null) fa = new FireResponseAnalyser(this);
    }

    public void reset()
    {
	isAlive = true;
	lastTimeEnergy = -1;
	lastTimePosition = -1;
	isCurrentTarget = false;
	lastTimeTargeted = -1;
	lastTimeFiredAt = -1;
	for(int i=0;i<MAX_FUTURE;i++) {
	    for(int j=0;j<TargetingStrategyManager.NUM_PREDICTIVE_METHODS;j++) {
		predictedLocation[i][j] = new Coordinate(-1000,-1000);
	    }
	}
    }

    /**
     * updateFromScan
     *   Data from Scanner.
     *
     * Based on code from JollyNinja
     */
    public void updateFromScan (String targetName, Coordinate targetPosition, 
				double targetVelocity, double targetHeading,
				double targetEnergy, double targetDist, 
				double targetBearing, double currentTime,
				double myEnergy, boolean bGravPoint, boolean bPosHistory)
    {
	name = targetName;
	position = targetPosition;
	avgVelocity = (velocity + targetVelocity) / 2;
	energy = targetEnergy;
	velocity = targetVelocity;
	isAlive = true;
	
	double dT = currentTime - lastTimePosition;
	double dH = heading_rad - targetHeading;
	double tmpOldAngVel = angularVelocity;

	angularVelocity = dH / dT;
	avgAngVelocity = (tmpOldAngVel + angularVelocity) / 2;
	
	heading_rad = targetHeading;
	bearing_rad = targetBearing;
	lastTimeEnergy = currentTime;
	lastTimePosition = currentTime;

	if(bGravPoint) {
	    // Gravity Point
	    gp.x = targetPosition.x;
	    gp.y = targetPosition.y;
	    gp.exponent = GravPoint.DEFAULT_TARGET_POSITION;
	    gp.strength = GravPoint.strengthForStandardTarget(self, this);
	}
	
	ja.update(targetPosition, velocity, heading_rad, currentTime);
	pa.update(targetPosition, velocity, heading_rad, (int)currentTime);

	if(self.getOthers() == 1) {
	    // update predictedLocation array
	    int futureTimeIndex = (int)(self.getTime() + MAX_FUTURE - 1);
	    futureTimeIndex = futureTimeIndex % MAX_FUTURE;
	    
	    // 0 is Stationary
	    predictedLocation[futureTimeIndex][0] = predictiveAimHere(targetDist,
								      20,
								      currentTime,
								      tsm.STATIONARY);
	    
	    // 1 is Linear
	    predictedLocation[futureTimeIndex][1] = predictiveAimHere(targetDist,
								      20,
								      currentTime,
								      tsm.LINEAR);
	    // 2 is Circular
	    predictedLocation[futureTimeIndex][2] = predictiveAimHere(targetDist,
								      20,
								      currentTime,
								      tsm.CIRCULAR);
	    // 3 is Linear Avg Vel
	    predictedLocation[futureTimeIndex][3] = predictiveAimHere(targetDist,
								      20,
								      currentTime,
								      tsm.LINEAR_AVG_VEL);
	    // 4 is Circular Avg Vel
	    predictedLocation[futureTimeIndex][4] = predictiveAimHere(targetDist,
								      20,
								      currentTime,
								      tsm.CIRCULAR_AVG_VEL);

	    // decide targeting strategy based on predictedLocation array
	    int currentTimeIndex = (int)self.getTime();
	    currentTimeIndex = currentTimeIndex % MAX_FUTURE;
	    /*self.out.print("curTimeIndex=" + currentTimeIndex + " p.x=" + (int)position.x
	      + " p.y=" + (int)position.y);*/
	    double bestDist = Double.MAX_VALUE;
	    double curDist = Double.MAX_VALUE;
	    int bestIndex = -1;

	    for(int i=0;i<TargetingStrategyManager.NUM_PREDICTIVE_METHODS; i++) {
		if(predictedLocation[currentTimeIndex][i] != null) {
		    
		    /*self.out.println(" i=" + i);*/
		    curDist = position.distanceFrom(predictedLocation[currentTimeIndex][i]);
		    /*self.out.print(" curDist=" + (int)curDist + " " +
		      (int)predictedLocation[currentTimeIndex][i].x + "," +
		      (int)predictedLocation[currentTimeIndex][i].y);*/
		    if(curDist < bestDist) {
			bestIndex = i;
			bestDist = curDist;
		    }
		}
		/*self.out.println(" bestIndex=" + bestIndex + "  bestDist=" + (int)bestDist);*/
		if(bestIndex==0) tsm.setStrategy(tsm.STATIONARY);
		if(bestIndex==1) tsm.setStrategy(tsm.LINEAR);
		if(bestIndex==2) tsm.setStrategy(tsm.CIRCULAR);
		if(bestIndex==3) tsm.setStrategy(tsm.LINEAR_AVG_VEL);
		if(bestIndex==4) tsm.setStrategy(tsm.CIRCULAR_AVG_VEL);
	    }
	}
    }

    /**
     * updateFromHitByBullet
     *   Data from a bullet that hit self.
     *   Only update energy if it is current.
     *
     * Code by Dan Cieslak
     */
    public void updateFromHitByBullet(double bulletPower, double currentTime,
				      double bulletHeading, Coordinate curPos)
    {
	damageSustained += MathHelper.getBulletDamage(bulletPower);

	double dT = currentTime - lastTimeEnergy;
	if (dT < TRANSIENT_VALID_TIME) {
	    energy += 3.0 * bulletPower;
	}

	lastBulletHitTime = currentTime;
	lastBulletFromHeading = bulletHeading;
	lastBulletHitMeHere = curPos;
	Double dAccruedDamage = (Double)self.strategyManager.accruedDamage.remove(self.currentStrategy.name);
	if(dAccruedDamage == null) {
	    self.out.println("Could not find accruedDamage for " + self.currentStrategy.name);
	    return;
	}
	self.currentStrategy.accruedDamageThisUse += MathHelper.getBulletDamage(bulletPower);
	dAccruedDamage = new Double(dAccruedDamage.doubleValue() + MathHelper.getBulletDamage(bulletPower));
	self.strategyManager.accruedDamage.put(self.currentStrategy.name, dAccruedDamage);
    }

    /**
     * updateFromHitByBullet
     *   Data from a bullet that hit target
     *   Only update energy if it is current.
     *
     * Code by Dan Cieslak
     */
    public void updateFromBulletHit(double bulletPower, 
				    double currentTime, 
				    double targetEnergy,
				    Coordinate curPos, Coordinate myPos)
    {
	damageInflicted += MathHelper.getBulletDamage(bulletPower);

	energy = targetEnergy;
	//position = curPos;
	lastTimeEnergy = currentTime;
	hitTarget(currentTime, myPos.distanceFrom(curPos), myPos.headingTo(curPos));
    }

    public Coordinate getEstimatedPosition(double currentTime)
    {
	return new Coordinate(position.x + velocity * (currentTime - lastTimePosition) * Math.sin(heading_rad),
			      position.y + velocity * (currentTime - lastTimePosition) * Math.cos(heading_rad));
    }
    
    /**
     * Eventually, this do this through a targetingStrategy class
     * optionally using data from last bullet (if it was recent)
     *
     * From JollyNinja
     */
    public double getEstimatedVelocity(double currentTime)
    {
	return velocity;
    }

    
    /**
     * Eventually, this do this through a targetingStrategy class
     * optionally using data from last bullet (if it was recent)
     *
     * From JollyNinja
     */
    public double getEstimatedHeading(double currentTime)
    {
	return heading_rad;
    }

    /**
     * Eventually, this do this through a targetingStrategy class
     * optionally using data from last bullet (if it was recent)
     *
     * From JollyNinja
     */
    public double getEstimatedAngVel(double currentTime)
    {
	return angularVelocity;
    }

    /**
     * Eventually, this do this through a targetingStrategy class
     * optionally using data from last bullet (if it was recent).
     * <br>
     * Get this out of DefaultTacticalStrategy.setGunRotation()
     * and into someplace sensible.
     * <br>
     * Author Dan Cieslak
     *
     * @param myPos - my location
     * @param firepower - how hard I am shooting
     * @param estDist - about how far away
     * @param currentTime - now
     * @return bearing in radians
     * @exception Exception if estPosition off field
     * @exception Exception if estPosition too far for a robot to
     *     reach based on robocode physics.
     */
    public double aimHere(Coordinate myPos, double firePower,
			  double estDist, double currentTime,
			  double fieldWidth, double fieldHeight) 
			  throws Exception
    {
	boolean bKeepOnField = false;

	double bulletVelocity = MathHelper.getShotVelocity(firePower);
	double estBulletFlightTime = estDist / bulletVelocity;
	double estVelocity = getEstimatedVelocity(currentTime);
	double estAngVelocity = getEstimatedAngVel(currentTime);
	double estHeading = getEstimatedHeading(currentTime);
	double elapsedTime = currentTime + estBulletFlightTime - lastTimePosition;
	double predBulletFlightTime = 0;  // Predicted flight time
	Coordinate estPosition = new Coordinate(-1000,-1000);
	boolean bPatternMatched = false;

	// Attempt pattern matching...may need to start threading this stuff
	PatternTargetingStrategy pts = new PatternTargetingStrategy(this, pa);
	estPosition = pts.predictCoordinate(currentTime, estBulletFlightTime);
	bPatternMatched = pts.isPredictionReliable;
	if(!bKeepOnField) bKeepOnField = bPatternMatched;

	if(!bPatternMatched) {
	    if(ja.isJiggling && ja.isOscilating) {  // use jiggleTargetStrat
		if(ja.self == null) ja.self = self;
		JiggleTargetingStrategy jts = new JiggleTargetingStrategy(this, ja);
		bKeepOnField = true;
		estPosition = jts.predictCoordinate(currentTime, estBulletFlightTime);
		targetMode = TargetingStrategyManager.JIGGLE;
	    } else {
		double localEstDist = estDist;
		double localEstBFT = estBulletFlightTime;
		int maxIterations = Math.max(1, 10 - self.skippedTurnsResetForOtherDeath);
		double closeness = 10 + 5 * self.skippedTurnsResetForOtherDeath;

		for(int i=0; i<maxIterations; i++) {
		    estPosition = currentTargetStrategy.predictCoordinate(currentTime,
									  localEstBFT);
		    if(Math.abs(localEstDist - self.myPos.distanceFrom(estPosition)) < closeness) break;
		    localEstDist = self.myPos.distanceFrom(estPosition);
		    localEstBFT = localEstDist / bulletVelocity;
		}
		targetMode = currentTargetStrategy.name;
	    }
	    //self.out.println(" estD=" + (int)estDist + " preD=" + (int)self.myPos.distanceFrom(estPosition));
	} else {
	    targetMode = TargetingStrategyManager.PATTERN;
	}
	
	fa.analyse();
	if(fa.isDodging) {
	    Coordinate estDodgePosition = fa.adjustAim(estPosition);
	    if(!estDodgePosition.offField()) estPosition = estDodgePosition;
	}

	// If the coordinate to aim towards is off the field, throw an exception.
	if(estPosition.offField()) {
	    if(bKeepOnField) {
		estPosition = Coordinate.limitToBattleField(estPosition);
	    } else {
		throw new Exception(INVALID_TARGET_LOCATION);
	    }
	}
	
	// Validate the target can reach this spot.
	double targetMoveTime = position.distanceFrom(estPosition) / MathHelper.MAX_ROBOT_VELOCITY;
	if(targetMoveTime > elapsedTime) {
	    // Think about adding logic that if target is close enough, simply
	    // aim directly at the target.
	    throw new Exception(TARGET_CANNOT_REACH_LOCATION);
	}

	cAimHere.x = estPosition.x;
	cAimHere.y = estPosition.y;

	return myPos.headingTo(estPosition);
    }

    private Coordinate predictiveAimHere(double estDist, double estBulletFlightTime,
					 double currentTime, String stratName)
    {
	Coordinate estPosition = new Coordinate(-1000,-1000);
	double bulletVelocity = estDist / estBulletFlightTime;
	double localEstDist = estDist;
	double localEstBFT = estBulletFlightTime;
	int maxIterations = 10; //Math.max(1, 10 - self.skippedTurnsResetForOtherDeath);
	double closeness = 10 + 5 * self.skippedTurnsResetForOtherDeath;
	TargetingStrategy ts = (TargetingStrategy)tsm.availableStrategies.get(stratName);
	if(ts == null) return new Coordinate(-1000,-1000);
	if(stratName == tsm.STATIONARY) maxIterations = 1; // no point iterating

	for(int i=0; i<maxIterations; i++) {
	    estPosition = ts.predictCoordinate(currentTime, localEstBFT);
	    if(Math.abs(localEstDist - self.myPos.distanceFrom(estPosition)) < closeness) break;
	    localEstDist = self.myPos.distanceFrom(estPosition);
	    localEstBFT = localEstDist / bulletVelocity;
	}
	estPosition = Coordinate.limitToBattleField(estPosition);
	return estPosition;
    }

    /**
     * Should we fire at the target?
     *
     * For shotPower >= 1 - must hit 1 of every 7 shots.
     * For shotPower <  1 - must hit 1 of every 6 shots.
     *
     * If hit rates are worse than this, randomly shoot
     * every FIRE_TIME ticks.
     */
    public boolean shouldFire(double shotPower)
    {
	if(self.getOthers() > 1) return true;  // Always shoot in melee
	if(firedAt <= 6) return true;          // Get at least 7 shots out
	// Ram disabled bots
	if(energy < .1 && self.getEnergy() > 1 && self.getTime() - lastTimeFiredAt < 150) return false;
	if(self.myPos.distanceFrom(this.position) < 75) return true; // Close quarters

	if(shotPower >= 1) {
	    if(getHitRate() > 1.0 / 7.0) return true;
	    else {
		if(self.getTime() - lastTimeFiredAt > FIRE_TIME) return true;
		// In danger of losing on damage score (should take total wins and losses into account)
		if(damageInflicted + self.survivalBonus * outSurvived - damageSustained < 0) 
		    return self.random.nextBoolean();
		else return false;
	    }
	} else {
	    if(getHitRate() > 1.0 / 6.0) return true;
	    else {
		if(self.getTime() - lastTimeFiredAt > FIRE_TIME) return true;
		// In danger of losing on damage score (should take total wins and losses into account)
		if(damageInflicted + self.survivalBonus * outSurvived - damageSustained < 0) 
		    return self.random.nextBoolean(); 
		else return false;
	    }
	}
    }

    /**
     * Marks target as selected.  Later will adjust gp.power for target
     */
    public void selectTarget(double curTime)
    {
	isCurrentTarget = true;
	lastTimeTargeted = curTime;
    }

    /**
     * Marks target as deselected.  Later will adjust gp.power for target
     */
    public void deselectTarget()
    {
	isCurrentTarget = false;
    }

    /** Records the shot */
    public void firedAtTarget(double bulletPower, double dist, double currentTime,
			      Coordinate myPos)
    {
	double shotTime = MathHelper.getShotVelocity(bulletPower);
	firedAt += 1;
	currentTargetStrategy.shotsFired += 1;
	lastTimeFiredAt = currentTime;
	//expectedImpactTime = currentTime + shotTime;
	Coordinate c = new Coordinate(position.x, position.y);
	Position p = new Position(position.x, position.y,
				  currentTime, velocity,
				  angularVelocity, heading_rad,
				  avgVelocity, avgAngVelocity,
				  dist, myPos.headingTo(c));
	Position p1 = new Position(cAimHere.x, cAimHere.y,
				   currentTime, velocity,  // Only care about x, y for this
				   angularVelocity, heading_rad,
				   avgVelocity, avgAngVelocity,
				   dist, myPos.headingTo(cAimHere));
	fireResponse[currentFirePos][WHEN_FIRED] = p;           // Where target started
	fireResponse[currentFirePos][AIMED_AT] = p1;            // Shot at this location
	fireResponse[currentFirePos][ACTUAL_LOCATION] = null;   // Where target winds up
	fireResponseTargetingStrat[currentFirePos] = targetMode;
	currentFirePos = (currentFirePos + 1) % MAX_HISTORY;
    }

    /** Records the hit */
    public void hitTarget(double currentTime, double targetDist, double targetBearing)
    {
	timesHit += 1;
	currentTargetStrategy.timesHit += 1;

	//self.out.println("Hit a target...");
    }

    /** Returns the hit-rate */
    public double getHitRate()
    {
	if(firedAt < 1) return 1.0;
	else return timesHit / firedAt ;
    }

    public void updateCompletedBullet(AdvancedBullet ab)
    {
	tsm.updateCompletedBullet(ab);
	completedBullets.add(ab);
	if(ab.fireRespPos != -1) {
	    if(fireResponse[ab.fireRespPos][Target.ACTUAL_LOCATION] == null) {
		Position p = new Position();
		p.x = position.x;
		p.y = position.y;
		fireResponse[ab.fireRespPos][Target.ACTUAL_LOCATION] = p;
		currentFireRespPos = Math.max(currentFireRespPos, ab.fireRespPos);
	    }
	}	
    }
}
