// Chord by Stelokim
// 20071019

package stelo;

import robocode.*;
import robocode.util.Utils;
import stelo.Moojuk.EnemyWave;

import java.awt.geom.*;     // for Point2D's
import java.util.ArrayList; // for collection of waves
import java.awt.Color;

public class Chord extends AdvancedRobot {
    public static int BINS = 47;
//    public static double _surfStats[] = new double[BINS]; // we'll use 47 bins
    private static double _surfStats[][][][] = new double[GFTWave.DISTANCE_INDEXES][GFTWave.VELOCITY_INDEXES][GFTWave.VELOCITY_INDEXES][BINS];
    
    public Point2D.Double _myLocation;     // our bot's location
    public Point2D.Double _enemyLocation;  // enemy bot's location

    public ArrayList _enemyWaves;
    public ArrayList _surfDirections;
    public ArrayList _surfAbsBearings;
	
	private static double lateralDirection;
	private static double lastEnemyVelocity;

    // We must keep track of the enemy's energy level to detect EnergyDrop,
    // indicating a bullet is fired
    public static double _oppEnergy = 100.0;

    // This is a rectangle that represents an 800x600 battle field,
    // used for a simple, iterative WallSmoothing method (by Kawigi).
    // If you're not familiar with WallSmoothing, the wall stick indicates
    // the amount of space we try to always have on either end of the tank
    // (extending straight out the front or back) before touching a wall.
    public static Rectangle2D.Double _fieldRect
        = new java.awt.geom.Rectangle2D.Double(18, 18, 764, 564);
    private static double lastEnemyEnergy;
    private static double enemyBulletPower = 0.1;
    private static double lastMyVelocity;
    
    public void run() {
		setColors(Color.BLUE, Color.BLUE, Color.WHITE);
		lateralDirection = 1;
		lastEnemyVelocity = 0;
		
        _enemyWaves = new ArrayList();
        _surfDirections = new ArrayList();
        _surfAbsBearings = new ArrayList();

        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);

        do {
            // basic mini-radar code
            turnRadarRightRadians(Double.POSITIVE_INFINITY);
        } while (true);
    }

    public void onScannedRobot(ScannedRobotEvent e) {
        _myLocation = new Point2D.Double(getX(), getY());

        double lateralVelocity = getVelocity()*Math.sin(e.getBearingRadians());
        double absBearing = e.getBearingRadians() + getHeadingRadians();
		double enemyDistance = e.getDistance();
		double enemyVelocity = e.getVelocity();
		
        setTurnRadarRightRadians(Utils.normalRelativeAngle(absBearing - getRadarHeadingRadians()) * 2);

		double bulletPower = 1.9;
		if (e.getDistance() < 50) bulletPower = 3.0;
		bulletPower = Math.min(getEnergy(), Math.min(bulletPower, e.getEnergy() / 5.0));
		bulletPower = limit(0.1, bulletPower, 3.0);
        
        _surfDirections.add(0,
            new Integer((lateralVelocity >= 0) ? 1 : -1));
        _surfAbsBearings.add(0, new Double(absBearing + Math.PI));

        // check energyDrop
        double energyDrop = lastEnemyEnergy - e.getEnergy();
        if (energyDrop < 3.01 && energyDrop > 0.09
            && _surfDirections.size() > 2) {
        	enemyBulletPower = energyDrop;
//            EnemyWave ew = new EnemyWave();
            EnemyWave ew = new EnemyWave(this.getVelocity(), lastMyVelocity, enemyDistance);
            ew.fireTime = getTime() - 1;
            ew.bulletVelocity = bulletVelocity(enemyBulletPower);
            ew.distanceTraveled = bulletVelocity(enemyBulletPower);
            ew.direction = ((Integer)_surfDirections.get(2)).intValue();
            ew.directAngle = ((Double)_surfAbsBearings.get(2)).doubleValue();
            ew.fireLocation = (Point2D.Double)_enemyLocation.clone(); // last tick

            _enemyWaves.add(ew);
        }

        _oppEnergy = e.getEnergy();

        // update after EnemyWave detection, because that needs the previous
        // enemy location as the source of the wave
        _enemyLocation = (Point2D.Double) project(_myLocation, absBearing, e.getDistance());

        updateWaves();
        doSurfing();

		if (enemyVelocity != 0) {
			lateralDirection = Math.signum(enemyVelocity * Math.sin(e.getHeadingRadians() - absBearing));
		}
		GFTWave wave = new GFTWave(this, bulletPower);
		wave.gunLocation = new Point2D.Double(getX(), getY());
		GFTWave.targetLocation = project(wave.gunLocation, absBearing, enemyDistance);
		wave.lateralDirection = lateralDirection;
		wave.setSegmentations(enemyDistance, enemyVelocity, lastEnemyVelocity);
		wave.bearing = absBearing;
		setTurnGunRightRadians(Utils.normalRelativeAngle(absBearing - getGunHeadingRadians() + wave.mostVisitedBearingOffset()));
		setFire(bulletPower);
		if (getEnergy() >= bulletPower) {
			addCustomEvent(wave);
		}
		setTurnRadarRightRadians(Utils.normalRelativeAngle(absBearing - getRadarHeadingRadians()) * 2);
		
		lastEnemyVelocity = enemyVelocity;
		lastEnemyEnergy = e.getEnergy();
		lastMyVelocity = getVelocity();
    }

    public void updateWaves() {
        for (int x = 0; x < _enemyWaves.size(); x++) {
            EnemyWave ew = (EnemyWave)_enemyWaves.get(x);

            ew.distanceTraveled = (getTime() - ew.fireTime) * ew.bulletVelocity;
            if (ew.distanceTraveled >
                _myLocation.distance(ew.fireLocation) + 50) {
	
//				logHit(ew, _myLocation); // flatten
                _enemyWaves.remove(x);
                x--;
            }
        }
    }

    public EnemyWave getClosestSurfableWave() {
        double closestDistance = 50000; // I juse use some very big number here
        EnemyWave surfWave = null;

        for (int x = 0; x < _enemyWaves.size(); x++) {
            EnemyWave ew = (EnemyWave)_enemyWaves.get(x);
            double distance = _myLocation.distance(ew.fireLocation)
                - ew.distanceTraveled;

            if (distance > ew.bulletVelocity && distance < closestDistance) {
                surfWave = ew;
                closestDistance = distance;
            }
        }

        return surfWave;
    }

    // Given the EnemyWave that the bullet was on, and the point where we
    // were hit, calculate the index into our stat array for that factor.
    public static int getFactorIndex(EnemyWave ew, Point2D.Double targetLocation) {
        double offsetAngle = (absoluteBearing(ew.fireLocation, targetLocation)
            - ew.directAngle);
        double factor = Utils.normalRelativeAngle(offsetAngle)
            / maxEscapeAngle(ew.bulletVelocity) * ew.direction;

        return (int)limit(0,
            (factor * ((BINS - 1) / 2)) + ((BINS - 1) / 2),
            BINS - 1);
    }

    // Given the EnemyWave that the bullet was on, and the point where we
    // were hit, update our stat array to reflect the danger in that area.
    public void logHit(EnemyWave ew, Point2D.Double targetLocation) {
        int index = getFactorIndex(ew, targetLocation);

        for (int x = 0; x < BINS; x++) {
            // for the spot bin that we were hit on, add 1;
            // for the bins next to it, add 1 / 2;
            // the next one, add 1 / 5; and so on...
            ew.buffer[x] += 1.0 / (Math.pow(index - x, 2) + 1);
            EnemyWave.fastBuffer[x] += 1.0 / (Math.pow(index - x, 2) + 1);
        }
    }

    public void onHitByBullet(HitByBulletEvent e) {
        // If the _enemyWaves collection is empty, we must have missed the
        // detection of this wave somehow.
        if (!_enemyWaves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(
                e.getBullet().getX(), e.getBullet().getY());
            EnemyWave hitWave = null;

            // look through the EnemyWaves, and find one that could've hit us.
            for (int x = 0; x < _enemyWaves.size(); x++) {
                EnemyWave ew = (EnemyWave)_enemyWaves.get(x);

                if (Math.abs(ew.distanceTraveled -
                    _myLocation.distance(ew.fireLocation)) < 50
                    && Math.round(bulletVelocity(e.getBullet().getPower()) * 10)
                       == Math.round(ew.bulletVelocity * 10)) {
                    hitWave = ew;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation);

                // We can remove this wave now, of course.
                _enemyWaves.remove(_enemyWaves.lastIndexOf(hitWave));
            }
        }
    }    
    public void onBulletHitBullet(BulletHitBulletEvent e) {
        // If the _enemyWaves collection is empty, we must have missed the
        // detection of this wave somehow.
    	Bullet bullet = e.getHitBullet();
        if (!_enemyWaves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(
            		bullet.getX(), bullet.getY());
            EnemyWave hitWave = null;

            // look through the EnemyWaves, and find one that could've hit us.
            for (int x = 0; x < _enemyWaves.size(); x++) {
                EnemyWave ew = (EnemyWave)_enemyWaves.get(x);

                if (Math.abs(ew.distanceTraveled -
                    _myLocation.distance(ew.fireLocation)) < 50
                    && Math.round(bulletVelocity(bullet.getPower()) * 10)
                       == Math.round(ew.bulletVelocity * 10)) {
                    hitWave = ew;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation);

                // We can remove this wave now, of course.
                _enemyWaves.remove(_enemyWaves.lastIndexOf(hitWave));
            }
        }
    }

    // CREDIT: mini sized predictor from Apollon, by rozu
    // http://robowiki.net?Apollon
    public Point2D.Double predictPosition(EnemyWave surfWave, int direction) {
    	Point2D.Double predictedPosition = (Point2D.Double)_myLocation.clone();
    	
    	double predictedVelocity = getVelocity();
    	double predictedHeading = getHeadingRadians();
    	double maxTurning, moveAngle, moveDir;

        int counter = 0; // number of ticks in the future
        boolean intercepted = false;
    	if (direction == 0) { // brake
    		while (predictedVelocity != 0.0) {
    			double nextVelocity = predictedVelocity - Math.signum(predictedVelocity) * 2.0;
    			if (nextVelocity * predictedVelocity < 0.0) nextVelocity = 0.0;
        		predictedVelocity = nextVelocity;
        		// calculate the new predicted position
        		predictedPosition = (Point2D.Double) project(predictedPosition, predictedHeading, predictedVelocity);
   			
    		}
    		return predictedPosition;
    	}


    	do {
    		moveAngle =
                wallSmoothing(predictedPosition, absoluteBearing(surfWave.fireLocation,
                predictedPosition) + (direction * (FAR_HALF_PI)), direction)
                - predictedHeading;
    		moveDir = 1;

    		if(Math.cos(moveAngle) < 0) {
    			moveAngle += Math.PI;
    			moveDir = -1;
    		}

    		moveAngle = Utils.normalRelativeAngle(moveAngle);

    		// maxTurning is built in like this, you can't turn more then this in one tick
    		maxTurning = Math.PI/720d*(40d - 3d*Math.abs(predictedVelocity));
    		predictedHeading = Utils.normalRelativeAngle(predictedHeading
                + limit(-maxTurning, moveAngle, maxTurning));

    		// this one is nice ;). if predictedVelocity and moveDir have
            // different signs you want to breack down
    		// otherwise you want to accelerate (look at the factor "2")
    		predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
    		predictedVelocity = limit(-8, predictedVelocity, 8);

    		// calculate the new predicted position
    		predictedPosition = (Point2D.Double) project(predictedPosition, predictedHeading, predictedVelocity);

            counter++;

            if (predictedPosition.distance(surfWave.fireLocation) <
                surfWave.distanceTraveled + (counter * surfWave.bulletVelocity)
                + surfWave.bulletVelocity) {
                intercepted = true;
            }
    	} while(!intercepted && counter < 500);

    	return predictedPosition;
    }

    public double checkDanger(EnemyWave surfWave, int direction) {
        int index = getFactorIndex(surfWave,
            predictPosition(surfWave, direction));
        
        double sum = 0.0;
        for (int i = 0; i < BINS; i++) {
        	sum += surfWave.buffer[i];
        }
        if (sum == 0.0) {
//        	System.out.println("sum0");
        	return EnemyWave.fastBuffer[index];
        }

        return surfWave.buffer[index];
    }

	private static final double FAR_HALF_PI = 1.3;
    public void doSurfing() {
        EnemyWave surfWave = getClosestSurfableWave();

        if (surfWave == null) { return; }

        double leastDanger = Double.POSITIVE_INFINITY;
        double dangerLeft = checkDanger(surfWave, -1);
        double dangerRight = checkDanger(surfWave, 1);
        double dangerMiddle = checkDanger(surfWave, 0);

        double goAngle = absoluteBearing(surfWave.fireLocation, _myLocation);
        
        if (dangerLeft < dangerRight) {
            goAngle = wallSmoothing(_myLocation, goAngle - (FAR_HALF_PI), -1);
            leastDanger = dangerLeft;
        } else {
            goAngle = wallSmoothing(_myLocation, goAngle + (FAR_HALF_PI), 1);
            leastDanger = dangerRight;
        }
        if (dangerMiddle < leastDanger) {
        	setAhead(0.0);
        	return;
        }

        setBackAsFront(this, goAngle);
    }

    // This can be defined as an inner class if you want.
    static class EnemyWave {
        Point2D.Double fireLocation;
        long fireTime;
        double bulletVelocity, directAngle, distanceTraveled;
        int direction;
		double[] buffer;
		static double[] fastBuffer = new double[BINS];

        public EnemyWave(double velocity, double lastVelocity, double distance) {
			int velocityIndex = (int) Math.abs(velocity / 2);
			int lastVelocityIndex = (int) Math.abs(lastVelocity / 2);
			int distanceIndex = (int)(distance / (GFTWave.MAX_DISTANCE / GFTWave.DISTANCE_INDEXES));
			buffer = _surfStats[distanceIndex][velocityIndex][lastVelocityIndex];
		}        
    }

    // CREDIT: Iterative WallSmoothing by Kawigi
    //   - return absolute angle to move at after account for WallSmoothing
    // robowiki.net?WallSmoothing
    public double wallSmoothing(Point2D.Double botLocation, double angle, int orientation) {
        while (!_fieldRect.contains(project(botLocation, angle, 160))) {
            angle += orientation*0.05;
        }
        return angle;
    }

    public static double limit(double min, double value, double max) {
        return Math.max(min, Math.min(value, max));
    }

    public static double bulletVelocity(double power) {
        return (20D - (3D*power));
    }

    public static double maxEscapeAngle(double velocity) {
        return Math.asin(8.0/velocity);
    }

    public static void setBackAsFront(AdvancedRobot robot, double goAngle) {
        double angle =
            Utils.normalRelativeAngle(goAngle - robot.getHeadingRadians());
        if (Math.abs(angle) > (Math.PI/2)) {
            if (angle < 0) {
                robot.setTurnRightRadians(Math.PI + angle);
            } else {
                robot.setTurnLeftRadians(Math.PI - angle);
            }
            robot.setBack(100);
        } else {
            if (angle < 0) {
                robot.setTurnLeftRadians(-1*angle);
           } else {
                robot.setTurnRightRadians(angle);
           }
            robot.setAhead(100);
        }
    }

	static class GFTWave extends Condition {
		static Point2D targetLocation;
	
		double bulletPower;
		Point2D gunLocation;
		double bearing;
		double lateralDirection;
	
		private static final double MAX_DISTANCE = 1000;
		private static final int DISTANCE_INDEXES = 5;
		private static final int VELOCITY_INDEXES = 5;
		private static final int GUN_HEAT_INDEXES = 2;
		private static final int BINS = 25;
		private static final int MIDDLE_BIN = (BINS - 1) / 2;
		private double maxEscapeAngle;
		private double binWidth;
		
		private static int[][][][][] statBuffers = new int[DISTANCE_INDEXES][VELOCITY_INDEXES][VELOCITY_INDEXES][GUN_HEAT_INDEXES][BINS];
	
		private int[] buffer;
		private static int[] fastBuffer = new int[BINS];;
		
		private AdvancedRobot robot;
		private double distanceTraveled;
		
		GFTWave(AdvancedRobot _robot, double _bulletPower) {
			this.robot = _robot;
			bulletPower = _bulletPower;
			maxEscapeAngle = maxEscapeAngle(bulletVelocity(bulletPower));
			binWidth = maxEscapeAngle / (double)MIDDLE_BIN;
		}
		
		public boolean test() {
			distanceTraveled += bulletVelocity(bulletPower); // advance
			// has arrived?
			if (distanceTraveled > gunLocation.distance(targetLocation) - 18) {
				double angle = Utils.normalRelativeAngle(absoluteBearing(gunLocation, targetLocation) - bearing);
				int bin = (int)Math.round((angle /
						(lateralDirection * binWidth)) + MIDDLE_BIN);
				bin = (int) limit(0, bin, BINS - 1);
				buffer[bin]++;
				fastBuffer[bin]++;
				robot.removeCustomEvent(this);
			}
			return false;
		}
	
		double mostVisitedBearingOffset() {
			return (lateralDirection * binWidth) * (mostVisitedBin() - MIDDLE_BIN);
		}
		
		void setSegmentations(double distance, double velocity, double lastVelocity) {
			int distanceIndex = (int)(distance / (MAX_DISTANCE / DISTANCE_INDEXES));
			int velocityIndex = (int)Math.abs(velocity / 2);
			int lastVelocityIndex = (int)Math.abs(lastVelocity / 2);
			int gunHeatIndex = 0;
			if (robot.getGunHeat() >= robot.getGunCoolingRate()) gunHeatIndex = 1;
//			System.out.println(gunHeatIndex);
			buffer = statBuffers[distanceIndex][velocityIndex][lastVelocityIndex][gunHeatIndex];
		}
	
		private int mostVisitedBin() {
			int mostVisited = MIDDLE_BIN;
			for (int i = 0; i < BINS; i++) {
				if (buffer[i] > buffer[mostVisited]) {
					mostVisited = i;
				}
			}
			
			if (buffer[mostVisited] < 5) {
				// use fast buffer
				System.out.println("fast");
				for (int i = 0; i < BINS; i++) {
					if (fastBuffer[i] > fastBuffer[mostVisited]) {
						mostVisited = i;
					}
				}				
			}
			return mostVisited;
		}	
	}
	
	static Point2D project(Point2D sourceLocation, double angle, double length) {
		return new Point2D.Double(sourceLocation.getX() + Math.sin(angle) * length,
				sourceLocation.getY() + Math.cos(angle) * length);
	}
	
	static double absoluteBearing(Point2D source, Point2D target) {
		return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
	}
}

						