package sgp;
import robocode.*;
import java.util.*;
import java.io.*;
import java.lang.String;

/**
 * Target - a class by Simon Parker
 */

public class Target extends Object
{
	private JollyNinja robot;
	public Coordinate position = new Coordinate();
	public double heading_deg = 0;
	public double energy = 100;
	public double energyDeltaEstimate = 0;
	public String name = "";
	public String module = "";
	public int deathCount = 0;
	public double damageInflicted = 0; //damage inflicted by this target on JollyNinja
	public double damageSustained = 0;
	public double lastTimeScanned = -1;
	public boolean isAlive = true;
	public double velocity = 0;
	public double lastBulletPower = 1.0;
	public double lastBulletHitTime = -1;
	public double angularVelocity = 0;
	public double closestRobotDistance = 10000;
	public Coordinate closestRobotPosition = new Coordinate(Environment.battleFieldWidth / 2, Environment.battleFieldHeight / 2);
	public EnemyBullet bullet = null;
	public EnemyBulletList bulletList = null;
	public RobotBulletList robotBulletList = null;
	private MovementLog movementLog= null;

	public final int MAX_IMPACT_POINTS = 128;
	private Impact[] impactArray = new Impact[MAX_IMPACT_POINTS];
	private int currentImpactArrayIndex = 0;
	private Intercept circularIntercept = new CircularIntercept();
	private Intercept jiggleIntercept = null;
	public JiggleAnalyser jiggleAnalyser = null;
	public LinearPredictorAnalyser linearPredictorAnalyser = null;
	public PatternAnalyser patternAnalyser = null;
	public AverageLinearAnalyser averageLinearAnalyser = null;
	public AntiDodgeAnalyser antiDodgeAnalyser = null;
	public PolarPatternAnalyser polarPatternAnalyser = null;
	
	public GunManager gunManager = null;
	
	public InterceptManager interceptManager = null;
	private int currentDodgeStrategy = InterceptManager.JIGGLE_OSC_PLUS_PATTERN_WHEN_VALID_CIRCULAR;
	
	HitRateArray hitRateArray = null;
	
	public Target (String moduleName, String targetName, JollyNinja theRobot)
	{
		robot = theRobot;
		bulletList = new EnemyBulletList(robot);
		robotBulletList = new RobotBulletList();

		name = targetName;
		module = moduleName;

		for (int i = 0; i < MAX_IMPACT_POINTS; i++)
		{
			impactArray[i] = new Impact();
		}

		if (jiggleAnalyser == null)
		{
			jiggleAnalyser = new JiggleAnalyser();
		}
		//???linearPredictorAnalyser = new LinearPredictorAnalyser(robot);

		if (patternAnalyser == null)
		{
			patternAnalyser = new PatternAnalyser();
		}
		
		averageLinearAnalyser = new AverageLinearAnalyser(50);
		
		if (antiDodgeAnalyser == null)
		{
			antiDodgeAnalyser = new AntiDodgeAnalyser();
		}
		
		if (hitRateArray == null)
		{
			hitRateArray = new HitRateArray(0.2);
		}
		
		polarPatternAnalyser = new PolarPatternAnalyser();

		gunManager = new GunManager(this);
		
		interceptManager = new InterceptManager
		(
			robot.jiggleAnalyser, robot.patternAnalyser, 
			robot.averageLinearAnalyser, robot.antiDodgeAnalyser, null, null
		);
		
		reset();

		//movementLog = new MovementLog(robot);
	}

	public void reset()
	{
//		robot.out.println("Resetting target, constructed " + constructionCount);
		isAlive = true;
		lastTimeScanned = -1;
		lastBulletHitTime = -1;
		closestRobotDistance = 10000;
		
		robotBulletList = new RobotBulletList();
		
		jiggleAnalyser.reset();
		//???linearPredictorAnalyser.reset();
		patternAnalyser.reset();
		averageLinearAnalyser.reset();
		antiDodgeAnalyser.reset();
		polarPatternAnalyser.reset();
		
		gunManager.reset(module, name);

		damageInflicted = 0; //damage inflicted by this target on JollyNinja
		damageSustained = 0;

		printDodgeInfo();
		
		gunManager.load(module, name);
	}


	public void update
	(
		ScannedRobotEvent e,
		double currentTime
	)
	{
		name = e.getName();
		velocity = e.getVelocity();
		isAlive = true;

		double angle_rad = Environment.getHeadingRadians() + e.getBearingRadians();
		double distance = e.getDistance();
		double relativeHeading_deg = Math.toDegrees(angle_rad - e.getHeadingRadians());

		double currentTargetPosX = Environment.getX() + distance * Math.sin(angle_rad);
		double currentTargetPosY = Environment.getY() + distance * Math.cos(angle_rad);

		position = new Coordinate(currentTargetPosX, currentTargetPosY);

		
		Coordinate robotPosition = Environment.getRobotPosition();

		double dT = currentTime - lastTimeScanned;
		double dH = Strategy.normalRelativeAngle(e.getHeading() - heading_deg);

		if ((dT != 0) && (dT < 4))
		{
			angularVelocity = dH / dT;
		}
		else
		{
			angularVelocity = 0.0;
		}

		double dE = energy - e.getEnergy();
		final int N = 60;
		energyDeltaEstimate = ((N - 1) * energyDeltaEstimate + dE) / N;

		boolean isFiring = ((dE >= 0.1) && (dE <= 3.0) && (dT <= 3)) || (dT > 3);
		if (isFiring)
		{
			dE = Math.min(3.0, dE);

			//assume that a bullet has been fired at me using a firing algorithm
			lastBulletPower = dE;
			
			//get intercept
			Intercept intercept = interceptManager.getIntercept
			(
				position, robotPosition, Environment.getHeading(), 
				Environment.getVelocity(), robot.angularVelocity, 
				lastBulletPower, 
				//???InterceptManager.PATTERN
				currentDodgeStrategy
			);
			
			hitRateArray.handleBulletFired(currentDodgeStrategy);

			if (intercept != null)
			{
				bulletList.add
				(
					new EnemyBullet
					(
						position, 
						currentTime, 
						lastBulletPower, 
						intercept.bulletHeading_deg
					)
				);
			}
			
			robot.patternAnalyser.notifyRobotBulletFired();
			robot.antiDodgeAnalyser.notifyRobotBulletFired();
		}

		energy = e.getEnergy();
		heading_deg = e.getHeading();
		lastTimeScanned = currentTime;

		jiggleAnalyser.update(position, velocity, heading_deg);
		//???linearPredictorAnalyser.update(position, velocity, heading_deg);
		patternAnalyser.update(position, velocity, heading_deg, (int)Math.round(currentTime));
		averageLinearAnalyser.update(position, velocity, heading_deg);
		antiDodgeAnalyser.update(position, velocity, heading_deg, (int)Math.round(currentTime));
		polarPatternAnalyser.update(position, relativeHeading_deg, velocity, distance, currentTime);
		
		robotBulletList.removeOldBullets(position);
						
		if (movementLog != null)
		{
			MovementRecord record = new MovementRecord((int)currentTime, position, velocity, heading_deg);
			movementLog.add(record);
		}
	}

	public void updateImpactPoints(Coordinate robotPosition, double robotVelocity, double robotHeading, double robotAngularVelocity, double currentTime, double weight)
	{
//		Intercept intercept = interceptManager.getIntercept
//		(
//			position, robotPosition, robotHeading,
//			robotVelocity,  robotAngularVelocity, lastBulletPower,
//			getBestDodgeType()
//		);
		
		circularIntercept.calculate
		(
			position.x,
			position.y,
			robotPosition.x,
			robotPosition.y,
			robotHeading,
			robotVelocity,
			lastBulletPower, //assumed bullet power
			robotAngularVelocity //assumed angular velocity
		);

		impactArray[currentImpactArrayIndex].set
		(
			new Coordinate(circularIntercept.impactPoint),
			circularIntercept.impactTime + currentTime,
			new Coordinate(robotPosition),
			circularIntercept.bulletHeading_deg,
			lastBulletPower,
			currentTime,
			weight
		);

		currentImpactArrayIndex++;
		if (currentImpactArrayIndex >= MAX_IMPACT_POINTS) currentImpactArrayIndex = 0;

	}

	public void updateInterceptImpactPoints(Coordinate robotPosition, double robotVelocity, double robotHeading, double robotAngularVelocity, double currentTime, double weight)
	{
		Intercept patternIntercept = new PatternIntercept(robot.patternAnalyser);
		patternIntercept.calculate
		(
			position.x,
			position.y,
			robotPosition.x,
			robotPosition.y,
			robotHeading,
			robotVelocity,
			lastBulletPower, //assumed bullet power
			robotAngularVelocity //assumed angular velocity
		);

		impactArray[currentImpactArrayIndex].set
		(
			new Coordinate(patternIntercept.impactPoint),
			patternIntercept.impactTime + currentTime,
			new Coordinate(robotPosition),
			patternIntercept.bulletHeading_deg,
			lastBulletPower,
			currentTime,
			weight
		);

		currentImpactArrayIndex++;
		if (currentImpactArrayIndex >= MAX_IMPACT_POINTS) currentImpactArrayIndex = 0;

	}


	public double getImpactDistance(Coordinate robotPosition, double time)
	{
		float distance = 0;
		for (int i = 0; i < MAX_IMPACT_POINTS; i++)
		{
			distance += impactArray[i].distanceFrom(robotPosition, time);
		}

		return distance;
	}

	public Coordinate getEstimatedPosition(double currentTime)
	{
		return new Coordinate
		(
			position.x + velocity * (currentTime - lastTimeScanned) * Math.sin(Math.toRadians(heading_deg)),
			position.y + velocity * (currentTime - lastTimeScanned) * Math.cos(Math.toRadians(heading_deg))
		);
	}

	public double getEstimatedVelocity(double currentTime)
	{
		return velocity;
	}

	public double getEstimatedHeading(double currentTime)
	{
		return heading_deg;
	}

	public void addRobotBullet(RobotBullet bullet)
	{
		robotBulletList.add(bullet);
	}

	public double getBulletHitPercentage()
	{
		return robotBulletList.getHitPercentage(position);
	}

	public void onBulletHit(BulletHitEvent event)
	{
		//bullet hits target
		if (event.getName() != name) return;

		robotBulletList.hit();
		gunManager.notifyBulletHit(event.getBullet());
	}

	public void saveMovementLog()
	{
		if (movementLog != null)
		{
			robot.out.println("Saving " + name);
			movementLog.save(name + ".csv");
		}
	}

	public double getWeightedValue(TargetWeight weight)
	{
		long currentTime = Environment.getTime();
		Coordinate targetPosition = getEstimatedPosition(currentTime);
		double roundNumber = Environment.getRobot().getRoundNum() + 1;
		double distance = 10000.0;
		double bearingToHim = 0;
		double hitRate = 100.0;
		if (weight.robotPosition != null)
		{
			distance = weight.robotPosition.distanceFrom(position);
			if (distance < 10.0) distance = 10000.0;
			bearingToHim = weight.robotPosition.headingTo(position);
			hitRate = getBulletHitPercentage();
		}

		double damageToMePerRound = damageInflicted;
		double damageToHimPerRound = damageSustained;
		double deathCountPerRound = deathCount / roundNumber;



		double timeSinceLastHit = currentTime - lastBulletHitTime;

		double timeSinceLastScanned = currentTime - lastTimeScanned;


		double headingDifference = Math.abs(Strategy.normalRelativeAngle(weight.robotHeading - heading_deg));
		
		double timeEstimate = distance / velocity;
		double predictionError = patternAnalyser.futurePositionArray.getError(velocity, timeEstimate);

		double result = 	(
					weight.kDistance * distance +
					weight.kHeadingDifference * headingDifference +
					weight.kEnergy * energy +
					weight.kDeltaEnergyEstimate * energyDeltaEstimate +
					weight.kDeathPerRound * deathCountPerRound +
					weight.kDamageInflictedPerRound * damageToMePerRound +
					weight.kDamageSustainedPerRound	* damageToHimPerRound +
					weight.kTimeSinceLastScanned * timeSinceLastScanned +
					weight.kVelocity * velocity +
					weight.kLastBulletPower * lastBulletPower +
					weight.kTimeSinceLastBulletHit * timeSinceLastHit +
					weight.kAngularVelocity * angularVelocity +
					weight.kClosestRobotDistance * closestRobotDistance +
					weight.kHitRate * hitRate +
					weight.kPredictionError * predictionError
				);
		return result;
	}

	public boolean getIsValid(Coordinate robotPosition, double currentTime)
	{
		if (lastTimeScanned < 0.0) return false;
		if (!isAlive) return false;
		if ((currentTime - lastTimeScanned) > 15)
		{
			return false;
		}

		if ((position != null) && (robotPosition != null))
		{
			if (position.distanceFrom(robotPosition) > 1250) return false;
		}
		return true;
	}
	
	public void fire(double bulletPower)
	{
		Bullet bullet = robot.setFireBullet(bulletPower);
		patternAnalyser.notifyRobotBulletFired();
		antiDodgeAnalyser.notifyRobotBulletFired();
		RandomAnalyser.notifyBulletFired();
		
		Coordinate robotPosition = Environment.getRobotPosition();
		RobotBullet rBullet = new RobotBullet(robotPosition, bullet);
		addRobotBullet(rBullet);
		gunManager.notifyBulletFired(rBullet);
	}
	
	public void setDead()
	{
		isAlive = false;
		deathCount++;
		
		gunManager.save(module, name);
//		save(module, name);
	}
	
	public int getActiveBulletCount()
	{
		return robotBulletList.list.size();
	}
	
	public boolean isGoodTarget()
	{
		Coordinate robotPosition = Environment.getRobotPosition();
		double distance = position.distanceFrom(robotPosition);
		
		return (distance < 400) ||
				(energy < 10) ||
				((Environment.getTime() - lastBulletHitTime) < 50) ||
				(closestRobotDistance < 200);
	}
	
	public void onHitByBullet(HitByBulletEvent e)
	{
		double damageDone = e.getPower() * 4;
		if (e.getPower() > 1.0)
		{
			damageDone += 2.0 * (e.getPower() - 1.0);
		}

		damageInflicted += damageDone;
		lastBulletPower = e.getPower();
		lastBulletHitTime = Environment.getTime();
		
		hitRateArray.handleBulletHit(currentDodgeStrategy);
		
		//chose different dodge strategy
		int previousDodgeStrategy = currentDodgeStrategy;
		
		currentDodgeStrategy = hitRateArray.getMinimumRateIndex();
		
		//currentDodgeStrategy = (int)Math.floor(Math.random() * InterceptManager.NUMBER_OF_INTERCEPT_TYPES);
		
		if (currentDodgeStrategy == InterceptManager.PATTERN_WHEN_VALID) 
		{
			currentDodgeStrategy = InterceptManager.PATTERN;
		}
		
		if (previousDodgeStrategy != currentDodgeStrategy)
		{
			printDodgeInfo();
		}
	}
	
	public void printDodgeInfo()
	{
		if (Environment.isDebugging)
		{
			System.out.println("Dodging Type: " + currentDodgeStrategy 
				+ " (" + hitRateArray.getRateString(currentDodgeStrategy) + ")");
			hitRateArray.printInfo();
		}
	}
				
	public double getCurrentIterativeLeadAngle()
	{
		return robotBulletList.currentLeadAngle;
	}
	
	public void save()
	{
		gunManager.save(module, name);
	}
	

}
