package tzu.intel;

import robocode.*;
import tzu.util.*;


/**
 * A class to hold often accessed game dimensions and
 * provides methods that work with those dimensions to
 * help improve game play.
 */
public final class BattleField implements Constants {


    private static double   width           = 0.0;
    private static double   height          = 0.0;
    private static double   lowerX          = 0.0;
    private static double   lowerY          = 0.0;
    private static double   upperX          = 0.0;
    private static double   upperY          = 0.0;
    private static double   minX            = 0.0;
    private static double   minY            = 0.0;
    private static double   maxX            = 0.0;
    private static double   maxY            = 0.0;
    private static double   midX            = 0.0;
    private static double   midY            = 0.0;
    private static double   diagonal        = 0.0;
    private static double   robotSize       = 0.0;
    private static double   targetBuffer    = 0.0;
    private static double   closeToCorner   = 0.0;
    private static Robot    myRobot;


    /**
     * Create a BattleField object.
     */
    public BattleField(Robot r) {

        myRobot = r;
        robotSize = BotMath.min(
            (myRobot.getWidth()  / 2 - 2),
            (myRobot.getHeight() / 2 - 2));

        width   = myRobot.getBattleFieldWidth();
        height  = myRobot.getBattleFieldHeight();

        minX    = robotSize;
        minY    = robotSize;
        maxX    = width  - robotSize;
        maxY    = height - robotSize;
        midX    = width  / 2;
        midY    = height / 2;

        lowerX  = minX + NEAR_WALL;
        lowerY  = minY + NEAR_WALL;
        upperX  = maxX - NEAR_WALL;
        upperY  = maxY - NEAR_WALL;

        closeToCorner = BotMath.min(width, height) / 3;
        diagonal = BotMath.sqrt(width * width + height * height);

        /* Adjust target buffer distance for a small battlefield. */
        targetBuffer = BotMath.min(maxX, maxY) / 2;
        targetBuffer = BotMath.min(targetBuffer, BUFFER_TARGET);
    }


    /** Same as robot.getBattleFieldWidth(). */
    public static double getWidth           () { return width; }
    /** Same as robot.getBattleFieldHeight(). */
    public static double getHeight          () { return height; }
    /** Returns the lower x boundary with a safety buffer built in. */
    public static double getLowerX          () { return lowerX; }
    /** Returns the lower y boundary with a safety buffer built in. */
    public static double getLowerY          () { return lowerY; }
    /** Returns the upper x boundary with a safety buffer built in. */
    public static double getUpperX          () { return upperX; }
    /** Returns the upper y boundary with a safety buffer built in. */
    public static double getUpperY          () { return upperY; }
    /** Returns the minimum x coordinate a robot can have. */
    public static double getMinX            () { return minX; }
    /** Returns the minimum y coordinate a robot can have. */
    public static double getMinY            () { return minY; }
    /** Returns the maximum x coordinate a robot can have. */
    public static double getMaxX            () { return maxX; }
    /** Returns the maximum y coordinate a robot can have. */
    public static double getMaxY            () { return maxY; }
    /** Returns half of the game width. */
    public static double getMidX            () { return midX; }
    /** Returns half of the game height. */
    public static double getMidY            () { return midY; }
    /** Returns the diagonal as measured from minX, minY to maxX, maxY. */
    public static double getDiagonal        () { return diagonal; }
    /** Returns shortest dimension from robot center to outer robot hull. */
    public static double getRobotSize       () { return robotSize; }
    /** Returns the distance at which a point x,y is considered close to a corner. */
    public static double getCloseToCorner   () { return closeToCorner; }
    /**
     * Returns the appropriate distance to stay away from an enemy bot.
     * given the current playing area dimension.
     */
    public static double getTargetBuffer    () { return targetBuffer; }


    /**
     * Given x,y coordinates and a bearing in degrees, returns the
     * distance from x,y to the game boundaries in the given bearing.
     * Game boundaries are defined as the absolute game dimensions
     * less half the smallest dimension of a robot.
     *
     * @param   x           The x coordinate.
     * @param   y           The y coordinate.
     * @param   bearing     Direction in degrees (compass orientation).
     * @return  Distance from x,y to game boundaries along the given
     *          bearing.
     *
     */
    public static double distanceToWall(double x,
                                        double y,
                                        double bearing) {

        if (bearing == A_0 || bearing == A_360) return (maxY - y);
        if (bearing == A_90)                    return (maxX - x);
        if (bearing == A_180)                   return (y - minY);
        if (bearing == A_270)                   return (x - minX);


        x = (bearing > A_0  && bearing < A_180 ? maxX - x : minX - x);
        y = (bearing > A_90 && bearing < A_270 ? minY - y : maxY - y);

        x /= BotMath.sin(bearing);
        y /= BotMath.cos(bearing);
        return BotMath.min(x,y);
    }


    /**
     * Returns true if given x,y coordinate is close to a wall
     * and the current given heading will take us into that wall.
     *
     * @param x         The x coordinate.
     * @param y         The y coordinate.
     * @param heading   The current heading in degrees.
     * @return          True if heading into wall; false otherwise.
     */
    public static boolean headedForWall(double x,
                                        double y,
                                        double heading) {

        return distanceToWall(x, y, heading) < HEADED_FOR_WALL;
    }

    /**
     * Tests if a coordinate is near a wall.
     *
     * @param x The x coordinate.
     * @param y The y coordinate.
     * @return  True if the x,y coordinate is near a wall;
     *          false otherwise.
     */
    public static boolean nearWall(double x,double y) {

        return (x < lowerX ||
                x > upperX ||
                y < lowerY ||
                y > upperY );
    }


    /**
     * If Bot is set, returns the best point to be in
     * in relation to that bot; otherwise returns the
     * battlefield midpoint.
     *
     * #param b Enemy bot to base position on.
     * @return  Best point to be at in relation to given bot target.
     */
    public static Point pickBestPosition(Bot b) {

        Point p = new Point(midX, midY);

        if (b != null) {

            double angleToMid =
                    BotMath.calcHeading(
                    b.getX(), b.getY(),
                    midX, midY);

            p.x = BotMath.calcX(b.getX(), angleToMid, targetBuffer);
            p.y = BotMath.calcY(b.getY(), angleToMid, targetBuffer);
        }
        return p;
    }


    /**
     * Returns true if there are a lot of robots for the given battle
     * field size.
     */
    public static boolean targetRichEnvironment() {

        return  botsPerHundredThousandPixels() >
                TARGET_RICH_RATIO;
    }


    /** Returns number of bots per hundred thousand pixels. */
    public static double botsPerHundredThousandPixels() {
        return myRobot.getOthers() / (maxX * maxY / 100000);
    }


    /** Returns true if coordinate x,y is near a corner; false otherwise. */
    public static boolean isInCorner(double x, double y) {

        return (whichCorner(x, y) != NOT_IN_A_CORNER);
    }


    /**
     * Returns an integer corresponding to which corner (if any)
     * the given coordinate is in.
     */
    public static int whichCorner(double x, double y) {

        int corner = closestCorner(x,y);

        if (distanceToCorner(x, y, corner) < closeToCorner) {
            return corner;
        }
        return NOT_IN_A_CORNER;
    }


    /** Returns a integer corresponding the the closest corner. */
    public static int closestCorner(double x, double y) {

        double ne = Math.sqrt(
                    Math.pow(width  - x, 2) +
                    Math.pow(height - y, 2));

        double se = Math.sqrt(
                    Math.pow(width  - x, 2) +
                    Math.pow(y, 2));

        double sw = Math.sqrt(
                    Math.pow(x, 2) +
                    Math.pow(y, 2));

        double nw = Math.sqrt(
                    Math.pow(x, 2) +
                    Math.pow(height - y, 2));

        if (ne < se && ne < sw && ne < nw) {
            return NORTHEAST_CORNER;
        } else if (se < sw && se < nw) {
            return SOUTHEAST_CORNER;
        } else if (sw < nw) {
            return SOUTHWEST_CORNER;
        }
        return NORTHWEST_CORNER;
    }


    /** Returns the distance to the closest corner. */
    public static double distanceToClosestCorner(double x, double y) {
        return distanceToCorner(x, y, closestCorner(x,y));
    }


    /** Returns the distance to the specified corner. */
    public static double distanceToCorner(double x, double y, int corner) {

        switch (corner) {
            case NORTHEAST_CORNER:
                return Math.sqrt(
                       Math.pow(width  - x, 2) +
                       Math.pow(height - y, 2));

            case SOUTHEAST_CORNER:
                return Math.sqrt(
                       Math.pow(width  - x, 2) +
                       Math.pow(y, 2));

            case SOUTHWEST_CORNER:
                return Math.sqrt(
                       Math.pow(x, 2) +
                       Math.pow(y, 2));

            case NORTHWEST_CORNER:
                return Math.sqrt(
                       Math.pow(x, 2) +
                       Math.pow(height - y, 2));
        }
        return Double.NaN;
    }


    /** Returns the bearing to the closest corner. */
    public static double bearingToClosestCorner(double x, double y) {
        return bearingToCorner(x,y, closestCorner(x,y));
    }


    /** Returns the bearing to the specified corner. */
    public static double bearingToCorner(double x, double y, int corner) {

        switch (corner) {

            case NORTHEAST_CORNER:
                return BotMath.arcTan(height - y, width - x);
            case SOUTHEAST_CORNER:
                return BotMath.arcTan(-y, width - x);
            case SOUTHWEST_CORNER:
                return BotMath.arcTan(-y, -x);
            case NORTHWEST_CORNER:
                return BotMath.arcTan(height - y, -x);
        }
        return Double.NaN;
    }


    /**
     * Calculate the total anti gravity force effect on your
     * robot by the battle field.  Returns the relative point
     * (x,y) that your robot is pushed towards by the
     * anti gravity force.  Returns null if there is no
     * anti gravity force in effect.
     */
    public static Point calcAntiGravity(Bot target) {

        double x                = 0.0;
        double y                = 0.0;
        double yourX            = myRobot.getX();
        double yourY            = myRobot.getY();
        double heading          = myRobot.getHeading();
        if (myRobot.getVelocity() < 0) heading = (heading + 180) % 360;

        /*
         * Calculate the anti-gravity force for all four walls
         * so our robot is not forced into them by the anti-gravity
         * of enemy bots.
         */

        x = ((yourX > midX)
                ? BotMath.min(width - BUFFER_WALL - yourX, 0)
                : BotMath.max(BUFFER_WALL - yourX, 0))
                * ANTI_GRAVITY_WALLS
                * myRobot.getOthers()
                * myRobot.getOthers();

        y = ((yourY > midY)
                ? BotMath.min(height - BUFFER_WALL - yourY, 0)
                : BotMath.max(BUFFER_WALL - yourY, 0))
                * ANTI_GRAVITY_WALLS
                * myRobot.getOthers()
                * myRobot.getOthers();

//        if (isInCorner(yourX, yourY) == false) {
//
//            if (y != 0 && x == 0) {
//                if (heading > 90 && heading < 270) {
//                    x = Math.abs(y) * -1;
//                } else {
//                    x = Math.abs(y);
//                }
//            }
//
//            if (x != 0 && y == 0) {
//                if (heading > 0 && heading < 180) {
//                    y = Math.abs(x);
//                } else {
//                    y = Math.abs(x) * -1;
//                }
//            }
//        }

        if (myRobot.getOthers() > 1) {

            /*
             * If this is a melee battle,
             * assign an anti-gravity force to the center so our bot
             * does not end up in the middle of battle all the time.
             * Ideally, all things considered equal, we want our bot
             * to move to the closest corner where it is safest.
             */
            x   += (yourX < midX
                ? -yourX
                : width - yourX)
                * ANTI_GRAVITY_CENTER
                * botsPerHundredThousandPixels();

            y   += (yourY < midY
                ? -yourY
                : height - yourY)
                * ANTI_GRAVITY_CENTER
                * botsPerHundredThousandPixels();

        } else if (target != null) {

            /*
             * If this is a one on one battle, pick the best position
             * on the battlefield with reference to the current target
             * and assign a gravity point to draw our robot towards
             * it.
             */
            Point p = pickBestPosition(target);

            if (BotMath.calcDistance(
                p.x, p.y, yourX, yourY) >
                BUFFER_BEST_POSITION) {

                x += ((p.x - yourX) * GRAVITY_BEST_POSITION);
                y += ((p.y - yourY) * GRAVITY_BEST_POSITION);
            }
        }

        return new Point(x,y);
    }
}
