package sgp;
import robocode.*;
import java.util.*;

/**
 * JiggleAnalyser - a class by Simon Parker
 */

public class JiggleAnalyser implements java.io.Serializable
{
	private AdvancedRobot robot;
	private Coordinate stoppedPosition0 = new Coordinate();
	private Coordinate stoppedPosition1 = new Coordinate();
	private Coordinate nextStoppedPosition = new Coordinate();

	private Coordinate position = new Coordinate();
	private double velocity = 0;
	private double previousVelocity = 0;
	private double velocity2 = 0;
	private double previousTime = 0;
	private double distanceFromStoppedPosition = 0;
	private double timeInCurrentMovement = 0;

	private double jiggleLength0, jiggleLength1;
	private int stopDuration0, stopDuration1;
	private double stoppedTime0, stoppedTime1;
	private final double MAX_JIGGLE_LENGTH = 200;
	private final double JIGGLE_OVERSHOOT_THRESHOLD = 40;
	private final double JIGGLE_EXTRA_TIME_THRESHOLD = 5;

	public boolean isJiggling = false;
	private boolean lastIsJiggling = false;
	public boolean isOscilating = false;
	private boolean lastIsOscilating = false;

	public boolean isUpdated = false;

	public JiggleAnalyser(AdvancedRobot theRobot)
	{
		robot = theRobot;
		reset();

	}

	public void reset()
	{
		isJiggling = false;
		isOscilating = false;
		isUpdated = true;

		stoppedPosition0.set(-1000, -1000);
		stoppedPosition1.set(-2000, -2000);

		jiggleLength0 = 1000;
		jiggleLength1 = 1000;

		stopDuration0 = 0;
		stopDuration1 = 0;

		stoppedTime0 = 0;
		stoppedTime1 = 0;

		distanceFromStoppedPosition = 0;

		velocity = 0;
		previousVelocity = 0;
		previousTime = 0;
		velocity2 = 0;
	}

	private int sign(double x)
	{
		if (x > 0.0) return 1;
		else if (x < 0.0) return -1;
		else return 0;
	}

	public void update(Coordinate currentPosition, double currentVelocity, double currentHeading_deg)
	{
		position.set(currentPosition);
		velocity = currentVelocity;
		timeInCurrentMovement = robot.getTime() - stoppedTime0;

//???		if ((velocity == 0) || ((sign(velocity) + sign(previousVelocity)) == 0))

		if (
			(velocity == 0) ||
//???			((Math.abs(velocity) > Math.abs(previousVelocity)) && (Math.abs(velocity2) >= Math.abs(previousVelocity)) && (Math.abs(previousVelocity) <= 3.0)) ||
			((sign(velocity) + sign(previousVelocity)) == 0)
		   )
		{
//			robot.out.println("Stopped!");
			updateStopped(currentPosition, currentVelocity, currentHeading_deg);
		}
		else
		{
			updateMoving(currentPosition, currentVelocity, currentHeading_deg);
		}

		isUpdated = (
						(!lastIsJiggling && isJiggling) ||
						(lastIsJiggling && !isJiggling) ||
						(!lastIsOscilating && isOscilating) ||
						(lastIsOscilating && !isOscilating)
					);
		lastIsJiggling = isJiggling;
		lastIsOscilating = isOscilating;
		previousVelocity = velocity;
		velocity2 = previousVelocity;
		previousTime = robot.getTime();

	}

	public void updateStopped(Coordinate currentPosition, double currentVelocity, double currentHeading_deg)
	{
		if (position.isEqualTo(stoppedPosition0))
		{
			//was stopped last time too
			stopDuration0++;

			isJiggling = ((stopDuration0 < (stopDuration1 + 3)) && isJiggling);
		}
		else
		{
			// only just stopped, was just moving
			jiggleLength1 = jiggleLength0; // save the length of the previous jiggle (used for setting isJiggling)
			jiggleLength0 = position.distanceFrom(stoppedPosition0); // the current jiggle was this far from the previous stopped position

			stoppedPosition1.set(stoppedPosition0); // save the previous stopped position
			stoppedPosition0.set(position); // set this position as the new stopped position

			//??? this assumes that we are oscillating?
			nextStoppedPosition.set(stoppedPosition1); //initial estimate of where the target is heading

			stopDuration1 = stopDuration0;
			stopDuration0 = 0;

			stoppedTime1 = stoppedTime0;

			stoppedTime0 = robot.getTime();

			timeInCurrentMovement = 0;

			distanceFromStoppedPosition = 0;

			isJiggling =
			(
				(jiggleLength1 <= MAX_JIGGLE_LENGTH) &&
				(jiggleLength0 <= MAX_JIGGLE_LENGTH)
			);
		}
	}


	public void updateMoving(Coordinate currentPosition, double currentVelocity, double currentHeading_deg)
	{
		double cycleTime = stoppedTime0 - stoppedTime1;
		distanceFromStoppedPosition = position.distanceFrom(stoppedPosition0);
		double distanceFromLastStoppedPosition = position.distanceFrom(stoppedPosition1);

		double signV = (currentVelocity >= 0) ? 1.0 : -1.0;

		double estimatedHeading_deg = position.headingTo(currentPosition);//???
		nextStoppedPosition.set
		(
			position.x + signV*(jiggleLength0 - distanceFromStoppedPosition)*Math.sin(Math.toRadians(estimatedHeading_deg)),
			position.y + signV*(jiggleLength0 - distanceFromStoppedPosition)*Math.cos(Math.toRadians(estimatedHeading_deg))
		);

		isOscilating = (distanceFromLastStoppedPosition < (jiggleLength0));

		isJiggling =
		(
			(distanceFromStoppedPosition < (JIGGLE_OVERSHOOT_THRESHOLD + jiggleLength0)) &&
			//(isOscilating) &&
			(timeInCurrentMovement < (JIGGLE_EXTRA_TIME_THRESHOLD + cycleTime)) &&
			(jiggleLength1 <= MAX_JIGGLE_LENGTH) &&
			(jiggleLength0 <= MAX_JIGGLE_LENGTH)
		);
	}

	public Coordinate getEstimatedPosition(double time)
	{
		double cycleTime = stoppedTime0 - stoppedTime1;
		double movingTime = cycleTime - stopDuration0;
		double v = jiggleLength0 / movingTime;

		double transformedTime = isOscilating ? (time + timeInCurrentMovement) % (2 * cycleTime) : (time + timeInCurrentMovement) % cycleTime;
		double k = distanceFromStoppedPosition / jiggleLength0;

		if (isOscilating)
		{
			if (transformedTime < movingTime)
			{
				k = v * transformedTime / jiggleLength0;
			}
			else if (transformedTime < cycleTime)
			{
				k = 1.0;
			}
			else if (transformedTime < (cycleTime + movingTime))
			{
				k = 1.0 - v * (transformedTime - cycleTime) / jiggleLength0;
			}
			else
			{
				k = 0.0;
			}
		}
		else
		{
			if (transformedTime < movingTime)
			{
				k = Math.floor((time + timeInCurrentMovement) / cycleTime) + (transformedTime/movingTime);
			}
			else if (transformedTime <= cycleTime)
			{
				k = Math.ceil((time + timeInCurrentMovement) / cycleTime);
			}
		}

/*
		robot.out.println("cycleTime " + cycleTime);
		robot.out.println("movingTime " + movingTime);
		robot.out.println("v " + v);
		robot.out.println("timeShift " + timeShift);
		robot.out.println("transformedTime " + transformedTime);
		robot.out.println("t " + time);
		robot.out.println("k " + k );
		robot.out.println("d " + distanceFromStoppedPosition);
		robot.out.println("jiggleLength0 " + jiggleLength0 + "\n");
*/
		return new Coordinate
		(
			stoppedPosition0.x + k * (nextStoppedPosition.x - stoppedPosition0.x),
			stoppedPosition0.y + k * (nextStoppedPosition.y - stoppedPosition0.y)
		);
	}


}
