package gh.twin;

import java.awt.Color;
import java.awt.geom.*;
import java.io.*;

import robocode.*;
import robocode.util.Utils;

/**
 * FirstBorn - a 2-vs-2 Twinbot by Gert Heijenk (GrubbmGait)
 *
 * This bot is slightly based upon GruwelTwin v0.1 and Griezel v0.4.
 * The teamleader (FirstBorn) will strongly suggest actions to his brother
 * There will be separate strategies for 2v2, 2v1, 1v2 and 1v1
 *
 * Revision information:
 * v0.1  20060817 Quick adaption of GruwelTwin, as I lost my sources.
 *                Try to stay close, less dive-in protection, lock when one opponent.
 * v0.2  20061102 Almost complete rewrite, added communication
 *                Changed a lot, most of it counterproductive
 * v0.3  20070221 Fixing radar and moveTarget selection
 *                replace strict oscillator by 'Gruwel'oscillator
 * v0.31 20070226 Fixing enemy recognition at startup
 *                much more agressive distancing
 * v0.32 20070507 bugfixing occasional radarfreeze (only with 1.3_Beta ??)
 *                simplified radartarget and movetarget selection (less radarsweeps, Teamleader does not command other anymore)
 *                try to reduce friendly fire by 'simulating' second radarbeam
 *                all of this to prepare for the GF-gun to come
 * v0.4  20070508 replace CT-gun by the GF-gun of Grinnik 0.4
 *                adapt gunsegmentation for TwinDuel situation
 *                tweak movement to get hit less
 * v0.41 20071028 use simulated radarbeam at both targets
 *                keep a bit more distance between brothers
 *                bugfix on firing when gun is not aimed
 */

public class FirstBorn extends TeamRobot {

	// gun variables
	private final static int	BINS			= 37;	// distance 550, bulletpower 1.9, halfbotwidth
	private final static int	MIDBIN			= BINS / 2;
	private final static double	BINFACTOR		= 0.07;	// roughly angle per bin
	// movement variables
	private final static double WALLAVOID		= 35;	// the area to avoid near walls
	private final static double BLINDMANSTICK	= 35;
	private final static double DEFDISTANCE		= 180;
	private final static double TWINDISTANCE	= 220;
	private final static double BROTHEROFFSET	= 2000D;
	private final static int	NRENEMIES		= 2;
	private final static int	PRIMARY			= 0;
	private final static int	SECONDARY		= 1;

	private static long		currTime;					// the current time (mainloop)
	private static boolean	firstBorn;

	private static Point2D.Double	myPos;
	private static double	myDirection = 1;

	// scanning and locking variables
	private boolean			radarScan;				// if true, busy with a radar scan
	private long			friendScan;				// the last time I scanned my twinbrother
	private double			lockLost;				// if zero, lock is lost

	// gun variables
	protected static int[][][][] twinBuf = new int[4][3][6][BINS];
//	protected static int[][][][][] twinBuf = new int[3][4][3][6][BINS];
	protected static double enemyLastVelocity;	// for gunsegmentation
	protected static int timeSinceVChange;		// for gunsegmentation

	// some info about the opponents
	protected int		strategy;				// 0:2v2 1:1v2 2:2v1 3:1v1
	protected static Message	enemy[];		// the opponent-twin
	private   static Message	brother;		// my brother
	protected int		gunTarget;				// 0, 1 is the enemy I want to shoot
	protected int		radarTarget;			// 0, 1 is the enemy I keep an eye on
	protected int		moveTarget;				// default always PRIMARY target

	// some general info
	private static RoundRectangle2D.Double playField; // the playable battlefield

	/**
	 * run: FirstBorn/TheOther default behavior
	 */
	public void run() {

		// Give the robot an appealing look
		setColors( Color.red.darker(), Color.yellow.darker(), Color.red);

		// check if I am the leader
		if (getEnergy() > DEFDISTANCE)
			firstBorn = true;

		// only support TwinDuel
		while (getOthers() != 3) ;

		// This info is kept over the full battle
		if (enemy == null) {
			// Set the One-Time stuff
        	playField = new RoundRectangle2D.Double( WALLAVOID, WALLAVOID,
        											 getBattleFieldWidth() - 2 * WALLAVOID,
        											 getBattleFieldHeight() - 2 * WALLAVOID, 50, 50);
			myPos = new Point2D.Double();
			enemy = new Message[ NRENEMIES];
			brother = new Message();
			for (int i = 0; i < NRENEMIES; i++) {
				enemy[i] = new Message();
			}
//			out.println("Distance = "+DEFDISTANCE);
		}

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);
		setTurnRadarRight( Double.POSITIVE_INFINITY);	// start scanning

		// mainloop
		while( true ) {
			currTime = getTime();
			myPos.setLocation( getX(), getY());

			if (currTime > 10) {
				doProcessData();
				doMovement();
				doGun();
				doRadar();
			}
			// perform all actions previously set
			execute();
		}
	}

	/**
	 * doProcessData: Determine movementTarget, gunTarget and radarTarget
	 */
	public void doProcessData() {
		int tmptarget;
		double attrac = 100;
		Message msg = new Message();

		// first determine if brother is in simulated radarbeams
//		tmptarget = Math.abs( radarTarget - 1);
		if (strategy % 2 == 0) {
			for (tmptarget = 0; tmptarget < NRENEMIES; tmptarget++) {
				if ((enemy[tmptarget].d < 9000) && (myPos.distance(brother.x, brother.y) < enemy[tmptarget].d)) {
					if (Math.abs(absoluteBearing(myPos, enemy[tmptarget].x, enemy[tmptarget].y) - 
								 absoluteBearing(myPos, brother.x, brother.y)) < (Math.PI / 8.0)) {
						enemy[tmptarget].d += BROTHEROFFSET;
//						out.println(getTime()+" Simulated scan on target "+tmptarget);
					}
				}
			}
		}

		if (enemy[PRIMARY].e < 25) attrac += 200;
		if (enemy[SECONDARY].e < 25) attrac -= 200;
 		gunTarget = tmptarget = (enemy[PRIMARY].d < enemy[SECONDARY].d + attrac ? PRIMARY : SECONDARY);
		if (strategy == 0) {
			radarTarget = (firstBorn == true ? PRIMARY : SECONDARY);
		}
		else {
			radarTarget = moveTarget = tmptarget;
		}

		// update my brother
		if (strategy % 2 == 0) {
			msg.name = getName();
			msg.x    = myPos.getX();
			msg.y    = myPos.getY();
			sendMsg( msg);
		}

//		out.println(getTime()+" Target G R : "+gunTarget+"  "+radarTarget);
	}

	/**
	 * doMovement: Perform the movement ;-)
	 */

	public void doMovement() {

		double  newposx, newposy;	// my future (end)position
		double	adjustAngle;		// the angle to adjust for wallavoidance
		double	perpAngle;			// the angle to reach perpendicularity
		double	distDiff;
//		double	myHeading;
		double	myDistance;
		double	myCheckDistance;
		boolean swapDirection = false;

		// Check if nearly at end of movement
		distDiff = enemy[moveTarget].d % BROTHEROFFSET;
		myDistance = Math.abs( getDistanceRemaining());
		if (myDistance == 0.0) {
			myDistance = (Math.sin( currTime/11) * Math.cos( currTime/29))*(120 + distDiff/2.0) + (Math.random()*100 - 50);
//			myDistance = (Math.sin( currTime/11) * Math.cos( currTime/29))*270 + (Math.random()*100 - 50);
			myDirection = (myDistance > 0 ? 1 : -1);
			myDistance = Math.abs(myDistance);
		}

		// Now keep a preferred distance, stay at approx DEFDISTANCE.
		distDiff -= DEFDISTANCE;
//		if (Math.abs( distDiff) > 10) {
			perpAngle = Utils.normalRelativeAngle(enemy[moveTarget].b + Math.PI/2 - getHeadingRadians());
			adjustAngle = myDirection * (distDiff < 0 ? (Math.sqrt(Math.abs( distDiff)) / 10) : -(Math.sqrt(Math.abs( distDiff)) / 25));
			if (Math.abs( perpAngle) > Math.PI/2) {
				adjustAngle = -adjustAngle;
				perpAngle = Math.atan( Math.tan( perpAngle));
			}
			setTurnRightRadians( perpAngle + adjustAngle);
//		}

		// perform some wall-avoidance, try to slide away if moving near wall
//		myHeading = getHeadingRadians();
		myCheckDistance = Math.min( BLINDMANSTICK, myDistance) * myDirection;
		newposx = myPos.x + Math.sin( getHeadingRadians()) * myCheckDistance;	// calculate end-of-stick position
		newposy = myPos.y + Math.cos( getHeadingRadians()) * myCheckDistance;

		if ((strategy % 2) == 0) {
			double tmpdist = (firstBorn == false ? TWINDISTANCE : TWINDISTANCE*0.85);
//			out.println(tmpdist+" brotherdist: "+brother.d);
			if (brother.d < tmpdist) {
				if (Point2D.distance(newposx, newposy, brother.x, brother.y) < brother.d) {
					swapDirection = !swapDirection;
//					out.println("Too close, swap direction");
				}
			}
		}
		// if endposition not in field, adapt
		if (!playField.contains( newposx, newposy)) {
			swapDirection = !swapDirection;
//			myDistance *= 1.5;
		}

		// set the distance and direction
		if (swapDirection == true) {
			myDirection = -myDirection;
			myDistance = Math.max(50, myDistance);
		}
		setAhead( myDistance * myDirection);

	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot( ScannedRobotEvent e) {

		Message scan = new Message();

		scan.name = e.getName();
		// if I scan my twinbrother, mark this area as 'bad juju'
		if (isTeammate( scan.name) == true) {
//			friendScan = getTime();
			return;
		}
		lockLost = 1;	// to indicate that the search is over

		scan.e = e.getEnergy();
		scan.d = e.getDistance();
		scan.b = Utils.normalRelativeAngle(getHeadingRadians() + e.getBearingRadians());
		scan.x = getX() + scan.d * Math.sin( scan.b);
		scan.y = getY() + scan.d * Math.cos( scan.b);
//		out.println(getTime()+" X:"+scan.x+" Y:"+scan.y);
		
		scan.v = e.getVelocity();
		scan.h = e.getHeadingRadians();
		scan.t = getTime();

		// if an enemy is hiding behind my twinbrother, just continue to look for someone else
//		if (friendScan == getTime()) {
//			scan.d += BROTHEROFFSET;
//			out.println(getTime()+" Real scan on target "+radarTarget);
//		}

		saveEnemy( scan);

		// only send messages when brother AND two opponents are alive
		if (getOthers() == 3)
			sendMsg( scan);
	}

	/**
	 * saveEnemy: keep data of all enemies (and brother)
	 */
	private void saveEnemy( Message scandata) {

//		out.print(currTime+" Saving data from "+scandata.name);
		if (isTeammate( scandata.name) == true) {
			brother = scandata;
//			out.println("  => brother !");
			return;
		}

		// At this place, determine the primary target
		// Teamleader is first, 'slave' is second choice
		int enemynr = (scandata.e > DEFDISTANCE ? PRIMARY : SECONDARY);
		if (enemy[enemynr].name == null) {
//			out.println(" => new enemy "+enemynr);
			enemy[enemynr] = scandata;
		}
		for (enemynr=0; enemynr < NRENEMIES; enemynr++) {
			if ((scandata.t != enemy[enemynr].t) && (scandata.name.equals( enemy[enemynr].name))) {
				enemy[enemynr] = scandata;
//				out.println("  and saving");
			}
		}
	}

	/**
	 * doGun: choose target and do the aiming
	 */
	public void doGun( ) {

		double velocityDiff;
//		double dist = (enemy[gunTarget].d % BROTHEROFFSET);

 		MicroWave gunWave = new MicroWave();
//		Point2D.Double myPos = gunWave.fireLocation	= new Point2D.Double( getX(), getY());	// my current position
		gunWave.fireLocation = new Point2D.Double( getX(), getY());	// my current position
			
		// some common stuff	
		double enemyAbsBearing = gunWave.HOTAngle = enemy[gunTarget].b;
//		enemyPos.setLocation( doProjectPos( myPos, enemyAbsBearing, dist));

		// gunstuff, use segmentation info of previous tick !
		gunWave.binFactor = ((enemy[gunTarget].v * Math.sin( enemy[gunTarget].h - enemyAbsBearing)) < 0 ? -BINFACTOR : BINFACTOR);
		gunWave.target = gunTarget;

		// accelerationsegment (used in next tick !!)
		int enemyAccBin = 2 + Math.min(3, (timeSinceVChange++ / 10));
		if (Math.abs( velocityDiff = enemyLastVelocity - (enemyLastVelocity = Math.abs(enemy[gunTarget].v))) > 0.0) {
			enemyAccBin = timeSinceVChange = ( velocityDiff > 0 ? 0 : 1);
		}
//		if ((enemyAccBin > 5) || (enemyAccBin < 0)) out.println("EnemyAccBin= "+enemyAccBin);
		// enemyheading segment (relative to absolutebearing)
		int enemyHeadBin = (int)(Math.abs( Math.acos( Math.abs( Math.cos( Utils.normalRelativeAngle( enemyAbsBearing - enemy[gunTarget].h))))) / (Math.PI / 5.95));
//		if ((enemyHeadBin > 2) || (enemyHeadBin < 0)) out.println("EnemyHeadBin= "+enemyHeadBin);
//		out.println( enemyHeadBin+" "+Math.toDegrees(Utils.normalRelativeAngle(enemyAbsBearing - enemy[gunTarget].h)));
		// enemydistance segment
		int enemyDistBin = (int)((enemy[gunTarget].d % BROTHEROFFSET) / 270);
//		if ((enemyDistBin > 4) || (enemyDistBin < 0)) out.println("EnemyDistBin= "+enemyDistBin);
//		int enemyVelBin = (int)((enemyLastVelocity + 1.99) / 4);
//		if ((enemyVelBin > 2) || (enemyVelBin < 0)) out.println("EnemyVelBin= "+enemyVelBin);
//		out.println( enemyHeadBin+" "+Math.toDegrees(Utils.normalRelativeAngle(enemyAbsBearing - enemy[gunTarget].h)));

		//		gunWave.gfBuf = twinBuf[enemyHeadBin][enemyAccBin]; 
//		gunWave.gfBuf = twinBuf[enemyVelBin][enemyDistBin][enemyHeadBin][enemyAccBin]; 
		gunWave.gfBuf = twinBuf[enemyDistBin][enemyHeadBin][enemyAccBin]; 

		// search best bin to aim
		int bindex = MIDBIN;
		int index = BINS - 1;
		do {
			if(gunWave.gfBuf[index] > gunWave.gfBuf[bindex]) {
				bindex = index;
			}
		} while (index-- > 0);
		setTurnGunRightRadians( Utils.normalRelativeAngle( enemyAbsBearing - getGunHeadingRadians() + ((bindex - MIDBIN) * gunWave.binFactor)));
			
		// Fire in the hose!
		if ((getGunHeat() == 0.0) && (Math.abs(getGunTurnRemaining()) < 10.0) && (getEnergy() > 1.9) && (enemy[gunTarget].d < 1500)) {
//		if ((getGunHeat() == 0.0) && (getEnergy() > 1.9)) {
			setFire( 1.9);
//			setFire( 1.9 + (int)(100 / enemy[gunTarget].d));	// 18 bytes extra
			addCustomEvent( gunWave);
//			out.println("Fire !"+enemyAccBin);
		}

		// set the power of the bullet, including energymanagement
//		double power = ( enemy[gunTarget].d > 100 ? 1.9 : 3.0);
//		power = ( dist > 850 ? 0.1 : (dist > 700 ? 0.49 : (dist > 170 ? 1.9 : 3.0)));
//		power = Math.min( getEnergy()/5, Math.min( (enemy[gunTarget].e/4) + 0.1, power));

		return;
	}

	/**
	 * doRadar: do radar behaviour (turn, lock etc.)
	 */
	public void doRadar( ) {

		if (--lockLost == 0) radarScan = false;
		if (radarScan == true) return;
		if (enemy[radarTarget].t < (currTime - 10)) {
			setTurnRadarRightRadians( Double.POSITIVE_INFINITY);
			radarScan = true;
			lockLost = 10;
		}
		else
			setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( enemy[radarTarget].b - getRadarHeadingRadians()));
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
	public void onRobotDeath(RobotDeathEvent e) {
		// If current target dies, force to find another
		strategy++;
		if (isTeammate( e.getName()) == false) {
			strategy++;			// meaning strategy += 2 !!
			for (int i=0; i < NRENEMIES; i++) {
				if (e.getName().equals( enemy[i].name)) {
					enemy[i].d = 9000;
//					out.println("Yeah, opponent "+i+" died ! => strategy: "+strategy);
				}
			}
		}
//		else
//			out.println("Aargh, my TwinBrother died => strategy: "+strategy);
		
	}

	/**
	 * onWin: Show my private victory dance
	 */
//	public void onWin(WinEvent e)
//	{
		//Victory dance	
//		turnRadarLeft( Double.POSITIVE_INFINITY);
//	}

	/**
	 * sendMsg: Send a message to your brother
	 */	
	public void sendMsg( Message escan) {
		// try to send message
		try {
			broadcastMessage( escan );
//			out.println(getTime()+" Message ("+escan.action+") sent");
		} catch ( IOException ex ) { 
//            System.out.println("sendMsg("+escan.action+"): IOException= " + ex);
		}
	}

	/**
	 * onMessageReceived: Received a message from my brother containing info about a bot
	 */
	public void onMessageReceived( MessageEvent e) {
		Message received = (Message) e.getMessage();
//		out.println(getTime()+" Message ("+received.name+") received");
		received.d = myPos.distance( received.x, received.y);
		received.b = absoluteBearing( myPos, received.x, received.y);
		saveEnemy( received);
	}

    // got this from RaikoMicro, by Jamougha, but I think it's used by many authors
    //  - returns the absolute angle (in radians) from source to target points
//    public double absoluteBearing(Point2D.Double source, Point2D.Double target) {
//        return Math.atan2(target.x - source.x, target.y - source.y);
//    }
    public double absoluteBearing(Point2D.Double source, double targetx, double targety) {
        return Math.atan2(targetx - source.x, targety - source.y);
    }

    /**
	 * MicroWave: a condition based Wave
	 * Credits to PEZ as this is based upon the MicroWave of Aristocles
	 */
	public class MicroWave extends Condition {

		double HOTAngle;
		Point2D.Double fireLocation;
		double waveFront;
		double binFactor;	
		int[]  gfBuf;
		int    target;

		public boolean test() {
			if (enemy[target].d == 9000) {
//				out.println("removed wave of non-existing enemy");
				removeCustomEvent(this);
			}
			else if ( fireLocation.distance( enemy[target].x, enemy[target].y) <= (waveFront += 14.3)) {
//				int gfbin =	(int)(Utils.normalRelativeAngle( 
//						Math.atan2( enemy[target].x - fireLocation.x, enemy[target].y - fireLocation.y)
//						- HOTAngle)	/ binFactor);
				int gfbin =	(int)(Utils.normalRelativeAngle( 
						absoluteBearing( fireLocation, enemy[target].x, enemy[target].y) - HOTAngle) / binFactor);
				if (Math.abs(gfbin) <= MIDBIN)
					gfBuf[ MIDBIN + gfbin]++;
//				else {
//					out.println("outside bins: "+gfbin);
//					out.println("fireLocation: "+fireLocation);
//					out.println("enemy "+target+" pos : "+enemy[target].x+" "+enemy[target].y);
//					out.println("HotAngle    : "+Math.toDegrees(HOTAngle));
//					out.println("RealAngle   : "+Math.toDegrees(Utils.normalRelativeAngle(absoluteBearing( fireLocation, enemy[target].x, enemy[target].y)
//							))+"  binFactor : "+binFactor);
//					out.println("Virtualangle: "+Math.toDegrees((gfbin*binFactor)));
//					out.println("WaveFront   : "+waveFront);
//				}
 
//				gfBuf[ MIDBIN + (int)(Utils.normalRelativeAngle( 
//						Math.atan2( enemy[target].x - fireLocation.x, enemy[target].y - fireLocation.y)
//						- HOTAngle)	/ binFactor)]++;
				removeCustomEvent(this);
			}
			return false;
		}
	}
}

/**
 * class Message: used to distribute information to my twinbrother
 */
//@SuppressWarnings("serial")
class Message implements java.io.Serializable {
	String name;
	double x;	// X-position
	double y;	// Y-position
	double e;	// energy
	double d;	// distance
	double b;	// bearing
	double v;	// velocity
	double h;	// heading
	double t;	// time of scan

}
																