package ags.muse.recon.waves;

import java.awt.Graphics2D;
import java.awt.Color;

import java.util.List;

import robocode.Event;
import robocode.Rules;
import ags.muse.recon.EnemyRobot;
import ags.muse.recon.RobotHistory;
import ags.muse.recon.RobotList;
import ags.muse.recon.SelfRobot;
import ags.util.AuxiliaryDataHolder;
import ags.util.Range;
import ags.util.points.AbsolutePoint;
import ags.util.points.RelativePoint;

public class EWave extends AuxiliaryDataHolder {
    private final EnemyRobot source;
    private final SelfRobot self;
    private final RobotHistory sourceHistory;
    private final RobotHistory selfHistory;
    private final double bulletPower;
    private final double velocity;
    private final double damage;
    private double radius;
    private final AbsolutePoint origin;
    private final double maximumEscapeAngle;
    private final double zeroAngle;
    private final double linearGF;
    private final int gfDirection;
    private final double probability;
    private final boolean melee;
    private boolean passed;
    private double passedGF;
    
    public EnemyRobot getSource() {
        return source;
    }
    public RobotHistory getSourceHistory() {
        return sourceHistory;
    }
    public RobotHistory getSelfHistory() {
        return selfHistory;
    }
    public double getBulletPower() {
        return bulletPower;
    }
    public double getVelocity() {
        return velocity;
    }
    public double getDamage() {
        return damage;
    }
    public double getRadius() {
        return radius;
    }
    public AbsolutePoint getOrigin() {
        return origin;
    }
    public double getMaximumEscapeAngle() {
        return maximumEscapeAngle;
    }
    public double getZeroAngle() {
        return zeroAngle;
    }
    public double getLinearGF() {
        return linearGF;
    }
    public int getGFDirection() {
        return gfDirection;
    }
    public double getProbability() {
        return probability;
    }
    public boolean hasPassed() {
        return passed;
    }
    public double getPassedGF() {
        return passedGF;
    }
    public boolean isMelee() {
        return melee;
    }
    
    public EWave(RobotList robots, EnemyRobot source, SelfRobot self, RobotHistory sourceHistory,
            RobotHistory selfHistory, double bulletPower, double radius, double probability) {
        this.source = source;
        this.self = self;
        this.sourceHistory = sourceHistory;
        this.selfHistory = selfHistory;
        this.bulletPower = bulletPower;
        this.velocity = Rules.getBulletSpeed(bulletPower);
        this.damage = Rules.getBulletDamage(bulletPower);
        this.radius = radius;
        this.origin = sourceHistory.getLocation();
        this.maximumEscapeAngle = Math.asin(Rules.MAX_VELOCITY/velocity);
        this.zeroAngle = origin.angleTo(selfHistory.getLocation());
        double sinAngle = Math.sin(selfHistory.getVelocity().direction - zeroAngle);
        this.gfDirection = (selfHistory.getOrientation() * sinAngle > 0) ? 1 : -1;
        this.linearGF = angleToGF(zeroAngle + selfHistory.getVelocity().magnitude * sinAngle / velocity);
        this.probability = probability;
        this.melee = (robots.getEnemies().size() > 1);
    }
    
    public EWave(EWave other) {
        this.source = other.source;
        this.self = other.self;
        this.sourceHistory = other.sourceHistory;
        this.selfHistory = other.selfHistory;
        this.bulletPower = other.bulletPower;
        this.velocity = other.velocity;
        this.damage = other.damage;
        this.radius = other.radius;
        this.origin = other.origin;
        this.maximumEscapeAngle = other.maximumEscapeAngle;
        this.zeroAngle = other.zeroAngle;
        this.linearGF = other.linearGF;
        this.gfDirection = other.gfDirection;
        this.probability = other.probability;
        this.melee = other.melee;
        this.passed = other.passed;
        this.passedGF = other.passedGF;
    }
    
    public void iterate(int ticks) {
        radius += velocity * ticks;
    }

    public void update(List<Event> events) {
        radius += velocity;
        if (!passed && radius > (30+origin.distance(self.getLocation()))) {
            passed = true;
            passedGF = angleToGF(origin.angleTo(self.getLocation()));
        }
    }
    
    public double angleToGF(double angle) {
        return gfDirection*robocode.util.Utils.normalRelativeAngle(angle-zeroAngle)/maximumEscapeAngle;
    }
    
    public double gfToAngle(double gf) {
        return robocode.util.Utils.normalAbsoluteAngle(gfDirection*gf*maximumEscapeAngle + zeroAngle);
    }
    
    private final int arcsegments = 5;
    private final Color waveColor = new Color(0, 255, 0, 32);
    public void draw(Graphics2D g) {
        g.setColor(waveColor);
        int lx = 0;
        int ly = 0;
        for (int i = -arcsegments; i <= arcsegments; i++) {
            RelativePoint rel = RelativePoint.fromDM(zeroAngle + (i * maximumEscapeAngle)/arcsegments, radius);
            AbsolutePoint p = origin.addRelativePoint(rel);
            int x = (int)Math.round(p.x);
            int y = (int)Math.round(p.y);
            if (i > -arcsegments) {
                g.drawLine(lx, ly, x, y);
            }
            lx = x;
            ly = y;
        }
        
        //drawSegment(g, new Range(-0.01, 0.01));
        //drawSegment(g, new Range(linearGF-0.01, linearGF+0.01));
    }
    
    public void drawSegment(Graphics2D g, Range range) {
        RelativePoint rel1 = RelativePoint.fromDM(zeroAngle + gfDirection*(range.getLower() * maximumEscapeAngle), radius);
        RelativePoint rel2 = RelativePoint.fromDM(zeroAngle + gfDirection*(range.getUpper() * maximumEscapeAngle), radius);
        AbsolutePoint p1 = origin.addRelativePoint(rel1);
        AbsolutePoint p2 = origin.addRelativePoint(rel2);
        int x1 = (int)Math.round(p1.x);
        int y1 = (int)Math.round(p1.y);
        int x2 = (int)Math.round(p2.x);
        int y2 = (int)Math.round(p2.y);
        g.drawLine(x1, y1, x2, y2);
    }
}
