package romz.component.radar.constantwidth;

import static romz.math.RobotMath.abs;
import static romz.math.RobotMath.atan;
import static romz.math.RobotMath.heading;
import static romz.math.RobotMath.min;
import static romz.math.RobotMath.signum;
import robocode.Rules;
import robocode.util.Utils;
import romz.component.radar.Radar;
import romz.model.Situation;
import romz.model.advice.RadarAdvice;
import romz.model.robot.Robot;

public class ConstantWidthRadar implements Radar {
	
	/** The distance we scan past the enemy to keep the lock */
	private static final double RADAR_SECURITY_DISTANCE = Robot.HALF_ROBOT_SIZE + Rules.MAX_VELOCITY;
	
	private static final double TB_TURNS_FOR_FULL_SCAN = FULL_TURN / Rules.RADAR_TURN_RATE;
	
	private Situation situation;
	
	private long lastFullScanTime = 0;
	
	@Override
	public RadarAdvice getRadarAdvice(Situation situation) {
		this.situation = situation;
		RadarAdvice radarAdvice;
		if (fullScanNeeded()) {
			radarAdvice = fullScan();
		} else {
			radarAdvice = lockRadar();
		}
		return radarAdvice;
	}

	private boolean fullScanNeeded() {
		if (!situation.enemy.scanned && lastFullScanIsFinished()) return true;
		return (situation.time - situation.enemy.timeScanned > RESCAN_DELAY);
	}

	private boolean lastFullScanIsFinished() {
		return situation.time - lastFullScanTime > TB_TURNS_FOR_FULL_SCAN;
	}
	
	private RadarAdvice fullScan() {
		RadarAdvice radarAction = new RadarAdvice();
		lastFullScanTime = situation.time;
		if (abs(situation.hero.radarTurnRemaining) < Rules.RADAR_TURN_RATE) {
			double angleToCenter = heading(situation.hero.x, situation.hero.y, situation.battleField.width / 2, situation.battleField.height / 2);
			double orientation = signum(Utils.normalRelativeAngleDegrees(angleToCenter - situation.hero.gunHeading));
			radarAction.radarTurn = orientation * FULL_TURN;
			radarAction.fullScan = true;
		}
		return radarAction;
	}

	private RadarAdvice lockRadar() {
		RadarAdvice radarAction = new RadarAdvice();
		double radarOffset = Utils.normalRelativeAngleDegrees(situation.enemy.absoluteBearing - situation.hero.radarHeading);
		double additionalTurn = signum(radarOffset) * min(atan(RADAR_SECURITY_DISTANCE / situation.enemy.distance), Rules.RADAR_TURN_RATE);
		radarAction.radarTurn = radarOffset + additionalTurn;
		return radarAction;
	}
	
}
