
package sgp;
import robocode.*;
import java.io.*;


/**
 * Manages firing and keeping track of which types of firing are 
 * most successful at hitting this target.
 */
public class GunManager 
{
	private HitRateArray hitRateArray = new HitRateArray(0.4);

	private int mostRecentFiringType = 1;
	private RobotBulletList bulletList = new RobotBulletList();
	private Target target = null;
	
	private InterceptManager interceptManager = null;
		
	public GunManager(Target theTarget)
	{
		target = theTarget;		
		interceptManager = new InterceptManager
			(
				target.jiggleAnalyser, 
				target.patternAnalyser, 
				target.averageLinearAnalyser,
				target.antiDodgeAnalyser,
				target.polarPatternAnalyser,
				target
			);
		reset(target.module, target.name);
	}
	
/**
 *  returns an intercept of based on a certain type of targeting. It uses 
 *  it's reference to the GunManager's associated target 
 *  to calculate the various quantities
 */
//	public Intercept getSpecificIntercept
//		(Coordinate currentPosition, double bulletPower, int type)
//	{
//		long time = Environment.getRobot().getTime();
//		return interceptManager.getIntercept
//			( 
//				currentPosition, target.getEstimatedPosition(time), 
//				target.getEstimatedHeading(time), target.getEstimatedVelocity(time),
//				target.angularVelocity, bulletPower, type
//			);
//	}
	
/**
 * 	Chooses a firing type and returns the appropriate intercept
 */
	public Intercept getIntercept(Coordinate robotPosition, double bulletPower)
	{
		// find the index of the best type
		int bestInterceptTypeIndex = hitRateArray.getMaximumRateIndex();
		//???int bestInterceptTypeIndex = InterceptManager.GRAVITY;
		
		int type = bestInterceptTypeIndex;
		
		if (mostRecentFiringType != type)
		{
			if (Environment.isDebugging)
			{
				System.out.println("Firing Type : " + type);
				hitRateArray.printInfo();
			}
		}
		
		mostRecentFiringType = type;

//Inline the getSpecificIntercept function
//		long time = Environment.getRobot().getTime();
//		Intercept intercept = interceptManager.getIntercept
//			( 
//				robotPosition, target.getEstimatedPosition(time), 
//				target.getEstimatedHeading(time), target.getEstimatedVelocity(time),
//				target.angularVelocity, bulletPower, type
//			);
//			
//		return intercept;
		
//		return getSpecificIntercept(robotPosition, bulletPower, type);

		long currentTime = Environment.getRobot().getTime();
		Coordinate targetPosition = target.getEstimatedPosition(currentTime);
		double targetHeading = target.getEstimatedHeading(currentTime);
		double targetVelocity = target.getEstimatedVelocity(currentTime);
		double targetAngularVelocity = target.angularVelocity;
		Intercept intercept = null;
		
		switch (type)
		{
		case InterceptManager.JIGGLE_ONLY:
			intercept = interceptManager.getJiggleIntercept(false, false);
			break;
		case InterceptManager.JIGGLE_OSC_PLUS_PATTERN:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, false);
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_PATTERN:
			intercept = interceptManager.getJiggleIntercept(true, true);
			if (intercept == null) return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, false);
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_CIRCULAR:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) intercept = interceptManager.getCircularIntercept();
			break;
		case InterceptManager.CIRCULAR:
			intercept = interceptManager.getCircularIntercept();
			break;
		case InterceptManager.LINEAR:
			intercept = interceptManager.getLinearIntercept();
			break;
		case InterceptManager.PATTERN:
			return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, false);
		case InterceptManager.PATTERN_WHEN_VALID:
			return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, true);
		case InterceptManager.JIGGLE_OSC_PLUS_PATTERN_WHEN_VALID:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, true);
			break;
		case InterceptManager.JIGGLE_OSC_PLUS_PATTERN_WHEN_VALID_CIRCULAR:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) intercept = interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, true);
			if (intercept == null) intercept = interceptManager.getCircularIntercept();
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_PATTERN_WHEN_VALID:
			intercept = interceptManager.getJiggleIntercept(true, true);
			if (intercept == null) return interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, true);
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_PATTERN_WHEN_VALID_CIRCULAR:
			intercept = interceptManager.getJiggleIntercept(true, true);
			if (intercept == null) intercept = interceptManager.getPatternIntercept(robotPosition, targetPosition, targetHeading, targetVelocity, targetAngularVelocity, bulletPower, true);
			if (intercept == null) intercept = interceptManager.getCircularIntercept();
			break;
		case InterceptManager.AVERAGE_LINEAR:
			intercept = interceptManager.getAverageLinearIntercept();
			break;
		case InterceptManager.ANTI_DODGE:
			intercept = interceptManager.getAntiDodgeIntercept();
			break;
		case InterceptManager.JIGGLE_OSC_PLUS_ANTI_DODGE:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) intercept = interceptManager.getAntiDodgeIntercept();
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_ANTI_DODGE:
			intercept = interceptManager.getJiggleIntercept(true, true);
			if (intercept == null) intercept = interceptManager.getAntiDodgeIntercept();
			break;
		case InterceptManager.JIGGLE_OSC_PLUS_CIRCULAR:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if (intercept == null) intercept = interceptManager.getCircularIntercept();
			break;
		case InterceptManager.RANDOM:
			intercept = new RandomIntercept();
			break;
		case InterceptManager.GRAVITY:
			intercept = new GravityIntercept(target);
			break;
		case InterceptManager.JIGGLE_OSC_PLUS_GRAVITY:
			intercept = interceptManager.getJiggleIntercept(true, false);
			if ((intercept == null) && (target != null)) intercept = new GravityIntercept(target);
			break;
		case InterceptManager.JIGGLE_ALL_PLUS_GRAVITY:
			intercept = interceptManager.getJiggleIntercept(true, true);
			if ((intercept == null) && (target != null)) intercept = new GravityIntercept(target);
			break;
		case InterceptManager.POLAR_PATTERN:
			if (interceptManager.polarPatternAnalyser != null) intercept = new PolarPatternIntercept(interceptManager.polarPatternAnalyser);
			else 
			{
				System.out.println("No PolarPatternAnalyser");
				intercept = interceptManager.getCircularIntercept();
			}
			break;

		}
		
		if (intercept == null) return null;
		
		intercept.calculate
		(
			robotPosition,
			targetPosition,
			targetHeading,
			targetVelocity,
			bulletPower,
			targetAngularVelocity
		);
		
		return intercept;


	}


/**
 * A bullet has been fired by the robot at this target using a firing strategy 
 * of a certain type.
 */
	public void notifyBulletFired(RobotBullet bullet)
	{
		bullet.type = mostRecentFiringType;
		bulletList.add(bullet);
		bulletList.removeOldBullets(target.position);
		hitRateArray.handleBulletFired(mostRecentFiringType);
	}
	
	public void notifyBulletHit(Bullet bullet)
	{
		//determine the type of firing used
		int firingType = bulletList.getType(bullet);
		
		if (firingType < 0) return;
		
		hitRateArray.handleBulletHit(firingType);
	}
	
	
	public void reset(String moduleName, String robotName)
	{
		hitRateArray.load(moduleName, robotName);
	}
	
	public void save(String moduleName, String robotName)
	{
		hitRateArray.save(moduleName, robotName);
	}
	
	public void load(String moduleName, String robotName)
	{
		hitRateArray.load(moduleName, robotName);
	}

}
