package pmc;
import robocode.*;
import java.awt.Color;

/**
 * SniperBot - a robot by Patrick Coleman
 */
public class SniperBot extends AdvancedRobot
{
	public static final double DIST3 = 300.0;
	public static final double DIST2 = 500.0;
	public static final double DIST1 = 700.0;
	boolean chasing = false;
	
	/**
	 * toRadians: returns the radians for given degrees
	 */
	public double toRadians(double degs){
	   return degs * Math.PI / 180.0;
	}
	
	/** 
	 * avoidWalls: returns -sz if moving that far will take you through a wall
	 */
	public double avoidWalls(double sz){
	    double pos = toRadians(getHeading());
	    
	    double nextX = getX() + sz * Math.cos(pos);
        if(nextX <= 1e-9 || nextX >= getBattleFieldWidth()) return -sz;
        
	    double nextY = getY() + sz * Math.sin(pos);
        if(nextY <= 1e-9 || nextY >= getBattleFieldHeight()) return -sz;
	    
	    return sz;
	}
	
	/**
	 * run: SniperBot's default behavior
	 */
	public void run() {
		// After trying out your robot, try uncommenting the import at the top,
		// and the next line:
		setColors(Color.magenta, Color.black, Color.yellow);
		
		// set things up independantly:
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		
		while(true) {
			if(chasing){
				// rotate the scanner left then right:
				setTurnRadarLeft(5.0);
				setTurnRadarRight(10.0);
				setTurnRadarLeft(5.0);
			} else setTurnRadarRight(360.0);
			
			// move randomly:
			double rotate = Math.random() * 90.0 - 45.0;
			turnLeft(rotate);
            double forwards = avoidWalls(Math.random() * 100.0 + 50.0);
			ahead(forwards);
			if(forwards < 0.0) turnLeft(90.0);

			execute();		
		}
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		chasing = true; 
		
		// guess where the robot will be next:
        double dist = e.getDistance();
        double direc = e.getBearingRadians() + toRadians(this.getHeading());
		double isNowX = this.getX() + dist * Math.cos(direc);
		double isNowY = this.getY() + dist * Math.sin(direc);
		double willBeX = isNowX + e.getVelocity() * Math.cos(e.getHeadingRadians());
		double willBeY = isNowY + e.getVelocity() * Math.sin(e.getHeadingRadians());
		
		double nxtDirec = Math.toDegrees(Math.atan2(willBeY - getY(), willBeX - getX()));
       
        fixBase(nxtDirec);
		fixRadar(nxtDirec);
        fixGun(nxtDirec);
		
		// shoot if close
		double strength = 0.0;
		if(dist < DIST3) strength = 3.0;
		else if(dist < DIST2) strength = 2.0;
		else if(dist < DIST1) strength = 1.0;
		if(strength > 0.5){
			fire(strength);
			chasing = false;
		}
		
		// if they're almost dead, move backwards:
		if(e.getEnergy() < 15.0){
			setAhead(-75);
		}
		
	}

	/**
	 * onHitByBullet: What to do when you're hit by a bullet
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		// nothing 
		//ahead(25);
	}
	
	/**
	 * fixRadar: moves the radar to point at where the bot will be
	 */
	void fixRadar(double direc){
       double currPoint = getRadarHeading();
       double dHead = (currPoint - direc) % 360.0;
       if(dHead < -180.0) dHead += 360.0;
       if(dHead >  180.0) dHead -= 360.0;
       setTurnRadarLeft(dHead);
	}
	
	/**
	 * fixGun: moves the gun to point at where the bot will be
	 */
    void fixGun(double direc){
	   double currPoint = getGunHeading();
       double dHead = (currPoint - direc) % 360.0;
       if(dHead < -180.0) dHead += 360.0;
       if(dHead >  180.0) dHead -= 360.0;
       setTurnGunLeft(2 * dHead);
	}
	
	/**
	 * fixBase: moves the base of the bot to point at where the bot will be
	 */
    void fixBase(double direc){
	   double currPoint = getHeading();
       double dHead = (currPoint - direc) % 360.0;
       if(dHead < -180.0) dHead += 360.0;
       if(dHead >  180.0) dHead -= 360.0;
       setTurnLeft(dHead);
       setAhead(50);
	}
}
