package ags.muse.recon.waves;

import java.awt.Color;
import java.awt.Graphics2D;
import java.util.ArrayList;
import java.util.List;

import robocode.Event;

import ags.muse.base.Rules;
import ags.muse.recon.EnemyRobot;
import ags.muse.recon.RobotHistory;
import ags.muse.recon.RobotList;
import ags.util.points.*;

public class EWaveDetector {
    private final Rules rules;
    private final RobotList robots;
    private final AbsolutePoint[] fieldCorners;

    ArrayList<EWave> waves = new ArrayList<EWave>();
    public ArrayList<EWave> getWaves() { return waves; }

    // Constructor
    public EWaveDetector(Rules rules, RobotList robots) {
        this.rules = rules;
        this.robots = robots;
        fieldCorners = new AbsolutePoint[] {
                AbsolutePoint.fromXY(0, 0),
                AbsolutePoint.fromXY(0, rules.BATTLEFIELD_HEIGHT),
                AbsolutePoint.fromXY(rules.BATTLEFIELD_WIDTH, 0),
                AbsolutePoint.fromXY(rules.BATTLEFIELD_WIDTH, rules.BATTLEFIELD_HEIGHT)
        };
    }

    // Update
    public void update(List<Event> events) {
        // Process all bots
        for (EnemyRobot e : robots.getEnemies()) {
            updateBot(e);
        }
        
        // Update waves in the air
        ArrayList<EWave> newWaves = new ArrayList<EWave>(waves.size());
        for (EWave e : waves) {
            e.update(events);
            double maxDist = 0;
            for (AbsolutePoint corner : fieldCorners) {
                double dist = corner.distance(e.getOrigin());
                if (dist > maxDist) maxDist = dist;
            }
            if ((e.getRadius() <= maxDist) && !e.hasPassed()) {
                newWaves.add(e);
            }
        }
        waves.clear();
        waves.addAll(newWaves);
    }

    // Update a specific bot
    private void updateBot(EnemyRobot e) {
        // Don't process if we don't have new data
        if (e.getDataAge() > 1) {
            return;
        }

        // They didn't fire at us if we weren't at least close to as close as the closest
        if (e.getClosestDist() + 50 < e.getDistance(rules.NAME)) {
            return;
        }

        // If too early, ignore
        if (robots.getSelf().getHistory().prev() == null) {
            return;
        }

        // Find the last non-estimated tick
        RobotHistory cursor = e.getHistory().prev();
        RobotHistory scursor = robots.getSelf().getHistory().prev().prev();
        if (cursor == null || scursor == null) {
            return;
        }
        
        // Count the number of ticks the wave could have been from
        int possibleTickCount = 1;
        while (cursor.isEstimated()) {
            cursor = cursor.prev();
            scursor = scursor.prev();
            possibleTickCount++;
        }

        // Get energy change
        double firePower = cursor.getEnergy() - (e.getEnergy() - e.getExpectedHealthChange());

        // Abort if not actually firing
        if (firePower >= 3.0000001 || firePower <= 0.0999999) {
            //if (firePower != 0) System.out.println("Not firing: "+firePower);
            return;
        }

        // Add virtual bullets for each possible firing tick
        final double waveProbability = 1.0 / possibleTickCount;
        for (int i=0; i<possibleTickCount; i++)
        {
            EWave wave = new EWave(robots, e, robots.getSelf(), cursor, scursor, firePower, 0, waveProbability);

            wave.iterate(i);
            waves.add(wave);
            
            cursor = cursor.next();
            scursor = scursor.next();
        }
    }

    public void paint(Graphics2D g) {
        g.setColor(new Color(0, 255, 255));
        for (EWave w : waves) {
            w.draw(g);
        }
    }

    public void remove(EWave wave) {
        waves.remove(wave);
    }
}
