package fushi.PvP1;

/* This class is used as a chassis simulator to predict robot positions
 * base on the set... commands given to the robot
 */
public class BodyPeer
{

   /* private members
    */
   private double angleToTurn;
   private double maxVelocity;
   private double maxTurnRate;
   private double distanceRemaining;
   private int moveDirection;
   private boolean slowingDown;
   private double systemMaxTurnRate;
   private double systemMaxVelocity;
   private double heading;
   private double acceleration;
   private double turnRate;
   private double velocity;
   private double x, y;
   private double maxBraking;
   private double maxAcceleration;
   
   /* use this method to simulate robot movement in the next turn;
    * first you have to set correct value for simulator with methods:
    *  - setTurn
    *  - setMaxVelocity
    *  - setMove
    *  - setX
    *  - setY
    *  - setHeading
    *  - setVelocity
    * after this, calling this function simulates exactly one frame of
    * robot movement
    */
   public void simulate()
   {
      updateHeading();
      updateMovement();
   }
   
   /* returns current simulator position
    */
   public Vector2d getPosition()
   {
      return new Vector2d( x, y );
   }
   
   /* returns current simulator heading
    */
   public double getHeading()
   {
      return heading;
   }
   
   /* returns current simulator velocity
    */
   public double getVelocity()
   {
      return velocity;
   }
   
   /* this method corresponds to setTurnRightRadians method for
    * advanced robot
    */
   public final void setTurn( double radians )
   {
      angleToTurn = radians;
   }
   
   /* this method corresponds to setMaxVelocity method for
    * advanced robot
    */
   public final void setMaxVelocity( double newVelocity )
   {
      maxVelocity = Math.abs( newVelocity );
   }
   
   /* this method corresponds to setAhead method for
    * advanced robot
    */
   public final void setMove( double distance )
   {
      distanceRemaining = distance;
      acceleration = 0;
      if ( distance == 0 )
         moveDirection = 0;
      else if ( distance > 0 )
         moveDirection = 1;
      else
         moveDirection = -1;
      slowingDown = false;
   }
   
   /* use the following methods to tell the 
    * actual robot's parameters to the simulator
    */
   public final void setX( double x )
   {
      this.x = x;
   }
   
   public final void setY( double y )
   {
      this.y = y;
   }
   
   public final void setHeading( double radians )
   {
      this.heading = radians;
   }
   
   public final void setVelocity( double vel )
   {
      this.velocity = vel;
   }
   
   /* simulate robot heading update for one frame 
    * based on current orders
    */
   void updateHeading()
   {
      turnRate = Math.min( maxTurnRate, 
                          (0.40000000000000002D + 0.59999999999999998D * 
                            (1.0D - Math.abs( velocity ) / systemMaxVelocity)) 
                            * systemMaxTurnRate );
      if ( angleToTurn > 0 ) {
         if ( angleToTurn < turnRate ) {
            heading += angleToTurn;
            angleToTurn = 0;
         } else {
            heading += turnRate;
            angleToTurn -= turnRate;
         }
      } else if ( angleToTurn < 0 ) {
         if( angleToTurn > -turnRate ) {
            heading += angleToTurn;
            angleToTurn = 0;
         } else {
            heading -= turnRate;
            angleToTurn += turnRate;
         }
      }
      
      heading = Util.normalAbsoluteAngle( heading );
   }
   
   /* simulate robot movement update for one frame 
    * based on current orders
    */
   void updateMovement()
   {
      if( distanceRemaining == 0 && velocity == 0 )
         return;
      
      if( !slowingDown && moveDirection == 0 ) {
         slowingDown = true;
         if( velocity > 0 )
            moveDirection = 1;
         else if( velocity < 0 )
            moveDirection = -1;
         else
            moveDirection = 0;
      }
      
      double desiredDistanceRemaining = distanceRemaining;
      
      if( slowingDown ) {
         if( moveDirection == 1 && distanceRemaining < 0 )
            desiredDistanceRemaining = 0.0D;
         else if( moveDirection == -1 && distanceRemaining > 1 )
            desiredDistanceRemaining = 0.0D;
      }
      
      double slowDownVelocity = (int)((maxBraking / 2D) * (Math.sqrt(4D * Math.abs(desiredDistanceRemaining) + 1.0D) - 1.0D));
      if( moveDirection == -1 )
         slowDownVelocity = -slowDownVelocity;
      if( !slowingDown ) {
         if( moveDirection == 1 ) {
             if( velocity < 0 )
                acceleration = maxBraking;
             else
                acceleration = maxAcceleration;
             if( velocity + acceleration > slowDownVelocity )
                slowingDown = true;
         } else if( moveDirection == -1 ) {
             if( velocity > 0 )
                acceleration = -maxBraking;
             else
                acceleration = -maxAcceleration;
             if( velocity + acceleration < slowDownVelocity )
                slowingDown = true;
         }
      }

      if( slowingDown ) {
         if( distanceRemaining != 0 && Math.abs( velocity ) <= maxBraking
                                    && Math.abs( distanceRemaining ) <= maxBraking )
            slowDownVelocity = distanceRemaining;
         double perfectAccel = slowDownVelocity - velocity;
         if( perfectAccel > maxBraking )
            perfectAccel = maxBraking;
         else if( perfectAccel < -maxBraking )
            perfectAccel = -maxBraking;
         acceleration = perfectAccel;
      }

      if( velocity > maxVelocity || velocity < -maxVelocity )
         acceleration = 0;
         
      velocity += acceleration;
      if( velocity > maxVelocity )
         velocity -= Math.min( maxBraking, velocity - maxVelocity );
      if( velocity < -maxVelocity )
         velocity += Math.min( maxBraking, -velocity - maxVelocity );

      double dx = velocity * Math.sin( heading );
      double dy = velocity * Math.cos( heading );
      x += dx;
      y += dy;
      
      if( slowingDown && velocity == 0 ) {
         distanceRemaining = 0;
         moveDirection = 0;
         slowingDown = false;
         acceleration = 0;
      }
      distanceRemaining -= velocity;
   }
   
   /* sets proper start values for private members
    */
   public BodyPeer()
   {
      x = 0;
      y = 0;
      acceleration = 0;
      maxAcceleration = 1;
      maxBraking = 2;
      maxVelocity = 8;
      maxTurnRate = Math.toRadians(10);
      systemMaxTurnRate = Math.toRadians(10);
      systemMaxVelocity = 8;
      distanceRemaining = 0;
      moveDirection = 0;
      slowingDown = false;
      angleToTurn = 0;
      turnRate = 0;
      heading = 0;
      velocity = 0;
   }
}