package tzu.event;

import robocode.*;
import tzu.gun.*;
import tzu.intel.*;
import tzu.util.*;

/**
 * Provides an event that will track a bullet's path and
 * update a target's reaction to that bullet if the bullet
 * misses the target.
 *
 * This class will register itself as a custom event when
 * it is created and remove itself when it's work is done.
 */
public final class BulletTracker extends Condition implements Constants {

    static final int     BULLET_TRACKER_EVENT_PRIORITY = 9;


    private static AdvancedRobot   myRobot;

    private Bullet          bullet;
    private FireSolution    solution;

    private Bot             target;
    private String          targetName;

    private double          originalX;
    private double          originalY;
    private double          originalHeading;
    private double          originalBearing;
    private double          originalDistance;
    private double          originalSpeed;
    private double          originalAcceleration;

    private long            expectedTimeOfImpact;

    /**
     * Create a BulletTracker object, update the target's
     * shot at count, and register this object as a custom
     * event.
     */
    public BulletTracker(AdvancedRobot ar,
                         Bullet b,
                         FireSolution fs) {

        myRobot                     = ar;
        bullet                      = b;
        target                      = fs.target;
        targetName                  = target.getName();
        solution                    = fs;
        expectedTimeOfImpact        = myRobot.getTime() + fs.timeToImpact;

        if (target.getLastUpdate()  < myRobot.getTime()) {

            originalX               = target.getX();
            originalY               = target.getY();
            originalHeading         = target.getHeading();
            originalBearing         = target.getBearing();
            originalDistance        = target.getDistance();
            originalSpeed           = target.getSpeed();
            originalAcceleration    = target.getChgSpeed();

        } else {

            originalX               = target.getPrevX();
            originalY               = target.getPrevY();
            originalHeading         = target.getPrevHeading();
            originalBearing         = target.getPrevBearing();
            originalDistance        = target.getPrevDistance();
            originalSpeed           = target.getPrevSpeed();
            originalAcceleration    = target.getPrevChgSpeed();
        }

        /*
         * Update the robot "shot at" stats.
         */
        switch (solution.method) {

            case FireSolution.AIM_PREDICTED:
                target.predictAimShotAt();
                break;
            case FireSolution.AIM_ESTIMATED:
                target.estimateAimShotAt();
                break;
            case FireSolution.AIM_PATTERN:
                target.patternAimShotAt();
                break;
            case FireSolution.AIM_AVERAGE:
                target.averageAimShotAt();
                break;
        }

        this.setPriority(BULLET_TRACKER_EVENT_PRIORITY);
        myRobot.addCustomEvent(this);
    }


    /**
     * Check if the bullet has hit or missed it's target.
     * @return Always returns false.
     */
    public boolean test() {

//        if (DEBUG && myRobot.getName().indexOf("2") > 0) {
//            myRobot.out.println("Tracking bullet: " + this);
//        }

        if (targetName.equals(bullet.getVictim())) {

            /*
             * Whatever aiming technique was used worked! Update the robot's
             * hit count and remove this event.  Its job is done.
             */


            switch (solution.method) {

                case FireSolution.AIM_PREDICTED:
                    target.predictAimHit();
                    break;
                case FireSolution.AIM_ESTIMATED:
                    target.estimateAimHit();
                    break;
                case FireSolution.AIM_PATTERN:
                    target.patternAimHit();
                    break;
                case FireSolution.AIM_AVERAGE:
                    target.averageAimHit();
                    break;
            }

            myRobot.removeCustomEvent(this);


        } else if (bullet.getVictim() != null) {

            /* We shot the wrong guy! */
            myRobot.removeCustomEvent(this);

        } else if (expectedTimeOfImpact <= myRobot.getTime()) {

            /*
             * The bullet should have hit the intended target by now.
             * Calculate the approximate heading and speed of the target
             * after it was fired at and save this as a reaction.
             */

            double relX             = target.getX() - originalX;
            double relY             = target.getY() - originalY;
            double distance         = Math.sqrt((relX * relX) + (relY * relY));
            double heading          = BotMath.arcTan(relY, relX);
            double speed            = distance / solution.timeToImpact;

            double relativeHeading  = heading - originalBearing;
            if (relativeHeading < 0) relativeHeading += A_360;
            Reaction reaction = new Reaction(relativeHeading, speed);

            target.updateReactions(originalHeading,
                                   originalBearing,
                                   originalSpeed,
                                   originalAcceleration,
                                   originalDistance,
                                   reaction);

            myRobot.removeCustomEvent(this);
        }
        return false;
    }

    public String toString() {

        return
            " x="           + BotMath.round2(bullet.getX())          +
            " y="           + BotMath.round2(bullet.getY())          +
            " energy="      + BotMath.round2(bullet.getPower())      +
            " speed="       + BotMath.round2(bullet.getVelocity())   +
            " heading="     + BotMath.round2(bullet.getHeading())    ;
    }
}
