package pl.robocode;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import pl.robocode.util.EnemyWave;
import pl.robocode.util.Util;
import robocode.AdvancedRobot;
import robocode.BulletHitEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;

public class Pacyfista extends AdvancedRobot {

    private static final int FROM_CENTER = 50;

    public Point2D.Double myLocation;
    public Point2D.Double enemyLocation;
    
    public double lastEnemyHeading;

    public List<EnemyWave> enemyWaves;    
    public List<Integer> surfDirections;

    public static double enemyEnergy = 100.0;

    public void run() {
        enemyWaves = new ArrayList<EnemyWave>();
        surfDirections = new ArrayList<Integer>();


        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
        setMaxTurnRate(Rules.MAX_TURN_RATE);

        while (true) {
            turnRadarRight(Rules.RADAR_TURN_RATE);
        }
    }

    @Override
    public void onScannedRobot(ScannedRobotEvent e) {
        myLocation = new Point2D.Double(getX(), getY());
        double absBearing = e.getBearingRadians() + getHeadingRadians();

        //turn the radar turn more not to loose enemy
        setTurnRadarRightRadians(Utils.normalRelativeAngle(absBearing - getRadarHeadingRadians()) * 2);
        
        double lateralVelocity = getVelocity() * Math.sin(e.getBearingRadians());
        surfDirections.add(0, new Integer((lateralVelocity >= 0) ? 1 : -1));

        double bulletPower = enemyEnergy - e.getEnergy();
        if (bulletPower <= 3 && surfDirections.size() > 2) {
            EnemyWave ew = new EnemyWave();
            ew.fireTime = getTime() - 1;
            ew.bulletVelocity = Rules.getBulletSpeed(bulletPower);
            ew.distanceTraveled = Rules.getBulletSpeed(bulletPower);
            ew.fireLocation = (Point2D.Double) enemyLocation.clone(); // last tick
            enemyWaves.add(ew);
        }

        enemyEnergy = e.getEnergy();

        // get enemy location based on myLocation, absolute braring and distance
        enemyLocation = Util.getNextLocation(myLocation, absBearing, e.getDistance());
        doTargeting(e, (Point2D.Double)enemyLocation.clone());
        updateWaves();
        doSurfing();
    }

    private void doTargeting(ScannedRobotEvent event, Point2D.Double enemyLocation) {
        double bearingToEnemy = event.getBearingRadians();
        double distanceToEnemy = event.getDistance();

        double enemyHeading = event.getHeadingRadians();
        double enemyVelocity = event.getVelocity();
        double absoluteBearing = bearingToEnemy + getHeadingRadians();

        double firePower = decideFirePower(distanceToEnemy);

        lastEnemyHeading = enemyHeading;
        double buletDist = Rules.getBulletSpeed(firePower);
        double vectorX = Math.sin(absoluteBearing) * distanceToEnemy;
        double vectorY = Math.cos(absoluteBearing) * distanceToEnemy;

        double headingDifference = enemyHeading - lastEnemyHeading;
        boolean turnedArround = false;
        for (int i = 0;  i < Point2D.distance(0.0D, 0.0D, vectorX, vectorY); i += buletDist) {
            vectorX += enemyVelocity * Math.sin(enemyHeading); 
            vectorY += enemyVelocity * Math.cos(enemyHeading);
           
            //change enemy direction
            enemyHeading += headingDifference;
            
            if(true) {
                //assumes that enemy will break and change direction
                if(turnedArround) {
                    enemyVelocity =+ Rules.ACCELERATION;
                    if(enemyVelocity > Rules.MAX_VELOCITY) {
                        enemyVelocity = Rules.MAX_VELOCITY;
                    }
                } else {
                    enemyVelocity -= Rules.DECELERATION;
                }

                if(enemyVelocity < 0) {
                    turnedArround = true;
                    enemyVelocity = 1;
                    enemyHeading += Math.PI;
                    enemyHeading = Utils.normalAbsoluteAngle(enemyHeading);
                }
                
            }
        }

        setTurnGunRightRadians(Math.asin(Math.sin(Math.atan2(vectorX, vectorY) - getGunHeadingRadians())));

        if (isReadyToFire()) {
            setFire(firePower);
        }
            
    }

    private boolean isReadyToFire() {
        return getGunHeat() == 0 && Math.abs(getGunTurnRemaining()) < Rules.MAX_TURN_RATE;
    }

    private double decideFirePower(double distanceToEnemy) {
        // low energy
        if (getEnergy() < 10) {
            return 1;
        }
        
        double firePower = 400 / distanceToEnemy;
        if (firePower > Rules.MAX_BULLET_POWER) {
            return Rules.MAX_BULLET_POWER;
        }
        return firePower;
    }

    private void updateWaves() {
        for (Iterator<EnemyWave> iterator = enemyWaves.iterator(); iterator.hasNext();) {
            EnemyWave ew = iterator.next();

            ew.distanceTraveled = (getTime() - ew.fireTime) * ew.bulletVelocity;
            if (ew.distanceTraveled > myLocation.distance(ew.fireLocation) + FROM_CENTER) {
                iterator.remove();
            }
        }
    }

    public EnemyWave getClosestSurfableWave() {
        double closestDistance = Double.MAX_VALUE;
        EnemyWave closestWave = null;

        for (EnemyWave ew : enemyWaves) {
            double distance = myLocation.distance(ew.fireLocation) - ew.distanceTraveled;
            
            //if vawe is closer than bullet velocity - it's useless to avoid it
            if (distance > ew.bulletVelocity && distance < closestDistance) {
                closestWave = ew;
                closestDistance = distance;
            }
        }

        return closestWave;
    }
    
    private boolean left = true;
    
    public void doSurfing() {
        EnemyWave surfWave = getClosestSurfableWave();

        if (surfWave == null) {
            return;
        }
                
        if((getTime() % 23) == 0) {
            left = !left;
        }

        double goAngle = Util.absoluteBearing(surfWave.fireLocation, myLocation);
        
        if (left) {
            goAngle = Util.wallSmoothing(myLocation, goAngle - (Math.PI / 2), -1);
        } else {
            goAngle = Util.wallSmoothing(myLocation, goAngle + (Math.PI / 2), 1);
        }
        double angle = Utils.normalRelativeAngle(goAngle - getHeadingRadians());

        /*
         * For turns greater than 90 degree - break and start moving backwards
         * 
         *    | -- instead of turning, break here and go backwards with smaller angle
         *   /|
         *  / |
         * /  |
         */   
        if (Math.abs(angle) > (Math.PI / 2)) {
            if (angle < 0) {
                setTurnRightRadians(Math.PI + angle);
            } else {
                setTurnLeftRadians(Math.PI - angle);
            }
            setBack(100);
        } else {
            setTurnRightRadians(angle);
            setAhead(100);
        }
    }

    @Override
    public void onBulletHit(BulletHitEvent event) {
        double bulletPower = event.getBullet().getPower();
        enemyEnergy -= Rules.getBulletDamage(bulletPower);
    }

}
