package pl;

import robocode.*;
import java.awt.geom.*;
import java.util.*;

/**
 * Drum is a robot that uses a not-so-great segmented Guess-Factor Gun and a
 * very basic segmented Wave Surfing movement.
 * 
 * @author Patrick Lin
 * 
 */
public class Drum extends AdvancedRobot {

	public static DrumGun drumGun;
	public static DrumTank drumTank;

	public void run() {
		if (getRoundNum() == 0) {
			drumGun = new DrumGun(this);
			drumTank = new DrumTank(this);
		}
		drumGun.clear();
		drumTank.clear();
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustRadarForRobotTurn(true);
		do {
			turnRadarRightRadians(Double.POSITIVE_INFINITY);
		} while (true);
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		setTurnRadarRightRadians(Utils.normalizeBearing(e.getBearingRadians() + getHeadingRadians()
				- getRadarHeadingRadians()) * 2);
		drumGun.beatDrum(e);
		drumTank.rideWave(e);
	}

	public void onHitByBullet(HitByBulletEvent e) {
		drumTank.onHitByBullet(e);
	}

	public void onBulletHitBullet(BulletHitBulletEvent e) {
		drumTank.onBulletHitBullet(e);
	}

	public void onWin(WinEvent e) {
		drumGun.clear();
		drumTank.clear();
	}

	public void onDeath(DeathEvent e) {
		drumGun.clear();
		drumTank.clear();
	}

	public void onPaint(java.awt.Graphics2D g) {
		drumGun.paintWaves(g);
		drumTank.paintWaves(g);
	}
}

class DrumGun {
	private static AdvancedRobot robot;
	private static Point2D.Double robotLocation;

	private static Point2D.Double enemyLocation;
	private static double previousEnemyVelocity;

	private static final double MAX_DISTANCE = 1000;
	private static final int DISTANCE_INDEXES = (int) (MAX_DISTANCE / 100);

	private final static int VELOCITY_INDEXES = 5;
	private final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;

	private final static int BINS = 47;
	private final static int MIDDLE_BIN = (BINS - 1) / 2;
	private final static double MAX_ESC_ANGLE = 0.8143399421265254D;
	private final static double BIN_WIDTH = MAX_ESC_ANGLE / MIDDLE_BIN;

	private static double[][][][] statsSlow = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][BINS];
	private static double[][] statsFast = new double[DISTANCE_INDEXES][BINS];

	private static ArrayList<SoundWave> waves = new ArrayList<SoundWave>();

	public DrumGun(AdvancedRobot _robot) {
		robot = _robot;
	}

	public void beatDrum(ScannedRobotEvent e) {
		robotLocation = new Point2D.Double(robot.getX(), robot.getY());
		double absBearing = robot.getHeadingRadians() + e.getBearingRadians();
		enemyLocation = Utils.project(robotLocation, absBearing, e.getDistance());
		double firePower = e.getDistance() < 200 ? 3.0 : 1.72;
		SoundWave w = new SoundWave();
		w.sign = Utils.sign(e.getVelocity() * Math.sin(e.getHeadingRadians() - absBearing));
		w.firedLocation = robotLocation;
		w.fireTime = robot.getTime();
		w.bearing = absBearing;
		w.velocity = Utils.bulletVelocity(firePower);
		int distanceIndex = (int) Utils.limit(0, DISTANCE_INDEXES - 1, e.getDistance()
				/ (MAX_DISTANCE / DISTANCE_INDEXES));
		int velocityIndex = (int) Math.abs(e.getVelocity() / 2);
		int previousVelocityIndex = (int) Math.abs(previousEnemyVelocity / 2);
		w.segmentSlow = statsSlow[distanceIndex][velocityIndex][previousVelocityIndex];
		w.segmentFast = statsFast[distanceIndex];

		previousEnemyVelocity = e.getVelocity();

		updateWaves();

		robot.setTurnGunRightRadians(Utils.normalizeBearing(absBearing
				- robot.getGunHeadingRadians() + (w.sign * BIN_WIDTH)
				* (getMostVisited(w) - MIDDLE_BIN)));
		if (robot.getGunHeat() == 0 && robot.getGunTurnRemaining() < 5)
			if (robot.setFireBullet(firePower) != null)
				w.firing = true;
		if (robot.getEnergy() >= 0.1)
			waves.add(w);
	}

	private int getMostVisited(SoundWave w) {
		double[] segment = w.segmentSlow;
		if (isEmpty(segment))
			segment = w.segmentFast;
		int mostVisited = MIDDLE_BIN;
		for (int i = 0; i < BINS; i++)
			if (segment[i] > segment[mostVisited])
				mostVisited = i;
		return mostVisited;
	}

	private boolean isEmpty(double[] segment) {
		for (int i = 0; i < segment.length; i++)
			if (segment[i] != 0.0) {
				return false;
			}
		return true;
	}

	public void updateWaves() {
		for (int i = 0; i < waves.size(); i++) {
			SoundWave w = waves.get(i);
			w.distanceTraveled = (robot.getTime() - w.fireTime) * w.velocity;
			double distance = w.firedLocation.distance(enemyLocation);
			if (w.distanceTraveled >= distance) {
				int bin = (int) Math.round((Utils.normalizeBearing(Utils
						.absoluteBearing(w.firedLocation, enemyLocation)
						- w.bearing))
						/ (w.sign * BIN_WIDTH))
						+ MIDDLE_BIN;
				if (w.firing) {
					for (int j = 0; j < BINS; j++) {
						w.segmentSlow[j] += 1.0 / (Math.pow(bin - j, 2) + 1);
						w.segmentFast[j] += 1.0 / (Math.pow(bin - j, 2) + 1);
					}
				} else {
					for (int j = 0; j < BINS; j++) {
						w.segmentSlow[j] += 1.0 / (Math.pow(bin - j, 2) + 2);
						w.segmentFast[j] += 1.0 / (Math.pow(bin - j, 2) + 2);
					}
				}
				waves.remove(i);
				i--;
			}
		}
	}

	public void clear() {
		waves.clear();
	}

	public void paintWaves(java.awt.Graphics2D g) {
		for (SoundWave w : waves) {
			java.awt.Color color = w.firing ? java.awt.Color.RED : java.awt.Color.CYAN;
			Utils.drawCircle(w.firedLocation.x, w.firedLocation.y, w.distanceTraveled, color, g);
		}
	}
}

class DrumTank {
	private static AdvancedRobot robot;
	private static Point2D.Double robotLocation;

	private static Point2D.Double enemyLocation;
	private static Point2D.Double previousEnemyLocation;
	private static double previousEnemyEnergy;

	private static double MAX_DISTANCE = 1000;
	private static int DISTANCE_INDEXES = (int) (MAX_DISTANCE / 100);

	private final static int VELOCITY_INDEXES = 5;
	private final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;

	private final static int BINS = 47;
	private final static int MIDDLE_BIN = (BINS - 1) / 2;
	private final static double MAX_ESC_ANGLE = 0.8143399421265254D;
	private final static double BIN_WIDTH = MAX_ESC_ANGLE / MIDDLE_BIN;

	private static Rectangle2D.Double fieldRect = new Rectangle2D.Double(18, 18, 764, 564);

	private static double[][][][] statsSlow = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][BINS];
	private static double[][] statsFast = new double[DISTANCE_INDEXES][BINS];

	private static ArrayList<SoundWave> waves = new ArrayList<SoundWave>();

	private static ArrayList<Integer> signs = new ArrayList<Integer>();
	private static ArrayList<Double> bearings = new ArrayList<Double>();
	private static ArrayList<Double> distances = new ArrayList<Double>();
	private static ArrayList<Double> velocities = new ArrayList<Double>();

	public DrumTank(AdvancedRobot _robot) {
		robot = _robot;
	}

	public void rideWave(ScannedRobotEvent e) {
		previousEnemyLocation = enemyLocation;
		robotLocation = new Point2D.Double(robot.getX(), robot.getY());
		double absBearing = robot.getHeadingRadians() + e.getBearingRadians();
		enemyLocation = Utils.project(robotLocation, absBearing, e.getDistance());

		signs.add(0, Utils.sign(robot.getVelocity() * Math.sin(e.getBearingRadians())));
		bearings.add(0, robot.getHeadingRadians() + e.getBearingRadians() + Math.PI);
		distances.add(0, e.getDistance());
		velocities.add(0, robot.getVelocity());

		double energyChange = previousEnemyEnergy - e.getEnergy();
		if (energyChange < 3.1 && energyChange > 0.09 && signs.size() > 2) {
			double firePower = energyChange;
			SoundWave w = new SoundWave();
			w.fireTime = e.getTime() - 1;
			w.firedLocation = previousEnemyLocation;
			w.sign = signs.get(2);
			w.bearing = bearings.get(2);
			w.velocity = Utils.bulletVelocity(firePower);
			w.distanceTraveled = Utils.bulletVelocity(firePower);
			int distanceIndex = (int) Utils.limit(0, DISTANCE_INDEXES - 1, distances.get(2)
					/ (MAX_DISTANCE / DISTANCE_INDEXES));
			int velocityIndex = (int) Math.abs(velocities.get(2) / 2);
			int previousVelocityIndex = (int) Math.abs(velocities.get(3) / 2);
			w.segmentSlow = statsSlow[distanceIndex][velocityIndex][previousVelocityIndex];
			w.segmentFast = statsFast[distanceIndex];
			if (robot.getEnergy() >= 0.1)
				waves.add(w);
		}

		previousEnemyEnergy = e.getEnergy();

		updateWaves();
		doRiding();
	}

	private void doRiding() {
		SoundWave w = getClosestWave();
		Point2D.Double myLocation = new Point2D.Double(robot.getX(), robot.getY());
		if (w != null) {
			double dangerLeft = getDanger(w, -1);
			double dangerRight = getDanger(w, 1);

			double angle = Utils.absoluteBearing(w.firedLocation, myLocation);
			if (dangerLeft < dangerRight)
				angle = wallSmooth(myLocation, angle - Utils.HALF_PI, -1);
			else
				angle = wallSmooth(myLocation, angle + Utils.HALF_PI, 1);

			angle = Utils.normalizeBearing(angle - robot.getHeadingRadians());
			byte moveDir;
			if (Math.abs(angle) > Utils.HALF_PI) {
				if (angle < 0)
					robot.setTurnRightRadians(angle + Math.PI);
				else
					robot.setTurnLeftRadians(angle - Math.PI);
				moveDir = -1;
			} else {
				robot.setTurnRightRadians(angle);
				moveDir = 1;
			}
			robot.setAhead(100 * moveDir);
		} else {
			double angle = Utils.absoluteBearing(enemyLocation, myLocation);
			angle = wallSmooth(myLocation, angle - Utils.HALF_PI, -1);
			robot.setTurnRightRadians(angle);
			robot.setAhead(100);
		}
	}

	private SoundWave getClosestWave() {
		double distance = Double.POSITIVE_INFINITY;
		SoundWave wave = null;

		for (int i = 0; i < waves.size(); i++) {
			SoundWave w = waves.get(i);
			double dist = w.firedLocation.distance(robot.getX(), robot.getY()) - w.distanceTraveled;

			if (distance > w.velocity && dist < distance) {
				wave = w;
				distance = dist;
			}
		}
		return wave;
	}

	// CREDIT: rozu
	private Point2D.Double predictFutureLocation(SoundWave w, int sign) {
		int futureTicks = 3;

		Point2D.Double predictedLocation = new Point2D.Double(robot.getX(), robot.getY());
		double predictedVelocity = robot.getVelocity();
		double predictedHeading = robot.getHeadingRadians();
		double maxTurnRate, turnAngle, moveDir;

		boolean intercepted = false;

		do {
			turnAngle = wallSmooth(predictedLocation, Utils
					.absoluteBearing(w.firedLocation, predictedLocation)
					+ (sign * (Math.PI / 2)), sign)
					- predictedHeading;
			moveDir = 1;

			if (Math.cos(turnAngle) < 0) {
				turnAngle += Math.PI;
				moveDir = -1;
			}

			turnAngle = Utils.normalizeBearing(turnAngle);

			maxTurnRate = Math.PI / 720D * (40D - 3D * Math.abs(predictedVelocity));
			predictedHeading = Utils.normalizeBearing(predictedHeading
					+ Utils.limit(-maxTurnRate, maxTurnRate, turnAngle));

			predictedVelocity += (predictedVelocity * moveDir < 0 ? 2 * moveDir : moveDir);
			predictedVelocity = Utils.limit(-8, 8, predictedVelocity);

			predictedLocation = Utils
					.project(predictedLocation, predictedHeading, predictedVelocity);

			futureTicks++;

			if (predictedLocation.distance(w.firedLocation) < w.distanceTraveled
					+ (futureTicks * w.velocity) + w.velocity)
				intercepted = true;
		} while (!intercepted && futureTicks < 500);

		return predictedLocation;
	}

	private double getDanger(SoundWave w, int sign) {
		Point2D.Double futureLocation = predictFutureLocation(w, sign);

		int bin = (int) Math.round((Utils.normalizeBearing(Utils
				.absoluteBearing(w.firedLocation, futureLocation)
				- w.bearing))
				/ (w.sign * BIN_WIDTH))
				+ MIDDLE_BIN;

		bin = (int) Utils.limit(0, BINS - 1, bin);

		if (!isEmpty(w.segmentSlow))
			return w.segmentSlow[bin];
		else
			return w.segmentFast[bin];
	}

	// CREDIT: David Alves? PEZ? I really don't know
	private double wallSmooth(Point2D.Double source, double angle, int sign) {
		while (!fieldRect.contains(Utils.project(source, angle, 160))) {
			angle += 0.05 * sign;
		}
		return angle;
	}

	private boolean isEmpty(double[] segment) {
		for (int i = 0; i < segment.length; i++)
			if (segment[i] != 0.0) {
				return false;
			}
		return true;
	}

	public void onHitByBullet(HitByBulletEvent e) {
		if (!waves.isEmpty()) {
			Point2D.Double hitLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet()
					.getY());
			SoundWave hitWave = null;

			for (int i = 0; i < waves.size(); i++) {
				SoundWave w = waves.get(i);

				if (Math.abs(w.distanceTraveled
						- w.firedLocation.distance(robot.getX(), robot.getY())) < 50
						&& Math.round(Utils.bulletVelocity(e.getBullet().getPower()) * 10) == Math
								.round(w.velocity * 10)) {
					hitWave = w;
					break;
				}
			}

			if (hitWave != null) {
				int bin = (int) Math.round((Utils.normalizeBearing(Utils
						.absoluteBearing(hitWave.firedLocation, hitLocation)
						- hitWave.bearing))
						/ (hitWave.sign * BIN_WIDTH))
						+ MIDDLE_BIN;

				for (int i = 0; i < BINS; i++) {
					hitWave.segmentSlow[i] += 1D / (Math.pow(bin - i, 2) + 1);
					hitWave.segmentFast[i] += 1D / (Math.pow(bin - i, 2) + 1);
				}

				waves.remove(waves.lastIndexOf(hitWave));
			}
		}
	}

	public void onBulletHitBullet(BulletHitBulletEvent e) {
		if (!waves.isEmpty()) {
			Point2D.Double hitLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet()
					.getY());
			SoundWave hitWave = null;

			for (int i = 0; i < waves.size(); i++) {
				SoundWave w = waves.get(i);

				if (Math.abs(w.distanceTraveled
						- w.firedLocation.distance(robot.getX(), robot.getY())) < 50
						&& Math.round(Utils.bulletVelocity(e.getBullet().getPower()) * 10) == Math
								.round(w.velocity * 10)) {
					hitWave = w;
					break;
				}
			}

			if (hitWave != null) {
				int bin = (int) Math.round((Utils.normalizeBearing(Utils
						.absoluteBearing(hitWave.firedLocation, hitLocation)
						- hitWave.bearing))
						/ (hitWave.sign * BIN_WIDTH))
						+ MIDDLE_BIN;

				for (int i = 0; i < BINS; i++) {
					hitWave.segmentSlow[i] += 1D / (Math.pow(bin - i, 2) + 1);
					hitWave.segmentFast[i] += 1D / (Math.pow(bin - i, 2) + 1);
				}

				waves.remove(waves.lastIndexOf(hitWave));
			}
		}
	}

	public void updateWaves() {
		for (int i = 0; i < waves.size(); i++) {
			SoundWave w = waves.get(i);

			w.distanceTraveled = (robot.getTime() - w.fireTime) * w.velocity;
			if (w.distanceTraveled > w.firedLocation.distance(robot.getX(), robot.getY()) + 50) {
				waves.remove(i);
				i--;
			}
		}
	}

	public void clear() {
		waves.clear();
	}

	public void paintWaves(java.awt.Graphics2D g) {
		for (SoundWave w : waves)
			Utils
					.drawCircle(w.firedLocation.x, w.firedLocation.y, w.distanceTraveled, java.awt.Color.YELLOW, g);
	}
}

class SoundWave {
	public double[] segmentSlow;
	public double[] segmentFast;
	public boolean firing;
	public Point2D.Double firedLocation;
	public double velocity, distanceTraveled, bearing, sign;
	public long fireTime;
}

class Utils {
	private final static double PI = Math.PI;
	public final static double TWO_PI = 2 * PI;
	public final static double HALF_PI = PI / 2.0;

	// CREDIT: Erm...
	public static double absoluteBearing(java.awt.geom.Point2D.Double p,
			java.awt.geom.Point2D.Double q) {
		return Math.atan2(q.x - p.x, q.y - p.y);
	}

	// CREDIT: Jonathon (?)
	public static double normalizeHeading(double angle) {
		return (angle % TWO_PI + TWO_PI) % TWO_PI;
	}

	// CREDIT: Jonathon (?)
	public static double normalizeBearing(double angle) {
		return normalizeHeading(angle + PI) - PI;
	}

	public static double bulletVelocity(double power) {
		return 20 - 3 * power;
	}

	public static double bulletDamage(double power) {
		return 4 * power + 2 * Math.max(power - 1, 0);
	}

	public static double bulletHeat(double power) {
		return 1 + (power / 5);
	}

	public static int sign(double a) {
		return a < 0 ? -1 : 1;
	}

	public static double limit(double min, double max, double value) {
		return Math.min(max, Math.max(min, value));
	}

	// CREDIT: PEZ (?)
	public static java.awt.geom.Point2D.Double project(java.awt.geom.Point2D.Double source,
			double angle, double length) {
		return new java.awt.geom.Point2D.Double(source.x + Math.sin(angle) * length, source.y
				+ Math.cos(angle) * length);
	}

	public static void drawCircle(double centerX, double centerY, double radius, java.awt.Color c,
			java.awt.Graphics2D g) {
		int upperX = (int) (centerX - radius);
		int upperY = (int) (centerY - radius);
		int diameter = (int) (radius * 2);
		g.setColor(c);
		g.drawOval(upperX, upperY, diameter, diameter);
	}
}