package rc.yoda.plugin;

import robocode.*;
import java.io.File;
import java.awt.Graphics2D;
import rc.yoda.utils.Factory;

/** 
 * PluginManager - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * PluginManager.java : v1.0 -- 2007/05/12
 */


/**
 * PluginManager a class that loads and 
 * interfaces between a robot and its 
 * plugins 
 *
 * @author Robert Codd
 * @version v1.0
 */
public class PluginManager 
{
	/**
	 * The class Path where the Plugin interface resides
	 */
	private static final String PLUGIN_CLASS_PATH = "rc.yoda.plugin.Plugin";
	
	/**
	 * A reference to the AdvancedRobot this PluginManager is working for
	 */
	private AdvancedRobot robot;
	
	/**
	 * An array of Plugin that this PluginManager is currently controlling
	 */
	private Plugin[] plugins;
	
	/**
	 * Class Constructor specifing the AdvancedRobot this PluginManager is working for
	 */
	public PluginManager(AdvancedRobot robot) {
		this.robot = robot;	
		init();
	}

	/**
	 * Initialzes need aspects of this PluginManger
	 */
	private void init() {
		populateArray();
	}

	/**
	 * Retrives an array of Classes that implement the Plugin interface
	 * in this Classes directory from the Factory Class
	 */
	private void populateArray() {
		String packageName = "",
			canonicalPath = "";
		Object[] parameters = {(Object)robot};	
		
		/*{
			Class thisClass = getClass();
			String name = thisClass.getName(),
				simpleName = thisClass.getSimpleName(),
				dir = robot.getDataDirectory().getPath();
				 
			packageName = name.substring(0, name.length() - simpleName.length() - 1);
			canonicalPath = dir.split(robot.getName().split(" ")[0].concat(".data"))[0] 
				+ packageName.replace(".",File.separator);			
		}*/
		
		try {
			Factory factory = new Factory(Class.forName(PLUGIN_CLASS_PATH), parameters, new Class[]{
				GunArray.class,
				MovementArray.class,
			});
			plugins = (Plugin[]) factory.getClasses();
		} catch (ClassNotFoundException e) {}		
	}

	/**
	 * Event method called by this robot when it detects
	 * that the enemy fired a bullet
	 *
	 * @param double deltaEnergy the power of the bullet fired
	 */
	public void onRobotFire(double deltaEnergy) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onRobotFire(deltaEnergy); }
	} 
	
	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onScannedRobot(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */	
	public void onHitByBullet(HitByBulletEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onHitByBullet(e); }		
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hits another robot
	 * 
	 * @param BulletHitEvent information about the robot 
	 * that got hit by the bullet
	 */	
	public void onBulletHit(BulletHitEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onBulletHit(e); }		
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hit a wall
	 *
	 * @param BulletMissedEvent information about the bullet
	 */	
	public void onBulletMissed(BulletMissedEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onBulletMissed(e); }		
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */	
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onBulletHitBullet(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot hits a wall
	 * 
	 * @param HitWallEvent information about the wall
	 */	
	public void onHitWall(HitWallEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onHitWall(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot 
	 * collides with another robot
	 *
	 * @param HitRobotEvent information about the collision and 
	 * the other robot in the crash
	 */		
	public void onHitRobot(HitRobotEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onHitRobot(e); }		
	}
	
	/**
	 * Event method called by Robocode when a robot dies
	 *
	 * @param RobotDeathEvent name of decaesed robot
	 */		
	public void onRobotDeath(RobotDeathEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onRobotDeath(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot
	 * wins a round
	 *
	 * @param WinEvent
	 */	
	public void onWin(WinEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onWin(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot dies
	 *
	 * @param DeathEvent 
	 */	
	public void onDeath(DeathEvent e) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onDeath(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */	
	public void onPaint(Graphics2D g) {
		for (int count = 0; count < plugins.length; count++) { plugins[count].onPaint(g); }		
	}
}