package DTF;
import robocode.*;
import robocode.util.*;
import sun.reflect.ReflectionFactory.GetReflectionFactoryAction;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.util.*;

import DTF.modules.GenePool;
import DTF.modules.Pattern;
import DTF.modules.PatternMatcher;


// API help : http://robocode.sourceforge.net/docs/robocode/robocode/Robot.html

/**
 * Kludgy - David Finch
 * Originally named TestBot.
 * Renamed to AntiSample before first release, because I initially optimized it against the samples.
 *   Initially had predictive targeting (with multiple prediction modes), could get out of corners, and dodge head on bullets, but not much else. 
 *   Also had some melee optimizations, choosing one target at a time to focus on.
 * Renamed to Kludgy after optimizing against better robots, most of which still beat it easily.
 *   Added what I thought was pattern matching, but matches single samples rather than a series, to improve targeting and dodging.
 *   Improved wall/corner avoidance.
 *   Added genetic algorithm trainer to tune various strategic values and fudge factors during a long match.
 * 1.2:
 *   improved corner avoidance.
 *   added dive protection.
 *   randomly not dodge, to reduce predictability.
 *   various additions with little visible improvement on score. Probably better to start fresh now.
 */

public class Kludgy extends AdvancedRobot
{
	//below variables are managed by genetic algorithms. values assigned here are meaningless except to indicate typical value
	static double seesaw_distance_factor=0.5; //factor in getSeesawDistance()
	static double miss_threshold=0.7; //multiplied by height to determine fire error threshold
	static double turn_on_hit=30;
	static double value_base=200; // melee
	static double pest_value_bonus=50.0; // melee
	static double target_switch_threshold=0.05; // melee
	static double avoid_robot_distance=200;
	static double avoid_robot_turn=20;
	static double avoid_wall_pct=0.2; //really avoid_corner
	static double turn_on_dodge=30.0;
	
	static double dodge_base=100;
	static double dodge_range_factor=0.2;
	static double dodge_random=0.2;
	
	static double fire_normal=2.0;
	static double fire_more_range=200;
	static double fire_less_range=400;
	static double fire_less_factor=0.75;
	
	static double wall_stick=130.0;
	
	static double max_dodge_until=40.0;
	
	static double wall_extra_pad=10.0;
	
	static double dive_threshold=30;
	static double dive_fudge=0.25; //0 to 0.5
	
	static double dodge_probability=0.75; //confuse predictors
	
	static double vary_move=100;
	static double vary_vary_move=0.3;
	
	//patternmatcher.match_threshold ~4000

	static double fire_bonus_melee=0.1; //multiplied by others-1 and added to fire power.
	
	static double silent_others=3; //todo improve silent mode and rename
	
	static double move_on_hit=100; // tweaked to remove unuseful midrange
	
	static double front_angle=45;
	
	static double run_distance=100; //for melee
	
	
	//todo something to let it dodge less in melee
	
	
	//above values are really guesses, and best values would vary between situations.
	//Use genetic algorithms starting from best-guess min/max values to optimize in each match.
	//Then after several matches against several bots, I look at what it came up with and decide how the min/max ranges can be improved.
	
	static double[] genes_min={0.2, 0.65, 25, 200,  50, 0.01,  200, 10, 0.15, 25,  20, 0.05, 0.5, 1.3, 120, 240, 0.2,  120, 20,  0, 12, 0.07, 0.6,  40, 0.2,  5000, 0.1, 3, -20, 25,  70};
	static double[] genes_max={0.7, 1.1,  50, 350, 100, 0.15,  350, 40, 0.5,  60, 150, 0.4,  2,   2.8, 300, 440, 0.65, 160, 75, 15, 35, 0.4,  1.1, 120, 0.5, 12000, 0.3, 9, 80, 55, 130};
	
	//a downside is that sometimes just changing each match works to its advantage, and when it settles down it becomes more predictable.
	//may need to introduce occasional mutation.

	
	static final double wall_pad=20;
	
	static GenePool genepool;
	
	int fireCount=0;
	
	static PatternMatcher myPattern=new PatternMatcher();
	
	
	//static to preserve stats between games, though not always a great idea.
	static Map<String,VirtualBot> enemies=new HashMap<String,VirtualBot>();
	
	int dodgeUntil=0;
	
	int cursorX=0;
	int cursorY=0;
	Color cursorColor=Color.black;
	
	String target; //current robot we're targeting
	String pest; //name of a recent pest that we give extra attention to
	
	double targetValue ;
	
	double lastrot;
	
	double movedir;
	
	boolean hitWall=false;
	
	boolean dodged=false;
	
	boolean headon=false;
	
	boolean silent=false; //todo rename
	
	double score;
	
	//start with safe "hasn't fired yet" values
	int lastFiredTurns=-100;
	int lastFiredAtTurns=-100;
	int lastTargetScanDist=500;
	
	private void geneInit() {
		if(genepool==null) {
			genepool=new GenePool((int)Math.sqrt(getNumRounds()), genes_min, genes_max);
		}
		double[] g=genepool.getCurrent(getRoundNum());
		seesaw_distance_factor=g[0];
		miss_threshold=g[1]; 
		turn_on_hit=g[2];
		value_base=g[3];
		pest_value_bonus=g[4];
		target_switch_threshold=g[5];
		avoid_robot_distance=g[6];
		avoid_robot_turn=g[7];
		avoid_wall_pct=g[8];
		turn_on_dodge=g[9];
		
		dodge_base=g[10];
		dodge_range_factor=g[11];
		dodge_random=g[12];
		
		fire_normal=g[13];
		fire_more_range=g[14];
		fire_less_range=g[15];
		fire_less_factor=g[16];	
		
		wall_stick=g[17];
		max_dodge_until=g[18];
		
		wall_extra_pad=g[19];
		
		dive_threshold=g[20];
		dive_fudge=g[21];
		
		dodge_probability=g[22];
		
		vary_move=g[23];
		vary_vary_move=g[24];
		
		PatternMatcher.match_threshold=(int)g[25];
		
		fire_bonus_melee=g[26];
		
		silent_others=g[27];
		
		move_on_hit=g[28];
		if(move_on_hit<0) move_on_hit-=30;
		else if(move_on_hit>40) move_on_hit+=50;
		else move_on_hit=0;
		
		front_angle=g[29];
		
		run_distance=g[30];
		if(run_distance<50) run_distance=0;
		
		assert(g.length==31);
			
		score=0;
	}
	
	
	public void run() {
		Util.init(this); //maybe unsafe with multiple instances, if we were to add more specific info.
		
		geneInit();
		
		setColors(Color.blue, Color.yellow, Color.black);
		
		setAdjustRadarForGunTurn(false);
		setAdjustRadarForRobotTurn(false);
		setAdjustGunForRobotTurn(false);
		
		for(VirtualBot e:enemies.values()) {
			e.dead=false;
			e.known=false;
		}
		
		//todo decide best initial move direction
		movedir=0.5;
		//todo decide best initial radar direction
		
		double bias=0;
		
		while(true) {
			silent=(getOthers()>=silent_others);
			
			//todo tracking when we have a target
			if(getRadarTurnRemaining()==0) {
				setTurnRadarRight(Double.POSITIVE_INFINITY);
			}
			double dist=getDistanceRemaining();
			if(Math.abs(dist)<1) {
				//reduces predictability and mixes things up a little
				if(silent && !dodged /* && Math.random()<0.5*/){
					avoid(angleToCenter()-getHeading(),getSeesawDistance(),0.0);
				} else {
					bias=bias*(1.0-vary_vary_move)+(Math.random()*2.0-1.0)*vary_move*vary_vary_move;
					setAhead(bias+getSeesawDistance()*movedir*(hitWall?1.2:1.0));
					movedir=(movedir>0)?-1:1;
				}
				
				avoidCorners();
				dodged=false;
				hitWall=false;
			} else {
				//correct misbehaviour when we reverse to dodge something
				movedir=(dist>0)?-1:1;
			}
			avoidWalls();
			
			myPattern.addPattern(new Pattern((int)getX(),(int)getY(), (int)getHeading(), getVelocity(), (int)getTime(), (int)getTime()-lastFiredAtTurns, lastTargetScanDist));
			
			
			execute();
		}
	}
	
	private double angleToCenter() { //todo rename now that it's weighted
		//todo replace with antigravity for melee navigation
		double sumWeight=0;
		double x=0,y=0;
		double myX=getX(), myY=getY();
		
		for(VirtualBot e:enemies.values()) {
			if(e.known && !e.dead) { //todo optimize weighting
				double weight=1/(100+Util.distance(e.X-myX, e.Y-myY));
				if(pest!=null && e.name.equals(pest)) weight*=2.0;
				
				sumWeight+=weight;
				x+=e.X*weight;
				y+=e.Y*weight;
			}
		}
		if(sumWeight>0) {
			x/=(double)sumWeight;
			y/=(double)sumWeight;
		} else {
			x=getBattleFieldWidth()*0.5;
			y=getBattleFieldHeight()*0.5;
		}
		return Util.heading(x-myX, y-myY);
	}
	
	private boolean isDive(double moveHeading, double headingToEnemy, double threshold) {
		return /*!silent &&*/ Math.abs((headingToEnemy-moveHeading+630)%180-90) < threshold;
	}

	private boolean isDive(double startHeading, double turnBy, double headingToEnemy, double threshold) {
		//if we'll cross the no-turn zone. probably incorrect for large values of turnBy 
		//0.6 and 0.4 instead of 0.5, 0.5 to give us a way out in a corner rather than get stuck in a dive
		boolean ret=/*!silent &&*/ Math.abs((headingToEnemy-(startHeading+turnBy*(0.5+dive_fudge))+630)%180-90) < threshold+Math.abs(turnBy*(0.5-dive_fudge));
		//out.println(startHeading+", "+turnBy+", "+headingToEnemy+", "+threshold+": "+ret);
		return ret;
	}
	
	
	//called once every turn
	//todo, maybe base wall_stick on distance remaining, and turn to towards target rather than either way
	//todo dive protection, when it's better to reverse than rotate
	private void avoidWalls() {
		if(getTurnRemaining()==0) {
			double x=getX();
			double y=getY();
			double h=getHeading();
			double v=getVelocity();
			if(v<0) h=(h+180)%360;
			v=Math.abs(v);
			
			boolean ignoreDive=!allowDodge();
			
			double gh=getGunHeading();
			boolean headingAway=false;
			for(int i=0;i<360;i++) {
				int dh=(i>>1);
				if((i&1)==1) dh=-dh;
				
				double x2=Util.sin(dh+h)*wall_stick+x; 
				double y2=Util.cos(dh+h)*wall_stick+y;
				if(Util.inArena(x2, y2, wall_extra_pad)) {
					if(i==0) headingAway=true;
					if(ignoreDive || !isDive(h, dh, gh, dive_threshold)) {
						if(dh!=0) {
							if(Math.abs(dh)>85) { //todo optimize
								// if abs(dh)>90, there's probably a better way out than turning (like reversing) and we ought to find it								
								//setTurnRight((dh+270)%180-90);
								//todo simplify
								if(!headingAway) { //bounce to avoid hit
									setTurnRight(0);
									setStop(true);
								}
							} else {
								setTurnRight(dh);
							}

						}
						return;
					}
				}
			}
			//should we stop moving here?
		}
	}
	
	
	//like setTurnRight, but no-ops if it would send us into a wall
	private boolean safeTurnRight(double dh) {
		//todo fix ugly duplication between this and avoidWalls
		double x=getX();
		double y=getY();
		double h=getHeading();
		if(getVelocity()<0) h=(h+180)%360;
		double x2=Util.sin(dh+h)*130.0+x; //todo optimize feeler distance
		double y2=Util.cos(dh+h)*130.0+y;
		if(Util.inArena(x2, y2)) {
			if(dh!=0) {
				//extra work is to detect if any angle between h and dh+h would be a dive
				if(!(allowDodge() && isDive(h,dh, getGunHeading(), dive_threshold))) {
					setTurnRight(dh);
					return true;
				}
			}
		}
		return false;
	}
	
	//called once per seesaw 
	private void avoidCorners() {
		if(getTurnRemaining()==0) {
			//todo find something better. This degrades it into a corner to corner diagonal center bot.
			
			//strategy: find the which of 4 directions we're heading in, and turn towards the associated corner
			double w=getBattleFieldWidth();
			double h=getBattleFieldHeight();
			double x=getX();
			double y=getY();
			double tx,ty;
			
			double heading=getHeading();
			boolean flip=false;
			if(getDistanceRemaining()<0.0) {
				heading=Util.normalHeading(heading+180.0);
				flip=true;
			}
			
			if(heading%90==0) return; //no change needed
			
			if(heading<90) {
				tx=w-wall_pad;
				ty=h-wall_pad;
			} else if(heading<180) {
				tx=w-wall_pad;
				ty=wall_pad;
			} else if(heading<270) {
				tx=wall_pad;
				ty=wall_pad;
			} else {
				tx=wall_pad;
				ty=h-wall_pad;
			}
			
			//average with position opposite our current arena position.
			tx=(tx+(w-x))/2;
			ty=(ty+(h-y))/2;
			
			double turn=Util.normalBearing(Util.heading(tx-x, ty-y)-heading) * (hitWall?0.5:avoid_wall_pct);
			if(Math.abs(turn)>=1.0) {
				setTurnRight(turn);
			}
		}
		
	}
	
	private double getSeesawDistance() {
		double area=getBattleFieldHeight()*getBattleFieldHeight();
		area = area/getOthers()+1;
		return Math.sqrt(area)*seesaw_distance_factor;
	}
	

	public void onRobotDeath(RobotDeathEvent e) {
		if(target!=null && e.getName().equals(target)) {
			target=null;
		}
		if(enemies.containsKey(e.getName())) {
			enemies.get(e.getName()).dead=true;
		}
		score+=50;
		out.println(e.getName()+" died");
	}
	
	private VirtualBot getEnemy(String name) {
		if(enemies.containsKey(name)) {
			return enemies.get(name);
		} else {
			VirtualBot enemy=new VirtualBot();
			enemy.name=name; //todo this line is dirty
			enemies.put(name, enemy);
			return enemy;
		}
		
	}
	
	private boolean allowDodge() {
		if(silent && (target==null || pest==null || !target.equals(pest))) return false;
		return true;
	}
	
	private void onFiredAt(ScannedRobotEvent e, double energy) {
		//in silent mode, ignore unless they're the last one who shot us.

		double range=e.getDistance();
		
		if(!allowDodge() || (silent && range>100)) return; //todo optimize
		
		//todo make constants
		double distrmn=getDistanceRemaining();
		
		//todo queue bullets to avoid rather than blindly ignoring all the bullets already in the air to react to the newest
		
		int turns=(int)getTime();
		
		
		if(/*e.getDistance()<400 &&*/ Math.abs(distrmn)>0 /* && Math.abs(distrmn)<50*/ && (!dodged || dodgeUntil<=turns)) {
			
			if(Math.random()>dodge_probability) return;
			
			double speed=(20.0-energy*3.0);
			
			int turnsAway=(int)(getTime()+range/speed);
			
			Pattern p=myPattern.predictFromCurrent(turnsAway);
			
			double dodgeDistance =  (dodge_base + dodge_range_factor*(range-dodge_base))*(1.0+dodge_random*(Math.random()-0.5));  //Math.min(range/5.0+100.0, getSeesawDistance()) ; //todo optimize
			
			if(dodgeDistance<60) dodgeDistance=60;
			
			dodgeUntil=turns+Math.min(turnsAway,(int)max_dodge_until); //maybe a linear factor+base would be better
			
			if(p!=null) {
				//out.println("Dodging our own pattern");
				
				//todo determine if this helps at all
				
				double heading=Util.heading(p.x-getX(), p.y-getY());
				double bearing=heading+getHeading();
				
				avoid(bearing, dodgeDistance, 0.0);				
			} else {
				//out.println("Reversing to avoid bullet");
				//if(Math.abs(distrmn)<25 && Math.random()<0.2) dodgeDistance=-dodgeDistance;
				
				//if(Math.abs(distrmn)<5) {
				//	setAhead(Math.signum(distrmn)*dodgeDistance);
				//} else {
					setBack(Math.signum(distrmn)*dodgeDistance);
				//}
			}
			//todo keep score to decide whether it's better to assume they're predicting us or just firing straight at us.
			
			
			
			dodged=true;
		}
		avoid(e.getBearing(),0.0,turn_on_dodge);
		
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		double dist=e.getDistance();
		double scanHeading=e.getBearing()+getHeading();
		
		String name=e.getName();

		boolean isPest=(getOthers()==1);
		
		//target nearby robots with little energy
		double value=value_base-Math.min(e.getEnergy(),100.0); //todo track damage to other robots because we get a killshot bonus based on total damage to them.
		if(pest!=null && name.equals(pest)) {
			value+=pest_value_bonus;
			isPest=true;
		}
		if(value<10) value=10;
		
		//hard to shoot opponents are of lesser value
		double pathBonus=(Util.cos(2*(e.getHeading()-scanHeading))+3.0)/(8.0+Math.abs(e.getVelocity())); //todo refine
		
		value=value*pathBonus/dist; //further enemies are less of a concern
		
		
		VirtualBot enemy=getEnemy(name);
		
		enemy.Scanned(e); //track the event
		
		//decide whether this robot is worth targeting
		if(target==null || value>targetValue+target_switch_threshold || name.equals(target)) {
			target=name;
			targetValue=value;
		}
		
		boolean isTarget=name.equals(target);
		
		if((isPest || isTarget) && enemy.fired) {
			//assume they fired at us
			onFiredAt(e,enemy.fireEnergy);
			lastFiredAtTurns=(int)getTime();
		}
		
		if(silent && dist<run_distance && getTurnRemaining()==0) {
			avoid(e.getBearing(),run_distance+100,20);
		}
		
		
		if(isTarget) {
			lastTargetScanDist=(int)dist;
			
			
			//double predictDist=0;
			double predictAngle=0;
			int turnsAway=0;
			double rot=0;
			double dist2=dist; //dist2 is predicted distance when our missile would hit them.
			//double traveldist=0;
			double dx=0,dy=0;
			
			
			
			//todo move fire energy calculation
			//calculate firing energy early because we need it to determine bullet speed
			double myEnergy=getEnergy();
			double energy=e.getEnergy();
			double fireEnergy;
			//calculate energy needed to kill
			if(energy<=4) {
				fireEnergy=energy/4.0;
			} else {
				fireEnergy=1.0+(energy-4.0)/6.0;
			}
			double fireLimit=fire_normal;
			if((fireCount&1)==1 || dist<fire_more_range || getOthers()>1) fireLimit=3.0;
			if(dist>Math.max(fire_more_range, fire_less_range)) fireLimit*=fire_less_factor;
			
			//to score higher damage in melee matches
			fireLimit=Math.min(Math.max(fireLimit+fire_bonus_melee*(double)(getOthers()-1),0.1),3.0);
			
			fireEnergy=Math.max(fireEnergy+0.001,0.1); //add a little to ensure kill, then bump up to minimum if necessary
			if(fireEnergy>3.0) fireEnergy=Math.min(fireEnergy,fireLimit); //cap unless we can kill in 1 shot
			fireEnergy=Math.min(fireEnergy,myEnergy-0.1); //avoid firing with enough to kill ourselves


			
			
			double bulletSpeed=20.0-(3.0*fireEnergy);

			VirtualBot prediction=new VirtualBot(); //todo avoid unnecessary object creation
			VirtualBot predictMe=new VirtualBot();
			int lastTurnsAway=0;
			int gunTurns,lastGunTurns=0;
			for(int i=0;i<10;i++) {
				//take our own movement into account, to minimize correction needed on the last turn when we fire 
				
				gunTurns=(int)Math.ceil(Math.abs(rot/20)-1); //subtract a little because assume we won't bother turning the gun on that last turn when it's close
				
				if(gunTurns>=lastGunTurns && i>0) {
					predictMe.Drive(gunTurns-lastGunTurns);
				} else {
					predictMe.copyFrom(this);
					predictMe.Drive(gunTurns);
				}
				lastGunTurns=gunTurns;
				
				
				//bullet speed is 20-3*power. assume 11
				//todo optimize the value subtracted from dist2, though it seems to work so far
				turnsAway=(int)Math.ceil((dist2-15)/bulletSpeed + Math.abs(rot/20));

				if(turnsAway>=lastTurnsAway && i>0) {
					//slight risk here if Drive(a)+Drive(b) does not always match Drive(a+b)
					if(turnsAway==lastTurnsAway) break; //nothing more to do
					prediction.Drive(turnsAway-lastTurnsAway);
				} else {
					prediction.copyFrom(enemy);
					prediction.Drive(turnsAway); 
				}
				lastTurnsAway=turnsAway;
				
				dx=prediction.X-predictMe.X;
				dy=prediction.Y-predictMe.Y;
				
				predictAngle=Util.heading(dx, dy);
				dist2=Util.distance(dx, dy);
				
				rot=Util.normalBearing(predictAngle - getGunHeading());
				
				
			}
			//headon always false at the moment
			if(headon) rot=(scanHeading-getGunHeading()+540)%360-180;
			
			
			setTurnGunRight(rot);
			lastrot=rot;
			
			double miss=Util.tan(rot)*dist2; //rough distance we expect to miss center by if we fire now
		
			// if miss estimate less than approx robot radius
			
			boolean firing=false;
			
			if(getGunHeat()==0 && Math.abs(miss)<getHeight()*miss_threshold) {
				cursorX=(int)prediction.X;
				cursorY=(int)prediction.Y;
				cursorColor=prediction.usedMatcher?Color.green:Color.red;

				
				
				firing=(fireEnergy>=0.1);
				
				if(myEnergy<0.1 && energy<myEnergy*4.0 && getOthers()==1) {
					//rare 1 on 1 standoff where we have 0.1 or less left but it's enough to kill them
					//go ahead and fire
					fireEnergy=myEnergy;
					firing=true;
				}
				
				if(firing) {
					enemy.firedAt(); //so that it knows to benchmark
					setFire(fireEnergy);
					score-=fireEnergy;
					fireCount++;
					lastFiredTurns=(int)getTime();
				}
			}
			//radar tracking
			//if facing many opponents, give the radar a good free spin for a couple turns after firing to avoid tunnel vision
			if((!firing && getGunHeat()<1.45) || getOthers()==1) {
				double trackTurn=Util.normalBearing(scanHeading-getRadarHeading());
				trackTurn+=Math.signum(trackTurn)*20.0;
				setTurnRadarRight(trackTurn);
			}
		}
	}

	public void onHitByBullet(HitByBulletEvent e) {
		//todo make damage threshold a constant
		if(e.getPower()>=1.0) {
			avoid(e.getBearing(), silent?0.0:move_on_hit, turn_on_hit);
			if(move_on_hit!=0) {
				//prevent interference from dodging
				dodgeUntil=(int)(getTime()+max_dodge_until);
				dodged=true;
			}
			pest=e.getName();
		}
		//setTurnRight(turn_on_hit);
		score-=e.getPower()*8.0/(getOthers()+1); //because our loss is their gain.
	}
	
	public void onBulletHit(BulletHitEvent e) {
		score+=e.getBullet().getPower()*(4.0+3.0); //because we subtract score when we fire, even though real score isn't affected, we add another +3 when we hit
	}
	
	public void onHitWall(HitWallEvent e) {
		hitWall=true;
	}

	public void onHitRobot(HitRobotEvent e) {
		avoid(e.getBearing(), avoid_robot_distance, avoid_robot_turn);
		pest=e.getName();
	}
	
	private void avoid(double bearing, double distance, double turn) {
		//todo see if we need to worry about corner hooking.
		double rot=Math.signum(bearing);
		double front=Math.abs(Util.normalBearing(bearing));
		if(front<90) {
			if(distance!=0.0) setBack(distance);
			rot=-rot;
		} else {
			if(distance!=0.0) setAhead(distance);
		}
		//check front to avoid excessive adjustments
		if(getTurnRemaining()==0 && (front<front_angle || front>(180-front_angle)) && turn!=0) safeTurnRight(rot*turn); //avoid interfering with other things
	}
	

	public void onPaint(Graphics2D g) {
		VirtualBot prediction=new VirtualBot();
		double now=getTime();
		for(String name:enemies.keySet()) {
			VirtualBot enemy=enemies.get(name);
			if(enemy.dead) continue;
			prediction.copyFrom(enemy);
			prediction.Drive((int)(now-prediction.asof));
			
			if(target!=null && name.equals(target)) {
				g.setColor(Color.red);
			} else {
				g.setColor(Color.green);
			}
			
			g.drawLine((int)enemy.X, (int)enemy.Y, (int)prediction.X, (int)prediction.Y);
		}
		int h=(int)getHeight();
		g.setColor(cursorColor);
		g.drawRect(cursorX-h/2, cursorY-h/2, h, h);
		
	}
	
	public void onWin(WinEvent event) {
		//+50 because onWin is called before onRobotDeath is.
		score+=50+50; //the extra 50 is not a component of our score, but to counteract that others get 50 for surviving us.
		
		genepool.scoreCurrent(getRoundNum(), score);
		
		out.println("I won");
		
	}
	
	public void onDeath(DeathEvent event) {
		genepool.scoreCurrent(getRoundNum(), score);
		out.println("I lost");
		
	}
	
}
