package ag.battledata;

import ag.movement.GravPoint;
import java.util.Vector;

import ag.neuralgir.NeuralNet;
import ag.neuralgir.TrainDataSet;

/**
 * Battle data for a single robot
 * @author Andree Grosse
 */

public class DataRobot
{	
	private String mName = null;
	
	// variables for gun statistics
	private int mBulletsFiredAt, mBulletsHitBy;
	private int mBulletsFired, mBulletsHit;
	private double mAverageGirStats;
	private double mAverageRobotStats;
	private int mSizeGirStats;
	private int mSizeRobotStats;
	private boolean mRobotCollision;
	private boolean mDead;
	
	// variables for dodging
	private NeuralNet mDodgeNet; 
	private Vector<TrainDataSet> mTrainSet;
	private Vector<VirtualBullet> mVirtualBullets;

	private int mDodgeInputSize;
	
	private boolean mEnemy;
	
	private GravPoint mGravPoint;
	
	
	/*
	 * mEnergyChange[0] is the enemies energy change between the last 
	 * tick and the current tick, mEnergyChange is the change between
	 * the tick 'current - 2' and 'current - 1'. If there is no teammate,
	 * only mEnergyChange[0] should be used.
	 */
	private double mEnergyChange[];
	// the vector containing the robots data
	private Vector<DataRobotSnapshot> mData;
	
	// ######################### public #########################
	/**
	 * Constructor
	 */
	public DataRobot(String name, boolean enemy){
			mData = new Vector<DataRobotSnapshot>();
			mName = name;
			mBulletsFiredAt = 0;
			mBulletsHitBy = 0;
			mBulletsFired = 0;
			mBulletsHit = 0;
			
			mEnemy = enemy;
			mEnergyChange = new double[]{.0, .0};
			mRobotCollision = false;
			mDead = false;

			mSizeGirStats = 0;
			mSizeRobotStats = 0;
			
			
			mVirtualBullets = new Vector<VirtualBullet>();
			mTrainSet = new Vector<TrainDataSet>();
			mGravPoint = new GravPoint(new DataVector(), 220, 4.5, 300);
			
			System.out.println("**DataRobot created: " + mName);
	}
	
	// ####################### setter & getter #######################
	/**
	 * Returns the robot's name
	 */
	public String getName(){
		return mName;	
	}
	
	/**
	 * Returns the number of data used for average gun statistics
	 * of Gir against this robot.
	 */
	public int getSizeGirStats (){
		return mSizeGirStats ;
	}
	
	/**
	 * Returns the number of data used for average gun statistics
	 * of this robot against Gir.
	 */
	public int getSizeRobotStats (){
		return mSizeRobotStats ;
	}
	
	/**
	 * Sets the number of data used for average gun statistics
	 * of Gir against this robot.
	 */
	public void setSizeGirStats(int sizeGirStats){
		mSizeGirStats = sizeGirStats;
	}
	
	/**
	 * Sets the number of data used for average gun statistics
	 * of this robot against Gir.
	 */
	public void setSizeRobotStats(int sizeRobotStats){
		mSizeRobotStats = sizeRobotStats;
	}
	
	/**
	 * Returns the robot's data vector
	 */
	public Vector<DataRobotSnapshot> getData() {
		return mData;
	}	
		
	/**
	 * Returns the neural net used to predict the angle of the bullets
	 * fired from this robot
	 * @return the dodgeNet
	 */
	public NeuralNet getDodgeNet(){
		return mDodgeNet;
	}
	
	/**
	 * Returns the input size of the dodge net.
	 * @return the input size of the dodge net.
	 */
	public int getDodgeInputSize(){
		return mDodgeInputSize;
	}
	
	/**
	 * Returns if this robot is dead.
	 * @return true if this robot is dead, otherwise false.
	 */
	public boolean getDead(){
		return mDead;
	}
	
	/**
	 * Sets the dead variable for this robot.
	 * @param dead new value for mDead
	 */
	public void setDead(boolean dead){
		mDead = dead;
	}
	
	/**
	 * Returns the vector containing the training data for the dodge net for
	 * this robot.
	 * @return the 'TrainDataSet' vector
	 */
	public Vector<TrainDataSet> getTrainSet(){
		return mTrainSet;
	}
	
	/**
	 * Returns the average gun statistics for Gir on this robot
	 * @return the average gun statistics for Gir
	 */
	public double getAverageGirStats(){
		return mAverageGirStats;
	}
	
	/**
	 * Returns the average gun statistics for this robot on Gir
	 * @return the average gun statistics for this robot
	 */
	public double getAverageRobotStats(){
		return mAverageRobotStats;
	}
		
	/**
	 * Sets the average gun statistics for Gir
	 * @param girStats new value for the gun statistics
	 */
	public void setAverageGirStats(double girStats){
		mAverageGirStats = girStats;
	}
	
	/**
	 * Sets the average gun statistics for this robot
	 * @param robotStats new value for the gun statistics
	 */
	public void setAverageRobotStats(double robotStats){
		mAverageRobotStats = robotStats;
	}
	
	/**
	 * Returns if this robot is an enemy
	 * @return true if this robot is an enemy, otherwise false
	 */
	public boolean getEnemy(){
		return mEnemy;
	}
	
	/**
	 * Returns the mEnergyChange variable for this robot
	 * @param i variable to use (0 or 1)
	 * @return  the mEnergyChange[i] variable
	 */
	public double getEnergyChange(int i){
		return mEnergyChange[i];
	}
	
	/**
	 * Sets the energy change number i
	 * @param i variable number to set (0 or 1)
	 * @param value new value for the energychange
	 */
	public void setEnergyChange(int i, double value){
		mEnergyChange[i] = value;
	}
	
	/**
	 * Sets the neural network for dodging.
	 * @param net new dodge net
	 */
	public void setDodgeNet(NeuralNet net){
		mDodgeNet = net;
	}
	
	/**
	 * Return if there has been a robot collision
	 * @return mRobotCollision
	 */
	public boolean getRobotCollision(){
		return mRobotCollision;
	}
	
	/**
	 * Sets the mRobotCollision variable
	 * @param collision new value for mRobotCollision  
	 */
	public void setRobotCollision(boolean collision){
		mRobotCollision = collision;
	}
	
	/**
	 * Returns the GravPoint object for this robot
	 * @return the GravPoint object
	 */
	public GravPoint getGravPoint(){
		return mGravPoint;
	}
	
	/**
	 * Returns the Vector containing all virtual bullets for this robot
	 * @return the 'VirtualBullets' vector
	 */
	public Vector<VirtualBullet> getVirtualBullets(){
		return mVirtualBullets;
	}
	// ###############################################################

	/**
	 * Adjusts the energy change variable number i
	 * @param i variable to use (0 or 1)
	 * @param change change in energy (newchange = oldchange + change)
	 */
	public void adjustEnergyChange(int i, double change){
		if(mEnemy)
			mEnergyChange[i] += change;
	}
	
	/**
	 * Should be called if we fired at this robot and missed.
	 */
	public void missedThisRobot(){
		mBulletsFiredAt++;
	}
	
	/**
	 * Should be called if we hit this robot.
	 */
	public void hitThisRobot(){
		mBulletsFiredAt++;
		mBulletsHitBy++;
	}
	
	/**
	 * Should be called if this robot fired at Gir and missed.
	 */
	public void firedAtGir(){
		mBulletsFired++;
	}
	
	/**
	 * Should be called if this robot fired at Gir and hit.
	 *
	 */
	public void hitGir(){
		if(mBulletsHit < mBulletsFired)
			mBulletsHit++;
	}
	
	/**
	 * Return current gun statistics for Gir
	 * @return Current gun statistics for Gir
	 */
	public double getCurrentGirStats() {
		if(mBulletsFiredAt == 0)
			return -2;
		return (double) mBulletsHitBy / mBulletsFiredAt;
	}
	
	/**
	 * Return current gun statistics for this robot
	 * @return Current gun statistics for this robot
	 */
	public double getCurrentRobotStats() {
		if(mBulletsFired == 0)
			return -2;
		return (double) mBulletsHit / mBulletsFired;
	}
	
	/**
	 * Adds a data-snapshot to a robot's data vector
	 * @param data The snapshot to be added
	 */
	public void addDataRobotSnapshot(DataRobotSnapshot data) {
		mData.add(data);
		mGravPoint.setPosition(data.getPosition());
	}
	
	/**
	 * Returns the position of the robot at
	 * 'tick of last snapshot + minustick'.
	 */
	public DataVector getPosition(int minustick){
		return mData.get(mData.size() - 1 + minustick).getPosition();
	}
	
	/**
	 * Retrieves the (normalized) absoulte direction of the robot at
	 * 'tick of lastsnapshot + minustick'.
	 */
	public DataVector getAbsDirection(int minustick){
		return mData.get(mData.size() - 1 + minustick).getAbsDirection();
	}
	
	/**
	 * Retrieves the (normalized) direction of the robot at
	 * 'tick of lastsnapshot + minustick'.
	 */
	public DataVector getDirection(int minustick){
		return mData.get(mData.size() - 1 + minustick).getDirection();
	}
	
	/**
	 * Retrieves the absolute velocity of the robot at
	 * 'tick of lastsnapshot + minustick'.
	 */
	public double getAbsVelocity(int minustick){	
		return mData.get(mData.size() - 1 + minustick).getAbsVelocity();
	}
	
	/**
	 *  Retrieves the signed velocity of the robot at
	 * 'tick of lastsnapshot + minustick'.
	 */
	public double getVelocity(int minustick){		
		return mData.get(mData.size() - 1 + minustick).getVelocity();
	}
	
	/**
	 * Retrieves the energy of the robot at
	 * 'tick of lastsnapshot + minustick'.
	 */
	public double getEnergy(int minustick){		
		return mData.get(mData.size() - 1 + minustick).getEnergy();
	}
	
	/**
	 * Removes old data. Should be called
	 * at the end of every turn
	 */
	public void removeOldData(){
		if(mData.size() > 400)
			mData.remove(0);		
	}
	
	/**
	 * Get the snapshot at tick 'tick' (if it exists)
	 * @param tick tick of desired snapshot
	 * @return the snapshot if it exists, otherwise null
	 */
	public DataRobotSnapshot getSnapshot(long tick){
		DataRobotSnapshot data;
		for(int i = 1; mData.size() >= i; i++){
			data = mData.get(mData.size() - i);
			if(data.getTick() == tick)
				return data;
			else if(data.getTick() < tick)
				return null;
		}
		return null;
	}
	
	/**
	 * Check's if data for the last 'number' turns before (and including) 
	 * 'tick' are available
	 * @param tick tick for which to check
	 * @param number Number of turns to check
	 * @return (int)'tick - tick of latest element in the data' if there is
	 * 	consecutive data available (i.e. 0 means the data is up to date)
	 *  or -1 if the data is not consecutive 
	 */
	public int checkData(long tick, int number){
		if(mData.size() < number)
			return -1;
		
		int offset = (int)(tick - mData.lastElement().getTick());
		for(int i = 0; i < number; i++){
			if(mData.get(mData.size() - 1 - i).getTick() != tick - i - offset)
				return -1;
		}
		return offset;
	}
	
	/**
	 * Adds a virtual bullet to this robots 'VirtualBullets' vector
	 * @param vb the virtual bullet to add
	 */
	public void addVirtualBullet(VirtualBullet vb){
		mVirtualBullets.add(vb);
	}
	
}
