package ags.util;

import robocode.util.Utils;

public class PreciseIntersect {
    private double min = Double.POSITIVE_INFINITY;
    private double max = Double.NEGATIVE_INFINITY;
    private final double ox, oy, zeroAngle, maxEscapeAngle;
    private final int gfDirection;
    public PreciseIntersect(double ox, double oy, double zeroAngle, double maxEscapeAngle, int gfDirection) {
        this.ox = ox;
        this.oy = oy;
        this.zeroAngle = zeroAngle;
        this.maxEscapeAngle = maxEscapeAngle;
        this.gfDirection = gfDirection;
    }
    
    public Range getGFRange() {
        if (min == Double.POSITIVE_INFINITY) {
            return null;
        }
        double invMaxEscapeAngle = gfDirection/maxEscapeAngle;
        return new Range(min*invMaxEscapeAngle, max*invMaxEscapeAngle);
    }
    
    // Get the intersecting guessfactor range of a bot centered at a given point
    public boolean processTick(double px, double py, double radius, double lastradius) {
        // Get the sides of the bot
        double bottom = py - 18;
        double top = py + 18;
        double left = px - 18;
        double right = px + 18;
        
        // Get the corners of the bot
        double dxLeft = left - ox;
        double dxLeftSq = dxLeft*dxLeft;
        
        double dxRight = right - ox;
        double dxRightSq = dxRight*dxRight;
        
        double dyBottom = bottom - oy;
        double dyBottomSq = dyBottom*dyBottom;
        
        double dyTop = top - oy;
        double dyTopSq = dyTop*dyTop;
        
        double c1dist = dxLeftSq + dyBottomSq;
        double c2dist = dxLeftSq + dyTopSq;
        double c3dist = dxRightSq + dyBottomSq;
        double c4dist = dxRightSq + dyTopSq;
        
        double minDist = Math.min(Math.min(c1dist, c2dist), Math.min(c3dist, c4dist));
        double rsq = radius * radius;
        if (rsq < minDist) return false;
        
        double maxDist = Math.max(Math.max(c1dist, c2dist), Math.max(c3dist, c4dist));
        double lrsq = lastradius * lastradius;
        if (lrsq > maxDist) return false; 
        
        // Add corners too if they're within the radius range
        // Left Bottom
        if (c1dist >= lrsq && c1dist <= rsq) growRange(dxLeft, dyBottom);
        // Left Top
        if (c2dist >= lrsq && c2dist <= rsq) growRange(dxLeft, dyTop);
        // Right Bottom
        if (c3dist >= lrsq && c3dist <= rsq) growRange(dxRight, dyBottom);
        // Right Top
        if (c4dist >= lrsq && c4dist <= rsq) growRange(dxRight, dyTop);
        
        // Check if there is actually intersections with any sides
        XCircleIntersect(lastradius, left,   bottom, top);
        XCircleIntersect(radius,     left,   bottom, top);
        XCircleIntersect(lastradius, right,  bottom, top);
        XCircleIntersect(radius,     right,  bottom, top);
        
        YCircleIntersect(lastradius, bottom, left,   right);
        YCircleIntersect(radius,     bottom, left,   right);
        YCircleIntersect(lastradius, top,    left,   right);
        YCircleIntersect(radius,     top,    left,   right);
        
        return true;
    }
    
    private boolean growRange(double dx, double dy) {
        double angle = Math.atan2(dx, dy);
        boolean changed = false;
        angle = Utils.normalRelativeAngle(angle - zeroAngle);
        if (angle < min) { min = angle; changed = true; }
        if (angle > max) { max = angle; changed = true; }
        return changed;
    }
    
    private boolean XCircleIntersect(double r, double x, double miny, double maxy) {
        double dx = x - ox;
        double D = r*r-dx*dx;
        boolean changed = false;
        
        // Negative discriminant, no intersection
        if (D < 0)
            return changed;
        
        // Get an intersect
        double d = Math.sqrt(D);
        double y1 = oy + d;
        if (y1 >= miny && y1 <= maxy)
            changed |= growRange(x - ox, y1 - oy);
        
        // Zero discriminant, no need for more
        if (D == 0)
            return changed;
        
        // Get another intersect
        double y2 = oy - d;
        if (y2 >= miny && y2 <= maxy)
            changed |= growRange(x - ox, y2 - oy);
        
        return changed;
    }

    private boolean YCircleIntersect(double r, double y, double minx, double maxx) {
        double dy = y - oy;
        double D = r*r-dy*dy;
        boolean changed = false;
        
        // Negative discriminant, no intersection
        if (D < 0)
            return changed;
        
        // Get an intersect
        double d = Math.sqrt(D);
        double x1 = ox + d;
        if (x1 >= minx && x1 <= maxx)
            changed |= growRange(x1 - ox, y - oy);
        
        // Zero discriminant, no need for more
        if (D == 0)
            return changed;
        
        // Get another intersect
        double x2 = ox - d;
        if (x2 >= minx && x2 <= maxx)
            changed |= growRange(x2 - ox, y - oy);
        
        return changed;
    }
}
