package tzu.gun;
import tzu.util.*;
import tzu.intel.*;

/**
 * Provides estimated aiming using iteration and takes into account
 * recognizable patterns of enemy movement.
 * NOTE: CHANGES TO THIS CLASS MAY ALSO APPLY TO EstimateAim.
 */
public class PatternAim implements Constants {

    /**
     * Use this robot deacceleration rate for the purposes of aiming.
     */
    static double   DEACCELERATE  = -2.0;

    /**
     * Given the current motion of a target and a bullet, uses iteration
     * to calculate a FireSolution that will allow the bullet to hit the
     * target, provided the target continues to follow the same motion.
     *
     * @return FireSolution or null if no reasonable solution found.
     */
    public static FireSolution calculate(Motion target, Motion bullet) {

        Bot bot     = target.getBot();
        double x    = target.getX();
        double y    = target.getY();
        double s    = target.getSpeed();
        double a    = target.getChgSpeed();
        double h    = target.getHeading();
        double t    = target.getChgHeading();
        double bx   = bullet.getX();
        double by   = bullet.getY();
        double bs   = bullet.getSpeed();

        double minX = BattleField.getMinX();
        double minY = BattleField.getMinY();
        double maxX = BattleField.getMaxX();
        double maxY = BattleField.getMaxY();

        long time   = 0;
        long bTime  = (long)(BotMath.calcDistance(x,y, bx, by)/bs);

        double d    = 0.0;
        double los  = 0.0;
        double md2w = bot.getMinDistToWall();
        long tsTime = 0;

        if (s > 0) {

            d = bot.getCurrEstMove();

            /*
             * If this bot is the type that likes to keep at roughly right
             * angles to us while speeding up and slowing down, use average
             * speed to calculate aim.
             */

            if (bot.getAvgSpeed() < 6           &&
                bot.getAvgAngleOfApproach()     -
                bot.getSdAngleOfApproach() > 65 &&
                bot.getSideToSidePct() > 30     &&
                bot.getAvgEstMove() < 110       &&
                bot.getAvgEstMove()             >
                bot.getSdEstMove()) {

                s = bot.getAvgSpeed();
                a = 0.0;
                t = 0.0;

            } else if (bot.getAvgSpeed() < 6    &&
                bot.getAvgAngleOfApproach()     -
                bot.getSdAngleOfApproach() > 55 &&
                bot.getSameDirectionPct()       >
                bot.getSideToSidePct()          &&
                bot.getSameDirectionPct() > 40) {

                s = bot.getAvgSpeed();
                a = 0.0;
                t = 0.0;
                d = 0.0;
            }

            if (bot.getEnergy() < 3 && bot.getAvgLast2EstMove() < 35) {
                s = 0.0;
            }

            a = BotMath.limit(a, DEACCELERATE, ROBOT_ACCELERATION);

            /**
             * If robot changes speed frequently, estimate using
             * average speed and do not factor in acceleration.
             */

            if (bot.getPivotIndicator() > 10) {
                s = bot.getAvgSpeed();
                a = 0.0;
            }

            if (bot.getStartTimeTopSpeed() != 0) {
                tsTime = bot.getLastUpdate() - bot.getStartTimeTopSpeed();
            }
        }

        while (s != 0.0  && time < bTime &&
               x >= minX && y >= minY    &&
               x <= maxX && y <= maxY)   {

            time++;
            if (t != 0) {
                h += BotMath.calcTurnRate(s) * ((t < 0) ? -1 : +1);
            }
            s = BotMath.limit(s + a, 0, MAX_SPEED);
            x = BotMath.calcX(x, h, s);
            y = BotMath.calcY(y, h, s);
            bTime = (long)(BotMath.calcDistance(x,y, bx, by)/bs);
            d += BotMath.calcDistance(target.getX(), target.getY(), x, y);

            if (bot.getRoundsPlayed() > 1       &&
                bot.getAvgSpeed() < 6           &&
                bot.getAvgAngleOfApproach()     -
                bot.getSdAngleOfApproach() > 65 &&
                bot.getSideToSidePct() > 30     &&
                bot.getAvgEstMove() < 110       &&
                bot.getAvgEstMove() > bot.getSdEstMove()) {

                if (d > bot.getAvgEstMove()) {
                    a = DEACCELERATE;
                }
            }

            if (s > 7.0) {

                /**
                 * If this robot is traveling at maximum speed and is
                 * past the point where it normally starts to slow
                 * down, then switch to deacceleration.
                 */
                tsTime++;
                if (tsTime >= bot.getAvgTimeTopSpeed() +
                              bot.getSdTimeTopSpeed()) {
                    a = DEACCELERATE;

                } else if (bot.getAvgSpeed() < 5 &&
                        tsTime >= bot.getAvgTimeTopSpeed()) {
                    a = DEACCELERATE;
                }

                /*
                 * If the robot has already covered a distance beyond what
                 * it normally covers, then switch to deacceleration in
                 * anticipation of a stop.
                 */
                if (d > bot.getAvgEstMove() + bot.getSdEstMove()) {
                    a = DEACCELERATE;

                } else if (
                        bot.getSdEstMove() > (
                        bot.getAvgEstMove() * .5) &&
                        d > bot.getAvgLast2EstMove()) {
                    a = DEACCELERATE;

                } else if ((bot.getAvgSpeed() < 5       ||
                            bot.getAvgEstMove() < 100)  &&
                        d > bot.getAvgEstMove())        {
                    a = DEACCELERATE;
                }

                /*
                 * If the target has never been this close to a wall
                 * before, then exit the loop.
                 */
                if (x < md2w || y < md2w ||
                    x > (BattleField.getWidth() - md2w) ||
                    y > (BattleField.getHeight() - md2w)) {
                        s = 0.0;
                }
            }
        }

        x = BotMath.limit(x, minX, maxX);
        y = BotMath.limit(y, minY, maxY);
        bTime = (long)(BotMath.calcDistance(x, y, bx, by)/bs);

        return new FireSolution(
            bot,
            BotMath.calcHeading(bx, by, x, y),
            FireSolution.AIM_PATTERN,
            x,y, bTime);
    }
}
