package simonton.movements;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.*;
import java.util.*;

import robocode.*;
import simonton.core.*;
import simonton.movements.WeeksOnEndMovement.EnemyWave;
import simonton.utils.*;

public class Surfer extends Movement {

	private static final double HALF_PI = Math.PI / 2;
	private static final int BINS = 47;
	private static final int GF_ZERO = (BINS - 1) / 2;
	private static final int BIN_PADDING = (int) Math.ceil(Math.asin(26.0 / 40)
			/ Math.asin(8.0 / 11) * GF_ZERO);

	private int oDir = 1;
	private int timeDecay;
	private double N;
	private double S;
	private double E;
	private double W;
	private double maxAttack;
	private double maxRetreat;
	private double minDistance;
	private double walkingStick;
	private double desiredDistance;
	private double weightRollingDepth;
	private double[] weights;
	private BotConduit[] waveMachineConduits;
	private WaveMachine[] waveMachines;
	private Point2D.Double myP;
	private Point2D.Double enemyP;
	private HitByBulletEvent hitByBulletEvent;
	private BulletHitBulletEvent bulletHitBulletEvent;
	private HashMap<Wave, double[]> waveDangers;
	private HashMap<Wave, double[][]> waveDangerBreakdown;

	// private HashMap<Wave, Set<Integer>> washedStrikes;

	public Surfer(int timeDecay, double wallMargin, double maxAttack,
			double maxRetreat, double desiredDistance,
			double walkingStickLength, double weightRollingDepth,
			double minDistance, WaveMachine... waveMachines) {

		this.timeDecay = timeDecay;
		this.maxAttack = maxAttack;
		this.maxRetreat = maxRetreat;
		this.minDistance = minDistance;
		this.waveMachines = waveMachines;
		this.walkingStick = walkingStickLength;
		this.desiredDistance = desiredDistance;
		this.weightRollingDepth = weightRollingDepth;
		weights = new double[waveMachines.length];
		waveMachineConduits = new BotConduit[waveMachines.length];
		for (int i = waveMachines.length; --i >= 0;) {
			waveMachineConduits[i] = new BotConduit(this, waveMachines[i]);
			weights[i] = .5;
		}
		N = Util.worldHeight - wallMargin;
		E = Util.worldWidth - wallMargin;
		S = wallMargin;
		W = wallMargin;
	}

	@Override
	public void run() {
		super.run();
		waveDangers = new HashMap<Wave, double[]>();
		// washedStrikes = new HashMap<Wave, Set<Integer>>();
		waveDangerBreakdown = new HashMap<Wave, double[][]>();
		for (int i = waveMachines.length; --i >= 0;) {
			waveMachineConduits[i].syncBot();
			waveMachines[i].run();
		}
	}

	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		positions.clear();
		intensities.clear();
		myP = new Point2D.Double(getX(), getY());
		enemyP = Util.project(this, e);

		handleAddedWave();
		removeWave(Util.deletedEnemyWave);
		// updateWashedStrikes();
		recalculateDangers();
		drive();

		// do this last, so that when we see a wave, we generate *last* turn's
		// data to calulate dangers (since he actually fired last turn).
		for (int machine = waveMachines.length; --machine >= 0;) {
			waveMachineConduits[machine].syncBot();
			if (bulletHitBulletEvent != null) {
				waveMachines[machine].onBulletHitBullet(bulletHitBulletEvent);
			}
			if (hitByBulletEvent != null) {
				waveMachines[machine].onHitByBullet(hitByBulletEvent);
			}
			waveMachines[machine].onScannedRobot(e);
		}
		bulletHitBulletEvent = null;
		hitByBulletEvent = null;
	}

	// private void updateWashedStrikes() {
	// for (Wave wave : Util.enemyWaves) {
	// if (wave.washCode(myP, Util.time) == 0) {
	// double center = Util.bearing(wave.origin, myP);
	// double delta = Util.asin(26 / wave.origin.distance(myP));
	// int min = wave.getFactorIndex(center - delta, GF_ZERO,
	// BIN_PADDING);
	// int max = wave.getFactorIndex(center + delta, GF_ZERO,
	// BIN_PADDING);
	// Set<Integer> strikes = washedStrikes.get(wave);
	// for (int i = min; i <= max; ++i) {
	// if (strikes.contains(i)) {
	// continue;
	// }
	// if (wave.isFactorHit(i, GF_ZERO, BIN_PADDING, Util.time,
	// myP)) {
	// strikes.add(i);
	// }
	// }
	// }
	// }
	// }

	@Override
	public void onHitByBullet(HitByBulletEvent e) {
		hitByBulletEvent = e;
		// for (Wave w : waveDangers.keySet()) {
		// if (w.equals(e.getBullet())) {
		// removeWave(w);
		// }
		// }
		updateWeights(e.getBullet());
	}

	@Override
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		bulletHitBulletEvent = e;
		updateWeights(e.getHitBullet());
	}

	private void handleAddedWave() {
		Wave toAdd = Util.addedEnemyWave;
		if (toAdd == null) {
			return;
		}
		double[][] breakdown = new double[waveMachines.length][];
		for (int machine = waveMachines.length; --machine >= 0;) {
			breakdown[machine] = waveMachines[machine].getDangers(toAdd, BINS,
					BIN_PADDING);
		}
		waveDangerBreakdown.put(toAdd, breakdown);
		waveDangers.put(toAdd, new double[BINS + 2 * BIN_PADDING]);
		// washedStrikes.put(toAdd, new HashSet<Integer>());
	}

	private void updateWeights(Bullet hitBullet) {
		Point2D.Double hitPoint = new Point2D.Double(hitBullet.getX(),
				hitBullet.getY());
		for (Wave wave : Util.enemyWaves) {
			if (wave.equals(hitBullet)) {
				double[][] machineDangerBreakdown = waveDangerBreakdown
						.get(wave);
				int factorIndex = wave.getFactorIndex(hitPoint, GF_ZERO,
						BIN_PADDING);
				for (int i = weights.length; --i >= 0;) {
					double[] dangers = machineDangerBreakdown[i];
					double danger = dangers[factorIndex - 1];
					if (dangers[factorIndex] > danger) {
						danger = dangers[factorIndex];
					}
					if (dangers[factorIndex + 1] > danger) {
						danger = dangers[factorIndex + 1];
					}
					weights[i] = Util.roll(weights[i], danger,
							weightRollingDepth);
				}
			}
		}
	}

	private void recalculateDangers() {

		int paddedPos;
		double weight;
		double weightedMachineDanger;
		double[] groupDangers;
		double[] machineDangers;
		double[][] machineDangerBreakdown;
		for (Wave wave : Util.enemyWaves) {
			groupDangers = waveDangers.get(wave);
			machineDangerBreakdown = waveDangerBreakdown.get(wave);
			for (int i = waveMachines.length; --i >= 0;) {
				weight = weights[i];
				machineDangers = machineDangerBreakdown[i];
				for (int bin = 0; bin < BINS; ++bin) {
					paddedPos = bin + BIN_PADDING;
					weightedMachineDanger = machineDangers[paddedPos] * weight;
					if (weightedMachineDanger > groupDangers[paddedPos]) {
						groupDangers[paddedPos] = weightedMachineDanger;
					}
				}
			}
		}
	}

	private void removeWave(Wave toRemove) {
		if (toRemove == null) {
			return;
		}
		waveDangers.remove(toRemove);
		waveDangerBreakdown.remove(toRemove);
		// washedStrikes.remove(toRemove);
	}

	private void drive() {

		Collections.sort(Util.enemyWaves, new Wave.TimeToImpactComparator(myP));
		List<Wave> toWash = Util.enemyWaves.subList(0, Math.min(5,
				Util.enemyWaves.size()));
		if (getGoodness(oDir, toWash) < getGoodness(-oDir, toWash)) {
			oDir = -oDir;
		}
		if (mustFlip(oDir, Util.myP)) {
			oDir = -oDir;
		}

		double dh = Util.normalize(getAngle(oDir) - getHeadingRadians());
		if (dh > HALF_PI) {
			setTurnRightRadians(dh - Math.PI);
			setBack(100);
		} else if (dh < -HALF_PI) {
			setTurnRightRadians(dh + Math.PI);
			setBack(100);
		} else {
			setTurnRightRadians(dh);
			setAhead(100);
		}
	}

	private boolean mustFlip(int oDir, Point2D.Double location) {
		double tangent = Util.bearing(enemyP, location) + oDir * HALF_PI;
		double goAngle = getAngle(oDir, location);
		double attack = oDir * Util.normalize(goAngle - tangent);
		return attack > maxAttack;
		// || (attack > 0 && enemyP.distance(location) < minDistance);
	}

	private double getGoodness(int oDir, List<Wave> toWash) {

		return getGoodness(oDir, (int) Util.time, myP, getHeadingRadians(),
				getVelocity(), 1, 5, toWash);
	}

	private double getGoodness(int oDir, int time, Point2D.Double loc,
			double h, double v, double goodness, int levelsToGo,
			List<Wave> toWash) {
		toWash = new LinkedList<Wave>(Util.enemyWaves);
		// Collection<Set<Integer>> strikeIndicies = new
		// LinkedList<Set<Integer>>();
		double maxTurn;
		double dh;
		int code;
//		int freezeODir = oDir;
//		int freezeTime = time;
//		Point2D.Double freezeLoc = loc;
//		double freezeH = h;
//		double freezeV = v;
//		double freezeGoodness = goodness;
//		double freezeDanger = Double.MAX_VALUE;

		// // set up a collection of strike indicies for each wave
		// for (int i = toWash.size(); --i >= 0; ) {
		// strikeIndicies.add(new HashSet<Integer>());
		// }

		do {
			++time;
			// Iterator<Set<Integer>> strikeIt = strikeIndicies.iterator();
			for (Iterator<Wave> washIt = toWash.iterator(); washIt.hasNext();) {
				Wave wave = washIt.next();
				// Set<Integer> strikes = strikeIt.next();
				code = wave.washCode(loc, time);
				if (code == 0) {

					double waveDanger = waveDangers.get(wave)[wave
							.getFactorIndex(loc, GF_ZERO, BIN_PADDING)];
					goodness *= 1 - waveDanger
							/ ((time - Util.time) / wave.timeTillNext + 1);
					// int factorIndex = wave.getFactorIndex(loc, GF_ZERO,
					// BIN_PADDING);
					// if (!washedStrikes.get(wave).contains(factorIndex)) {
					// strikes.add(factorIndex);
					// }
					// when a wave is washing, record all the guess factors at
					// which it could strike
					// double center = Util.bearing(wave.origin, loc);
					// double delta = Util.asin(26 / wave.origin.distance(loc));
					// int min = wave.getFactorIndex(center - delta, GF_ZERO,
					// BIN_PADDING);
					// int max = wave.getFactorIndex(center + delta, GF_ZERO,
					// BIN_PADDING);
					// Set<Integer> prewash = washedStrikes.get(wave);
					// for (int i = min; i <= max; ++i) {
					// if (strikes.contains(i) || prewash.contains(i)) {
					// continue;
					// }
					// if (wave.isFactorHit(i, GF_ZERO, BIN_PADDING, Util.time
					// + elapsed, loc)) {
					// strikes.add(i);
					// }
					// }
				} else if (code < 0) {

					// once a wave passes, deduct all the factors at which it
					// could have struck from our overall goodness
					washIt.remove();
					// strikeIt.remove();
					// double[] dangers = waveDangers.get(wave);
					// for (Integer factorIndex : strikes) {
					// goodness *= 1 - dangers[factorIndex]
					// / (elapsed / wave.timeTillNext + 1);
					// }
				}
			}
			intensities.add(new Float(1 - goodness));
			positions.add(loc);
			if (toWash.isEmpty()) {
				return goodness;
			}

			// project our position next tick
			if (mustFlip(oDir, loc)) {
				oDir = -oDir;
			}
			maxTurn = Util.getTurnRateRadians(Math.abs(v));
			dh = Util.normalize(getAngle(oDir, loc) - h);
			if (dh > HALF_PI) {
				h += Util.limit(dh - Math.PI, -maxTurn, maxTurn);
				if (v > 0) {
					v -= 2;
				} else if (v <= -7) {
					v = -8;
				} else {
					v -= 1;
				}
			} else if (dh < -HALF_PI) {
				h += Util.limit(dh + Math.PI, -maxTurn, maxTurn);
				if (v > 0) {
					v -= 2;
				} else if (v <= -7) {
					v = -8;
				} else {
					v -= 1;
				}
			} else {
				h += Util.limit(dh, -maxTurn, maxTurn);
				if (v < 0) {
					v += 2;
				} else if (v >= 7) {
					v = 8;
				} else {
					v += 1;
				}
			}
			loc = Util.project(loc, v, h);
		} while (true);
	}

	private double getAngle(int oDir) {
		return getAngle(oDir, myP);
	}

	private double getAngle(int oDir, Point2D.Double location) {
		double angle = Util.bearing(enemyP, location) + oDir * HALF_PI;
		double farthestWall;
		if (location.x < enemyP.x) {
			farthestWall = enemyP.x;
		} else {
			farthestWall = Util.worldWidth - enemyP.x;
		}
		if (location.y < enemyP.y) {
			if (enemyP.y > farthestWall) {
				farthestWall = enemyP.y;
			}
		} else {
			if (Util.worldHeight - enemyP.y > farthestWall) {
				farthestWall = Util.worldHeight - enemyP.y;
			}
		}
		double toGo = farthestWall - 18 - enemyP.distance(location);
		if (toGo > 0) {
			angle -= oDir * Math.min(maxRetreat, toGo / 16);
		} else {
			angle -= oDir * Math.max(-maxRetreat, toGo / 16);
		}

		double distance;
		if ((distance = N - location.y) < walkingStick) {
			if (distance < Util.cos(angle) * walkingStick) {
				angle = Util.asin(-oDir * distance / walkingStick)
						+ (oDir * HALF_PI);
			}
		} else if ((distance = location.y - S) < walkingStick) {
			if (distance < -Util.cos(angle) * walkingStick) {
				angle = Util.asin(-oDir * distance / walkingStick)
						+ (oDir * HALF_PI) + Math.PI;
			}
		}
		if ((distance = E - location.x) < walkingStick) {
			if (distance < Util.sin(angle) * walkingStick) {
				angle = Util.acos(oDir * distance / walkingStick)
						+ (oDir * HALF_PI);
			}
		} else if ((distance = location.x - W) < walkingStick) {
			if (distance < -Util.sin(angle) * walkingStick) {
				angle = Util.acos(oDir * distance / walkingStick)
						+ (oDir * HALF_PI) + Math.PI;
			}
		}
		return angle;
	}

	//
	// @Override
	// public void onPaint(Graphics2D g) {
	// g.setColor(Color.RED);
	// Point2D.Double myP = new Point2D.Double(getX(), getY());
	// Util.paintLine(g, myP, Util.project(myP, walkingStick, getAngle(1)));
	// Util.paintLine(g, myP, Util.project(myP, walkingStick, getAngle(-1)));
	// }
	List positions = new ArrayList();
	List intensities = new ArrayList();

	public void onPaint(Graphics2D g) {
		for (int i = positions.size(); --i >= 0;) {
			float intensity = ((Number) intensities.get(i)).floatValue();
			intensity *= intensity;
			g.setColor(new Color(intensity, 0, 1 - intensity));
			Point2D.Double p = (Point2D.Double) positions.get(i);
			g.fillOval((int) p.x - 1, (int) p.y - 1, 3, 3);
		}

		// g.drawLine((int) myLocation.x, (int) myLocation.y,
		// (int) hisLocation.x, (int) hisLocation.y);

		if (enemyP != null) {
			g.setColor(Color.blue);
			double r = desiredDistance;
			g.drawOval((int) (enemyP.x - r), (int) (enemyP.y - r),
					(int) (2 * r), (int) (2 * r));
		}

		for (Wave ew : Util.enemyWaves) {
			double dist = (getTime() - ew.fireTime) * ew.speed;
			for (int j = BINS; --j >= 0;) {
				Point2D.Double p = Util.project(ew.origin, dist, (j - GF_ZERO)
						* ew.maxEscape / GF_ZERO + ew.angle);
				float intensity = (float) waveDangers.get(ew)[BIN_PADDING + j];
				intensity = Math.min(1f, intensity * 5);
				g.setColor(new Color(intensity, 1 - intensity, 0));
				g.fillOval((int) p.x - 1, (int) p.y - 1, 3, 3);
			}
		}

		// for (WaveMachine wm : waveMachines) {
		// wm.onPaint(g);
		// }
	}
}
