/*
 * Preprocessor - Utility class for neural nets of Robocode bots.
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.neural.util;

public class Preprocessor {

    public static void intToBinary(int toConvert,
                                       double[] out, int startIdx, int length) {
        for (int i = startIdx; i < startIdx + length; i++) {
            out[i] = (double)(toConvert % 2);
            toConvert >>= 1;
        }
    }

    public static int binaryToInt(double[] in, int startIdx, int length) {
        int converted = 0;
        for (int i = startIdx + length -1 ; i >= startIdx; i--) {
            converted <<= 1;
            if (in[i] > 0.5) {
                converted++;
            }
        }
        return converted;
    }

    public static void intToGrayCoded(int toConvert,
                                      double[] out, int startIdx, int length) {
        int toConvertGray = GrayCode.encode(toConvert);
        intToBinary(toConvertGray, out, startIdx, length);
    }

    public static int grayCodedToInt(double[] in, int startIdx, int length) {
        return GrayCode.decode(binaryToInt(in, startIdx, length));
    }

    public static void doubleToBinary(double toConvert, int fractionBits,
                                      double[] out, int startIdx, int length) {
        for (int i = 0; i < fractionBits; i++) {
            toConvert *= 2.0;
        }
        intToBinary((int)Math.round(toConvert), out, startIdx, length);
    }

    public static double binaryToDouble(double[] in, int fractionBits,
                                        int startIdx, int length) {
        int convertedInt = binaryToInt(in, startIdx, length);
        double converted = (double)convertedInt;
        for (int i = 0; i < fractionBits; i++) {
            converted /= 2.0;
        }
        return converted;
    }

    public static void doubleToGrayCoded(double toConvert, int fractionBits,
            double[] out, int startIdx, int length) {
        for (int i = 0; i < fractionBits; i++) {
            toConvert *= 2.0;
        }
        intToGrayCoded((int)Math.round(toConvert), out, startIdx, length);
    }

    public static double grayCodedToDouble(double[] in, int fractionBits,
                                           int startIdx, int length) {
        double converted = (double)grayCodedToInt(in, startIdx, length);
        for (int i = 0; i < fractionBits; i++) {
            converted /= 2.0;
        }
        return converted;
    }

    public static double normalize(double value, double min, double max,
                                   double buffer) {
        return (value - min) / (max - min) * (1.0 - 2.0 * buffer) + buffer;
    }

    public static double denormalize(double value, double min, double max,
                                     double buffer) {
        return (value - buffer) / (1-0 - 2.0 * buffer) * (max - min) + min;
    }

    public static double normalize(double value, double min, double max) {
        return normalize(value, min, max, 0.1);
    }

    public static double denormalize(double value, double min, double max) {
        return denormalize(value, min, max, 0.1);
    }
}
