/*
 * Written by Kinsen Choy
 */

package kinsen.melee;
import java.awt.Graphics2D;
import java.awt.Point;
import java.util.ArrayList;

/**
 * Utils - A class containing often used functions.
 */
public class Utils
{

	// Debug: Text to print out in console
	private static ArrayList textConsole = new ArrayList();
	// Debug: Whether to print text on the next line
	private static boolean nextLine = true;

	// Battlefield dimensions
	private static double battleFieldWidth;
	private static double battleFieldHeight;

	// Distance to stay away from the wall when wall smoothing
	private static final double wallMargin = 80;

	// Sets arena size
	public static void setBounds(double width, double height)
	{
		battleFieldWidth = width;
		battleFieldHeight = height;
	}

	// Determines if the given position is outside the arena
	public static boolean outOfBounds(double x, double y)
	{
		if (x < 0 || x > battleFieldWidth || y < 0 || y > battleFieldHeight)
			return true;
		else
			return false;
	}

	// Determines if the given position is outside the arena
	public static boolean robotOutOfBounds(double x, double y)
	{
		if (x < wallMargin || x > battleFieldWidth - wallMargin || y < wallMargin || y > battleFieldHeight - wallMargin)
			return true;
		else
			return false;
	}

	/*
	 * distance: Returns the distance from one location to another
	 */
	public static double distance(double X1, double Y1, double X2, double Y2)
	{
		return Math.sqrt(Math.pow(X1 - X2, 2) + Math.pow(Y1 - Y2, 2));
	}

	/*
	 * angleFromPoint: Returns the angle from a point to another
	 */
	public static double angleFromPoint(double startX, double startY, double targetX, double targetY)
	{
		return Math.toDegrees(Math.atan2(targetX - startX, targetY - startY));
	}

	/*
	 * normalRelativeAngle: Returns an angle in the set (-180, 180]
	 */
	public static double normalRelativeAngle(double angle)
	{
		if (angle > -180 && angle <= 180)
			return angle;
		double fixedAngle = angle;
		while (fixedAngle <= -180)
			fixedAngle += 360;
		while (fixedAngle > 180)
			fixedAngle -= 360;
		return fixedAngle;
	}

	/*
	 * normalAbsoluteAngle: Returns an angle in the set [0, 360)
	 */
	public static double normalAbsoluteAngle(double angle)
	{
		if (angle >= 0 && angle < 360)
			return angle;
		double fixedAngle = angle;
		while (fixedAngle < 0)
			fixedAngle += 360;
		while (fixedAngle >= 360)
			fixedAngle -= 360;
		return fixedAngle;
	}

	/*
	 * Debug: Converts gun type index to gun name
	 */
	public static String convertGunType(byte gunIndex)
	{
		String gunType;
		switch (gunIndex)
		{
			case 0:
				gunType =  "Head-On";
				break;
			case 1:
				gunType = "Circular";
				break;
			case 2:
				gunType = "Linear";
				break;
			case 3:
				gunType = "Half-Linear";
				break;
			case 4:
				gunType = "Stop and Go";
				break;
			case 5:
				gunType = "Oscillator";
				break;
			case 6:
				gunType = "Averaged Circular";
				break;
			case 7:
				gunType = "Guess Factor";
				break;
			case 8:
				gunType = "Pattern Matching";
				break;
			default:
				gunType = "Invalid bullet type";
				break;
		}
		return gunType;
	}

	/*
	 * print: Debug: Prints text to the custom on-screen console without a line break at the end
	 */
	public static void print(String text)
	{
		System.out.print(text);
		consoleAddText(text);
		nextLine = false;
	}

	/*
	 * println: Debug: Prints text to the custom on-screen console with a line break at the end
	 */
	public static void println(String text)
	{
		System.out.println(text);
		consoleAddText(text);
		nextLine = true;
	}

	/*
	 * consoleAddText: Debug: The function called by print and println to add text to the console
	 */
	private static void consoleAddText(String text)
	{
		if (nextLine)
		{
			textConsole.add(text);
			if (textConsole.size() == 20)
				textConsole.remove(0);
		}
		else
			textConsole.set(textConsole.size() - 1, (String) textConsole.get(textConsole.size() - 1) + text);
	}

	//  Debug: Overloaded functions for print and println
	public static void print(boolean bool)
	{
		print(new Boolean(bool).toString());
	}

	public static void print(int number)
	{
		print(new Integer(number).toString());
	}

	public static void print(double number)
	{
		print(new Double(number).toString());
	}

	public static void println(boolean bool)
	{
		println(new Boolean(bool).toString());
	}

	public static void println(int number)
	{
		println(new Integer(number).toString());
	}

	public static void println(double number)
	{
		println(new Double(number).toString());
	}

	/*
	 * drawText: Debug The function called by the robot to draw the text onto the screen
	 */
	public static void drawText(Graphics2D g)
	{
		int maxHeight = textConsole.size() * 12;
		for (int i = 0; i < textConsole.size(); i++)
		{
			g.drawString((String) textConsole.get(i), 1, maxHeight - i * 12 + 1);
		}
	}
}