package cs.s2.stat;

import java.util.ArrayList;
import java.util.List;

/**
 * This class handles multiple stat buffers, as if they were one.
 */
public final class StatBufferSet {
	private static final class StatItem {
		StatBuffer buffer;
		double weight;
	}

	private List<StatItem> autobin = new ArrayList<StatItem>();

	/**
	 * Add a stat buffer to this buffer set.
	 */
	public void add(StatBuffer bin, double weight) {
		StatItem si = new StatItem();
		si.buffer = bin;
		si.weight = weight;
		autobin.add(si);
	}

	/**
	 * Get a stat buffer from this buffer set.
	 */
	public StatBuffer get(int index) {
		return autobin.get(index).buffer;
	}

	/**
	 * Remove a statbuffer from this buffer set
	 */
	public void remove(int index) {
		autobin.remove(index);
	}

	/**
	 * Returns the number of stat buffers.
	 */
	public int size() {
		return autobin.size();
	}

	/**
	 * Gets the single best GF from each StatBuffer.
	 */
	public double[] getBestGFList(StatData data) {
		double[] output = new double[autobin.size()];
		int i = 0;
		for (StatItem s : autobin) {
			output[i++] = s.buffer.getBestGF(data);
		}
		return output;
	}

	/**
	 * Gets a list of all the bins for the raw data.
	 */
	private double[][] getBinList(StatData data) {
		double[][] output = new double[autobin.size()][];
		int i = 0;
		for (StatItem s : autobin) {
			output[i++] = s.buffer.getStatBin(data);
		}
		return output;
	}

	/**
	 * Gets the list of weights.
	 */
	private double[] getWeightList() {
		double[] output = new double[autobin.size()];
		int i = 0;
		for (StatItem s : autobin) {
			output[i++] = s.weight;
		}
		return output;
	}

	/**
	 * Gets the combined score data for the given stat data.
	 */
	public double[] getCombinedScoreData(StatData sd) {
		double[][] data = getBinList(sd);
		double[] weights = getWeightList();
		return getCombinedScoreData(data, weights);
	}

	/**
	 * This combines the data over all the bins in such a way that it can be
	 * read easily as if it were a single stat bin.
	 */
	private double[] getCombinedScoreData(double[][] data, double[] weights) {
		if (weights.length != data.length)
			throw new IllegalArgumentException(
					"Data length and weights length must match.");

		/* Determine the longest and shortest bin set */
		int shortest = Integer.MAX_VALUE;
		int longest = Integer.MIN_VALUE;
		for (double[] d : data) {
			if (d.length > longest)
				longest = d.length;
			if (d.length < shortest)
				shortest = d.length;
		}

		/* Get each of their scores, and record the largest of each. */
		double[][] score = new double[data.length][];
		double[] largest = new double[data.length];
		for (int i = 0; i < data.length; ++i) {
			score[i] = StatBuffer.getBinScores(data[i], StatBuffer.scanWidth);
			for (int j = 0; j < score[i].length; ++j)
				if (score[i][j] > largest[i])
					largest[i] = score[i][j];
		}

		/* Equalize each of the stat bins */
		for (int i = 0; i < data.length; ++i)
			for (int j = 0; j < score[i].length; ++j)
				score[i][j] /= largest[i];

		/* If they are equal, take the easy route */
		if (shortest == longest) {
			largest = new double[shortest];
			for (int i = 0; i < data.length; ++i)
				for (int j = 0; j < shortest; ++j)
					largest[j] += data[i][j] * weights[i];

			return largest;
		}

		/* Otherwise we have to do things the 'hard' way! */
		/* We use the longest and combine the sums from the others, based on gf */
		largest = new double[longest];
		for (int i = 0; i < data.length; ++i) {
			double[] tmpSum = new double[longest];
			/* Get the gf for the new length and multiply it by the old weight */
			for (int j = 0; j < score[i].length; ++j) {
				double index = StatBuffer.GFToIndex(StatBuffer.indexToGF(j,
						score[i].length), longest);
				StatBuffer.updateBin(tmpSum, index, score[i][j]);
			}
			/* Add to output buffer */
			for (int j = 0; j < longest; ++j) {
				largest[j] += tmpSum[j] * weights[i];
			}
		}

		return largest;
	}

	/**
	 * Gets the best GF based on weights.
	 */
	public double getBestGF(StatData data) {
		double[] score = getCombinedScoreData(data);
		int bestIndex = (score.length - 1) / 2;
		for (int i = 0; i < score.length; ++i)
			if (score[i] > score[bestIndex])
				bestIndex = i;
		return StatBuffer.indexToGF(bestIndex, score.length);
	}

	/**
	 * Updates this stat buffer handler with the given raw data
	 */
	public void update(StatData data, double weight) {
		for (StatItem s : autobin) {
			s.buffer.update(data, weight);
		}
	}
}