package gh;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;
import java.util.ArrayList;
import java.util.Vector;

/**
 * GrypRepetyf - a robot by Gert Heijenk
 *
 * Evolving into solid patternmatching gun and an improved GrubbmGrb movement
 *
 * Revision information:
 * v0.1  20050724 Full GrubbmGrb 1.1.0 with only Orbital movement and CT-gun enabled
 * v0.2  20050727 Removed unnecessary stuff, introducing CoolMovement,
 *				  improved HOT-detection, messed up wallavoidance/bouncing
 * v0.3  20050728 repaired wallavoidance/bouncing, improved close range performance
 * v0.4  20050820 use StopNGo oscillator now, no more CoolMovement,
 *				  solved bullet-detection bug, improved HOT-detection
 * v0.5  20051003 use slow movement with 'turboboost' on enemyfire
 *                This is the opposite of my initial idea of 'handbrake' to lure HOT and LT/CT with ONE movement
 * v0.6  20051013 use (too) strict PM-gun loosely based upon pez.Frankie
 * v0.7  20051018 improved roundbreak behaviour, use precise playback
 * v0.8  20051020 replace movement by second (main) movement of GrubbmGrb to recognize effect of PM-gun
 * v0.9  20051022 use 'loose' match instead of 'strict' match
 * v0.9fix 20051023 removed a ';' to enable PM-gun
 * v0.10 20051111 also match on 'mirror'-pattern (speed and headingchange can be +/+, -/+, +/- or -/-)
 * v0.11 20060222 stop recording if opponent is disabled, use backup-gun in that case
 *                distance dependent dive-in protection
 * v0.12 20060304 keep a preferred distance
 * v0.13 20060207 tune endgame against 'energysavers'
 */
public class GrypRepetyf extends AdvancedRobot
{
	// movement related stuff
	private static final int BORDERSIZE			= 17;	// the thickness of the border minus 1
	private static final double FULLSTOP		= 0.0;	// the absolute minimum speed
	private static final double MINSPEED		= 2.0;	// the minimum speed (sharp turn)
	private static final double MEDSPEED		= 4.0;	// the medium speed (fast turn)
	private static final double MAXSPEED		= 8.0;	// the maximum speed

	private static final double BLINDMANSTICK	= 130;	// the distance when to check wallavoidance
	private static final int DEFDISTANCE		= 500;	// the default distance to keep from enemy
	private static final double WALLTOUCH		= 28;	// the distance to keep from walls

	private static final double PI 				= Math.PI;	// for short reference
	private static final double HALFPI			= PI / 2;	// for short reference
	private static final double SIXTHPI			= PI / 6;	// 30 degrees
	private static final double DEFDIVEANGLE	= 3 * PI / 8;	// 67.5 degrees
	
	// enemy related information
	private EnemyInfo enemy;

	// movement related information
	private double	jiggleOffset;	// the static (random) offset for each move
//	private double  distAngle;		// the angle to turn to get to preferred distance
	private double	prefSpeed;		// the preferred speed
//	private long	changeSpeedTime = 0;	// the time when I have to change the speed
	private long	airClearTime;	// time that the air is clear of enemybullets
	private boolean endGame;		// try to hold initiative when enemy stops firing

	// scanning and locking variables
	private long	scanTime;		// the time the last robotscanned_event was received
	private long	prevTime;		// the time the previous robotscanned_event was received

	// some info about myself
	private Point2D.Double	myPos = new Point2D.Double();		// my current position
	private Point2D.Double	myNextPos = new Point2D.Double();	// my next position
//	private Point2D.Double	myPrevPos = new Point2D.Double();	// my previous position
	private double			myHeading;			// my current heading
	private double			myCheckDistance;	// my distance to check for wall-avoidance
	private double			firePower;			// the optimal firepower
	private double			currDirection;		// 1 if forward, -1 if backward
	private double			lastDirection;		// the direction I had when I hit another robot
	private boolean			robothit;			// I hit another robot

	// statistics counters
	static int	bullhit;	// the total number of bullets that hit
	static int	bullmis;	// the total number of bullets that missed
	static int	wallhit;	// the total number of times I hit the wall
	static int	skipturn;	// the total number of skipped turns
	static int	bullPM;		// the total number of bullets fired with PM
	static int	bullBCK;	// the total number of bullets fired with backup gun (sortof linear)

	// some general info
	private double	fieldWidth;		// the width of the battlefield
	private double	fieldHeight;	// the heigth of the battlefield
    private RoundRectangle2D.Double playField;	// the playable battlefield
	private Rectangle2D.Double fireField;		// the shootable battlefield

	// some flags for the several challanges
	private boolean TC_flag = false;	// Targetting/PM challenge (no movement, always fire 3.0, no energy-management)
	private boolean MC_flag = false;	// Movement challenge	(do not fire)
	private boolean RR_flag = true;		// RR Release if set, development otherwise

//========================
    // PM stuff adapted to my own needs
	static final int PM_LENGTH = 12000;
	static final int PM_MAXFILL = (int)(PM_LENGTH * 0.8);
    static final int BREAK_KEY = 128;

//	static final int SEARCH_LOOSE_MATCH;
//	static final int SEARCH_TIGHT_MATCH;
	static Frame movie[];
    static boolean movieIsFull = false;
    static int movieSize = 0;
//	Candidate looseRow;
//	Candidate tightRow;
	int		lastValidMovieSize = 0;
    double	bulletVelocity;
	int		gunPMStage;
	double	guessedEnemyBearing;
	boolean	PMActive;
	boolean	gunOverride;			// if gunOverride, do not record Frames anymore, and use backup-gun
	int		speedFactor;			// -1 if playbackspeed must be mirrored, 1 if plain playback
	int		angleFactor;			// -1 if playbackangle must be mirrored, 1 if plain playback
//========================

	/**
	 * run: GrypRepetyf's default behavior
	 */
	public void run() {
		int i;

		// Give the robot an appealing look
		setColors( Color.red.brighter(), Color.orange, Color.red.darker());

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        playField = new RoundRectangle2D.Double( WALLTOUCH, WALLTOUCH,
	    			fieldWidth - 2 * WALLTOUCH, fieldHeight - 2 * WALLTOUCH, 50, 50);
        fireField = new Rectangle2D.Double( BORDERSIZE, BORDERSIZE,
					fieldWidth - 2 * BORDERSIZE, fieldHeight - 2 * BORDERSIZE);

		// Fill in some default info about the Enemy
		enemy = new EnemyInfo();

		// do not start moving until enemy fired its gun
		setMaxVelocity( FULLSTOP);

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);

		// Make initial direction less predictable
		currDirection = (Math.random() < 0.5 ? 1 : -1);

        // create frames for PM-gun
		if (movie == null) {
			movie = new Frame[PM_LENGTH + 1];
			for (i = 0; i <= PM_LENGTH; i++) {
				movie[i] = new Frame();
			}
		}
		// create candidates for PM-gun
//		looseRow = new Candidate();
//		tightRow = new Candidate();

		// PM gun mark round-break
		if (movieSize > PM_MAXFILL) movieSize = PM_MAXFILL;
		lastValidMovieSize = movieSize;
		record( 0, BREAK_KEY+(getRoundNum()%64), 1);

		// mainloop
		while( true) {
			turnRadarRight( Double.POSITIVE_INFINITY);	// start scanning and never stop
		}
	}

	/**
	 * doMovement: Handle the oscillator movement of GrubbmGrb
	 *
	 */
	void doMovement() {
		double  myDistance;					// my current distance to travel 
		double	myTurnAngle;				// the preferred turnangle
		double  newposx, newposy;			// my future (end)position
		double	perpAngle;                  // angle to reach perpendicularity
		double  diveAngle;					// angle to compensate for distance
		double	distDiff;                   // the difference between preferred distance and real distance
		double  maxDiveAngle;				// the maximum dive-in angle

		// calculate angle for perpendicularity, I might need it
		myTurnAngle = Utils.normalRelativeAngle(enemy.getBearing() + HALFPI);
		perpAngle = Math.atan(Math.tan(myTurnAngle));
//		out.println("Speed: "+getVelocity());

		// Check if nearly at end of movement
		if (Math.abs(getDistanceRemaining()) < 1) { 
			myDistance = (Math.sin( scanTime/13 + getRoundNum()) * Math.cos( scanTime/29))*270 + (Math.random()*100 - 50);
			currDirection = (myDistance > 0 ? 1 : -1);
			// move away after robothit (to avoid rambots)
			if (true == robothit) {
				if (currDirection == lastDirection) {
					currDirection = -currDirection;
					myDistance = -myDistance;
				}
				robothit = false;
			}
			lastDirection = currDirection;
			setAhead( myDistance);

            diveAngle = 0;
			// When new travel-distance is taken, also reconsider the enemy-distance and angle
//			if (endGame == true)
				// in endGame, get closer to enemy
//				distDiff = enemy.getDistance();
//			else
				distDiff = enemy.getDistance() - DEFDISTANCE;
			if (Math.abs( distDiff) > 100) {
				diveAngle = (Math.abs(myTurnAngle) <= HALFPI ? currDirection : -currDirection);
				if (distDiff < 0) {
					diveAngle *= (Math.sqrt(Math.abs(distDiff)) / 25);
//					out.println("Go to the outside");
				}
				else {
					diveAngle *= -(Math.sqrt(Math.abs(distDiff)) / 50);
//					out.println("Go to the inside");
				}
			}
//			if (endGame == true)
				// when in endGame, close in on enemy
//				jiggleOffset = 0;
//			else
				jiggleOffset = Math.toRadians(Math.random()*46-23);
			myTurnAngle = Math.atan( Math.tan( perpAngle + diveAngle + jiggleOffset));
//			myTurnAngle = Math.atan( Math.tan( perpAngle + jiggleOffset));
			setTurnRightRadians( myTurnAngle);
		}

		// when in endGame, close in on enemy but take care!
		if (endGame == true) {
			diveAngle = -0.5 * (Math.abs( myTurnAngle) <= HALFPI ? currDirection : -currDirection);
			setTurnRightRadians( Math.atan( Math.tan( perpAngle + diveAngle)));
		}

		// perform some wall-avoidance, try to slide away if moving near wall
		myHeading = getHeadingRadians();
		myDistance = Math.abs( getDistanceRemaining());
		myCheckDistance = Math.min( BLINDMANSTICK, myDistance) * currDirection;
		newposx = myPos.getX() + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
		newposy = myPos.getY() + Math.cos( myHeading) * myCheckDistance;
		// if endposition not in field, adapt
		if (!playField.contains( newposx, newposy)) {
			myTurnAngle = Math.atan( Math.tan( calcAngleToWall()));
			setTurnRightRadians( myTurnAngle);
//			out.println("turn: "+Math.toDegrees(myTurnAngle));
		}

		// perform some dive-in protection here
		if (endGame == true)
			// allow steeper dive when in endGame
			maxDiveAngle = SIXTHPI;
		else
			maxDiveAngle = DEFDIVEANGLE - (enemy.getDistance() / 600) * SIXTHPI;
//		out.println("Dist: "+enemy.getDistance()+" maxDA: "+Math.toDegrees(maxDiveAngle));
		if ((Math.abs(enemy.getBearing() - getTurnRemainingRadians()) < maxDiveAngle) && (currDirection == 1)) {
//			out.println("F: perpAngle: "+perpAngle+" currDir: "+ currDirection);
			currDirection = -currDirection;
			setAhead( myDistance * currDirection);
			setTurnRightRadians( Math.atan( Math.tan( perpAngle + jiggleOffset)));
		}
		if ((Math.abs(enemy.getBearing() - getTurnRemainingRadians()) > (PI - maxDiveAngle)) && (currDirection == -1)) {
//			out.println("B: perpAngle: "+perpAngle+" currDir: "+ currDirection);
			currDirection = -currDirection;
			setAhead( myDistance * currDirection);
			setTurnRightRadians( Math.atan( Math.tan( perpAngle + jiggleOffset)));
		}

		// If a wallhit is eminent (not inside 'safezone'), turn sharply by reducing speed
		if (!playField.contains( myPos)) {
			setMaxVelocity( Math.abs( getTurnRemaining()) > 15 ? MINSPEED : prefSpeed);
//			out.println("Danger ! "+mySpeed);
		}
		else {
//			// change the speed now and then to lure any aiming-technique
//			if (changeSpeedTime < scanTime) {
//				changeSpeedTime = scanTime + 3L + (long)(Math.random() * 15);
//				prefSpeed = MEDSPEED + Math.random() * MAXSPEED;
//			}
            prefSpeed = MAXSPEED;
			setMaxVelocity( prefSpeed);
		}
	}

	/**
	 * calcAngleToWall: Calculate the wall'smoothing' (or something that looks like it)
	 */
	public double calcAngleToWall() {
		double di;

		for (di = 0; di < PI; di += PI/36) {
			if (checkInPlayField( di) == true) {
				break;
			}
			if (checkInPlayField( -di) == true) {
				di = -di;
				break;
			}
		}
		return ( di);
	}

	/**
	 * checkInPlayField: Check if my future position is in the playField
	 */
	public boolean checkInPlayField( double di) {
		double px, py;

		px = myPos.getX() + Math.sin( myHeading + di) * myCheckDistance;
		py = myPos.getY() + Math.cos( myHeading + di) * myCheckDistance;
		return (playField.contains( px, py));
	}

	/**
	 * doCheckEnemyFired: Check if the enemy has fired a bullet. Used for endGame purposes
	 */
	void doCheckEnemyFired() {

		double powerFired = enemy.checkBulletFired();
		if (powerFired > 0.0) {
			airClearTime = scanTime + (long)(Math.max((enemy.getDistance() + 150) / (20 - 3 * powerFired), 20));
//			out.println("AirClearTime: "+airClearTime);
		}
		// check if enemy stopped firing for energysavings
		endGame = false;
		if ((scanTime > airClearTime) && (enemy.getEnergy() <= 8.0)) {
			endGame = true;
//			out.println("Endgame initiated!");
		}
	}

	/**
	 * doRadar: Just do a simple widening radarlock
	 */
	void doRadar() {
		setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( getHeadingRadians() + enemy.getBearing() - getRadarHeadingRadians()));
	}

	/**
	 * doFirePower: Select the optimal firepower for the current target
	 */
	void doFirePower() {
		if (TC_flag == true) {
			firePower = Math.min( 3.0, getEnergy());
			return;
		}
		else if (MC_flag == true) {
			firePower = 0.0;
			return;
		}

		if (enemy.getDistance() > 850)
			firePower = 0.1;
		else if (enemy.getDistance() > 700)
		    firePower = 0.49;
		else if (enemy.getDistance() > 250)
		    firePower = 1.9;
		else
			firePower = 3;

		// do energymanagement
		if (endGame == true)
			// try to keep initiative while in endGame
			firePower = Math.min( (getEnergy() - enemy.getEnergy())/5, firePower);	// energy management
		else
			firePower = Math.min( getEnergy()/5, firePower);	// energy management
		firePower = Math.min( enemy.getEnergy()/4 + 0.1, firePower);	// energy management
		firePower = Math.max( 0.1, firePower);				// but always shoot at least 0.1

		if (getEnergy() <= 0.1) firePower = 0.0;			// don't fire if it disables yourself
	}

	/**
	 * doMoveGun: Move the gun to the optimal angle to hit the target
	 */
	void doMoveGun() {

		if (true == MC_flag) return;

        // for PM gun: record pattern (except first scan)
        if ((prevTime != 0) && (gunOverride == false)) {
			record(enemy.getHeadingChange(), enemy.getSpeed(), scanTime - prevTime);
        	if ((scanTime - prevTime) > 1) {
            	out.println(scanTime+" timeDelta: " + (scanTime - prevTime));
        	}
		}

		// always move gun like backup-gun
		setTurnGunRightRadians( Utils.normalRelativeAngle( doGunOrbit() - getGunHeadingRadians()));
		// if gun is almost cool AND firepower != zero, overrule with PM-gun
		if (((getGunHeat() / getGunCoolingRate()) <= 3.0) && (firePower > 0.0) && (getEnergy() > 0.0)) {
			double tmpangle = doGunPM() - getGunHeadingRadians();
			if (tmpangle < 10)		// valid match found
				setTurnGunRightRadians( Utils.normalRelativeAngle( tmpangle));
		}
	}

	/**
	 * doFireGun: Fire the gun. Calculate angle for all virtual guns
	 */
	void doFireGun() {

		if (MC_flag == true) return;
		// Only fire if possible and target locked.
		if ((getGunHeat() == 0.0) && (getGunTurnRemaining() == 0.0)) {
			// In TC-mode, just fire power 3.0 bullets
			if ((TC_flag == true) && (firePower != 3.0))
				firePower = 3.0;
			// fire only if bulletpower is set, so don't disable yourself !
//			if ((firePower > 0.0) && (getEnergy() - firePower > 0.0) && (getEnergy() > 0.0)) {
			if ((firePower > 0.0) && (getEnergy() > 0.0)) {
				if ((TC_flag == true) || (getEnergy() - firePower > 0.01)) {
					setFire( firePower);
					gunPMStage = 0;
					if (PMActive == true)
						bullPM++;
					else
						bullBCK++;
				}
			}
		}
	}

	/**
	 * doGunOrbit: Assume enemy is orbiting me with constant speed and distance.
	 *				Gun taken from Iiley's Smog 2.5
	 */
	double doGunOrbit() {
		double tmpb = getHeadingRadians() + enemy.getBearing() - enemy.getHeading() + HALFPI;
		if (enemy.getSpeed() < 0) tmpb -= PI;
		return( getHeadingRadians() + enemy.getBearing() + Math.atan2(Math.cos(tmpb) * Math.abs(enemy.getSpeed()), 14 + Math.sin(tmpb)));
	}

	/**
	 * doGunPM: Calculate angle using Pattern Matching.
	 */
	double doGunPM() {
        int startPlaybackIndex = 0;
		int timeOffset = 0;

		bulletVelocity = 20 - (3 * firePower);
		if (gunPMStage == 0) {
			// first find some 'loose' patterns
			startPlaybackIndex = searchLoosePMMatch();
			timeOffset = 3;
		}
		if (gunPMStage == 1) {
			// now find some 'tight' patterns
			startPlaybackIndex = searchLoosePMMatch();
			timeOffset = 2;
		}
		if (gunPMStage == 2) {
			// at last find the pattern with the most probable outcome
			startPlaybackIndex = searchLoosePMMatch();
			timeOffset = 1;
		}

		if (startPlaybackIndex > 0) {
			gunPMStage++;
			guessedEnemyBearing = playScene( startPlaybackIndex, timeOffset);
			PMActive = true;
		}
		else {
			guessedEnemyBearing = (double)BREAK_KEY;
			PMActive = false;
		}

		return( guessedEnemyBearing);
	}

	/**
	 * onScannedRobot: What to do when you see another robot (just save everything and do nothing here)
	 *
	 */
	public void onScannedRobot( ScannedRobotEvent e) {
		// put the data into the EnemyInfo class
		// do the complete running of the bot in this event
		prevTime = scanTime;
		scanTime = getTime();
//		myPrevPos.setLocation( myPos.getX(), myPos.getY());	// Remember previous position
		myPos.setLocation( getX(), getY());					// Remember current position
		myNextPos.setLocation( getX() + Math.sin( getHeadingRadians()) * getVelocity(), getY() + Math.cos( getHeadingRadians()) * getVelocity());	// Next position (approx)

		enemy.onScannedRobot( e);
		enemy.myInfo( getHeadingRadians(), myPos);

		doCheckEnemyFired();	// Check if enemy fired
		if ((e.getEnergy() == 0.0) || (getEnergy() == 0.0)) gunOverride = true;
		if (TC_flag == false) doMovement();			// Check if adaptions have to be made for movement
		doRadar();				// Check the state of the radar
		doFireGun();			// Fire the gun (with angle of previous tick)
		doFirePower();			// Select the fire power to use the next tick (gunmovement depends on it)
		doMoveGun();			// Move the gun to the desired position (for next tick)
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
//	do nothing if someone else dies, you have won!

	/**
	 * onHitRobot: Bounce off !
	 */	
	public void onHitRobot(HitRobotEvent e) {
		// if I hit the opponent, just bounce off
		enemy.onHitRobot();
		if (e.isMyFault()) {
			robothit = true;	// 2move2
		}
	}

	/**
	 * onHitWall:  Handle collision with wall.
	 */
	public void onHitWall(HitWallEvent e)
	{
		wallhit++;
	}

	/**
	 * onHitByBullet:  I am hit, handle some (movement) statistics.
	 */
	public void onHitByBullet(HitByBulletEvent e)
	{
		enemy.onHitByBullet( e);
	}

	/**
	 * onSkippedTurn:  Handle a skipped turn.
	 */
	public void onSkippedTurn(SkippedTurnEvent e)
	{
		skipturn++;
		out.println("AAaaaarghhhhh");
	}

	/**
	 * onWin: Show my private victory dance
	 */
	public void onWin(WinEvent e)
	{
		//Victory dance	
		printStats();
		setTurnGunRight( Double.POSITIVE_INFINITY);
		setTurnRadarLeft( Double.POSITIVE_INFINITY);
		setTurnRight(10);
		ahead(20);
		waitFor(new RadarTurnCompleteCondition(this));
	}

	/**
	 * onDeath: Show some statistics
	 */
	public void onDeath(DeathEvent e)
	{
		// Show some statistics	
		printStats();
	}

	/**
	 * printStats: Print statistics on end of round
	 */
	public void printStats( )
	{
		// skip statistics of this round if PM-buffer gets too big
		if (movieSize > PM_MAXFILL) movieSize = lastValidMovieSize;

		out.println("Bullets HIT:" + bullhit);
		out.println("Bullets mis:" + bullmis);
		if (false == MC_flag) out.println("Promillage :" + (bullhit*1000)/(bullhit + bullmis));
		out.println("PM fired:"+bullPM+"  backup fired:"+bullBCK);
		out.println("Wall hits  :" + wallhit);
		out.println("SkipTurns  :" + skipturn);
		out.println("MovieLength:" + movieSize);
	}

	/**
	 * onBulletHit: Yes, I hit somebody
	 */
	public void onBulletHit(BulletHitEvent e)
	{
		enemy.onBulletHit( e);
		bullhit++;
	}

	/**
	 * onBulletMissed: Damn, I missed (again)
	 */
	public void onBulletMissed(BulletMissedEvent e)
	{
		bullmis++;
	}

//== PM-stuff =================================
	/**
	 * record: Record movement of enemy, meaning speed and headingchange
	 */
    void record( double eHeadingDelta, double eVelocity, long timeDelta) {
        double speedChange = (eVelocity - enemy.getPrevSpeed()) / timeDelta;
        eHeadingDelta /= timeDelta;
		eVelocity = enemy.getPrevSpeed();
        for (int i = 1; i <= timeDelta; i++) {
//			out.println(scanTime+" V "+(eVelocity+(i*speedChange))+" HD "+Math.toDegrees(eHeadingDelta));
	        if (movieSize < PM_LENGTH) {
				movie[movieSize].setFrame( eVelocity+(i*speedChange), eHeadingDelta);
//				out.println(scanTime+" h "+((movie[movieSize]/256)-10)+" v "+((movie[movieSize]%256)-MAXSPEED));
	            movieSize++;
			}
        }
    }

	/**
	 * searchTightPMMatch: Search the longest tight match for the current enemymovement
	 */
    private int searchTightPMMatch() {
        int index = movieSize - 1;
		int endOffset = (int)(enemy.getDistance() * 1.3 / bulletVelocity); // skip end of rounds, no playback possible
        int searchIndex = index - endOffset;
		int matchIndex = -1;
        int matchLength = 0;
		int maxMatchLength = 0;
		int curSpeedFac;
		int curAngleFac;
        // search back from searchindex until match with index
		for (int i = 0; i < searchIndex; i++) {
			if (movie[index].getKey() == movie[searchIndex - i].getKey()) {
				// first match found, now check 'mirror'factors and find length of match
				curSpeedFac = movie[index].getSpeedFac() * movie[searchIndex - i].getSpeedFac();
				curAngleFac = movie[index].getAngleFac() * movie[searchIndex - i].getAngleFac();
				matchLength = 1;
				do {
					if (movie[index - matchLength].getKey() == movie[searchIndex - i - matchLength].getKey()) {
						if (curSpeedFac == 0)
							curSpeedFac = movie[index - matchLength].getSpeedFac() * movie[searchIndex - i - matchLength].getSpeedFac();
						else if (checkFactor( curSpeedFac, movie[index - matchLength].getSpeedFac(), movie[searchIndex - i - matchLength].getSpeedFac()) == false)
							break;
						if (curAngleFac == 0)
							curAngleFac = movie[index - matchLength].getAngleFac() * movie[searchIndex - i - matchLength].getAngleFac();
						else if (checkFactor( curAngleFac, movie[index - matchLength].getAngleFac(), movie[searchIndex - i - matchLength].getAngleFac()) == false)
							break;
						// match is OK
						matchLength++;
					}
					else
						break;
				}
				while (matchLength < (searchIndex - i));
				if (matchLength > maxMatchLength) {
//					tightRow.setCandidate( searchIndex - i, curSpeedFac, curAngleFac, matchLength);
					maxMatchLength = matchLength;
					matchIndex = searchIndex - i;
					speedFactor = curSpeedFac;
					angleFactor = curAngleFac;
				}
			}
			else {
				// if round-break, skip last part of previous round to guarantee playback possible
				if ((movie[searchIndex - i].getKey() % 256) >= BREAK_KEY) {
					i += endOffset;
				}
			}
		}
//		out.println("MaxMatchLen:"+maxMatchLength);
		if (maxMatchLength < 10) {
			matchIndex = 0;		// no match, use backup gun
		}
		return( matchIndex);
    }

	/**
	 * searchLoosePMMatch: Search the longest loose match for the current enemymovement
	 */
    private int searchLoosePMMatch() {
        int index = movieSize - 1;
		int endOffset = (int)(enemy.getDistance() * 1.3 / bulletVelocity); // skip end of rounds, no playback possible
        int searchIndex = index - endOffset;
		int matchIndex = -1;
        int matchLength = 0;
		int maxMatchLength = 0;
		int curSpeedFac;
		int curAngleFac;
        // search back from searchindex until match with index
		for (int i = 0; i < searchIndex; i++) {
			if (looseMatch( movie[index].getKey(), movie[searchIndex - i].getKey()) == true) {
				// first match found, now check 'mirror'factors and find length of match
				curSpeedFac = movie[index].getSpeedFac() * movie[searchIndex - i].getSpeedFac();
				curAngleFac = movie[index].getAngleFac() * movie[searchIndex - i].getAngleFac();
				matchLength = 1;
				do {
					if (looseMatch( movie[index - matchLength].getKey(), movie[searchIndex - i - matchLength].getKey()) == true) {
						if (curSpeedFac == 0)
							curSpeedFac = movie[index - matchLength].getSpeedFac() * movie[searchIndex - i - matchLength].getSpeedFac();
						else if (checkFactor( curSpeedFac, movie[index - matchLength].getSpeedFac(), movie[searchIndex - i - matchLength].getSpeedFac()) == false)
							break;
						if (curAngleFac == 0)
							curAngleFac = movie[index - matchLength].getAngleFac() * movie[searchIndex - i - matchLength].getAngleFac();
						else if (checkFactor( curAngleFac, movie[index - matchLength].getAngleFac(), movie[searchIndex - i - matchLength].getAngleFac()) == false)
							break;
						// match is OK
						matchLength++;
					}
					else
						break;
				}
				while (matchLength < (searchIndex - i));
				if (matchLength > maxMatchLength) {
//					looseRow.setCandidate( searchIndex - i, curSpeedFac, curAngleFac, matchLength);
					maxMatchLength = matchLength;
					matchIndex = searchIndex - i;
					speedFactor = curSpeedFac;
					angleFactor = curAngleFac;
				}
			}
			else {
				// if round-break, skip last part of previous round to guarantee playback possible
				if ((movie[searchIndex - i].getKey() % 256) >= BREAK_KEY) {
					i += endOffset;
				}
			}
		}
//		out.println("MaxMatchLen:"+maxMatchLength);
		if (maxMatchLength < 15)
			matchIndex = 0;		// no match, use backup gun

		return( matchIndex);
    }

	/**
	 * looseMatch: if the match is only 1 off (+ or -), accept it as match
	 *			  this is accepted for speed and for angledifference
	 */
    boolean looseMatch( int pat, int ckval) {
		int speed;
		int dangle;
		int ckspeed = (ckval % 256);
		int	ckdangle = (ckval / 256);

		// if roundbreak, always end of match
		if (ckspeed >= BREAK_KEY) return false;
		speed = (pat % 256);
		if (Math.abs(speed - ckspeed) <= 1) {
			dangle = (pat / 256);
			if (Math.abs( dangle - ckdangle) <= 1)
				return true;
		}
		return false;
    }

	/**
	 * checkFactor: check if the factor is the same
	 *			  if one of the factors is zero, it is always ok
	 */
    boolean checkFactor( int chkval, int val1, int val2) {

		if ((val1 != 0) && (val2 != 0) && (val1 * val2 != chkval))
		    return false;
		return true;
    }

	/**
	 * playScene: Play the found scene until bullet will hit
	 *			  Just looks like iterating the circular gun :-D
     *            Take notice of delay before firing
	 */
    double playScene( int index, int delay) {
		Point2D.Double point = new Point2D.Double();
		double gunAngle;
		double tmpx = enemy.getPos().getX();
        double tmpy = enemy.getPos().getY();
		double head = enemy.getHeading();
		double speed;
		double bspeed;
		index = index+2;
		int hittime = -1;
		// nextposition is calculated (by hittime = -1), because of headingchange
		point.setLocation( tmpx, tmpy);
		do {
			speed = movie[index+hittime].getSpeed() * speedFactor;
			tmpx += Math.sin( head) * speed;
			tmpy +=	Math.cos( head) * speed;
			head += movie[index+hittime].getHeadingChange() * angleFactor;
			hittime++;
			// if position not in field, adapt
			if ((!fireField.contains( tmpx, tmpy)) && (hittime > 0)) {
				bspeed = myNextPos.distance( point) / hittime;
				if (bspeed < 19.7) firePower = (20 - bspeed) / 3.0;
//				else if ((myNextPos.distance( point) / 19.7) > hittime + 5) bpower = 0.0;
				else firePower = 0.1;
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (myNextPos.distance( point) - 18) / (20 - (3 * firePower))) > (hittime - delay));
//		out.println("Tn: "+(getTime()+hittime)+"HPos X: "+point.x+" Y: "+point.y);
//		looseRow.setResult( point, firePower);

		// Return gunAngle
		gunAngle = ( HALFPI - Math.atan2( point.getY() - myNextPos.getY(), point.getX() - myNextPos.getX()) );

        return( gunAngle);
    }

//===================================

	/**
	 * EnemyInfo: Info about enemies
	 */
	class EnemyInfo {
		/*
		 * Storage for all sorts of info about enemy.
		 */
		private double  enemyBearing;
		private double  enemyHeading;
		private double  enemyHeadingChange;
		private double  enemySpeed;
		private double  enemyDistance;
		private double	enemyEnergy = 100.0;
		private double	enemyPrevSpeed;
		private double	enemyPrevEnergy;
		private double	energyDropCorrection;
		private double	enemyBulletPower;
		private boolean	collision;	// if true, robot collision. if false, possible enemy hit wall
		private Point2D.Double enemyPos	= new Point2D.Double();
		private Point2D.Double enemyNextPos	= new Point2D.Double();
		private Point2D.Double enemyPrevPos = new Point2D.Double();

		public EnemyInfo() {
		}

		public void onScannedRobot (ScannedRobotEvent e) {
			enemyBearing = e.getBearingRadians();
			enemyHeadingChange = robocode.util.Utils.normalRelativeAngle(e.getHeadingRadians() - enemyHeading);
			enemyHeading = e.getHeadingRadians();
			enemyDistance = e.getDistance();
			enemyPrevSpeed = enemySpeed;
			enemySpeed = e.getVelocity();
			enemyPrevEnergy = enemyEnergy;
			enemyEnergy = e.getEnergy();
		}

		public void myInfo ( double myhead, Point2D.Double mypos) {
			double absbearing_rad = myhead + enemyBearing;
			enemyPrevPos.setLocation( enemyPos.getX(), enemyPos.getY());
			enemyPos.setLocation( mypos.getX() + Math.sin( absbearing_rad) * enemyDistance, mypos.getY() + Math.cos( absbearing_rad) * enemyDistance);
			enemyNextPos.setLocation( enemyPos.getX() + Math.sin( enemyHeading) * enemySpeed, enemyPos.getY() + Math.cos( enemyHeading) * enemySpeed);
		}
		
		public void onBulletHit( BulletHitEvent e) {
			double damage = 4 * e.getBullet().getPower() + 2 * Math.max( e.getBullet().getPower() - 1, 0);
			energyDropCorrection += damage;
		}

		public void onHitByBullet( HitByBulletEvent e) {
			double damage = -3 * e.getPower();
			energyDropCorrection += damage;
		}

		public void onHitRobot() {
//			energyDropCorrection += 0.6;
			collision = true;
		}

		public double checkBulletFired() {
			double bpower = 0.0;
//			out.println(scanTime+" pEn "+enemyPrevEnergy+" En "+enemyEnergy);
			if (enemyEnergy != enemyPrevEnergy) {
				if (collision == true) {
					energyDropCorrection += 0.6;
//					out.println(scanTime+" Robot collision");
				}
				if ((collision == false) && (enemySpeed == 0) && (Math.abs( enemySpeed - enemyPrevSpeed) > 2)) {
					energyDropCorrection += (Math.abs(enemyPrevSpeed / 2) - 0.5);
//					out.println(scanTime+" Enemy hit the wall");
				}
				bpower = enemyPrevEnergy - (enemyEnergy + energyDropCorrection);
				if (bpower < 0.09) bpower = 0.0;
			}
//			if (bpower > 0.0) {
//				out.println(scanTime+" Bullet Fired !!! Energydrop: "+bpower);
//				out.println("eV "+enemySpeed+" epV "+enemyPrevSpeed+" X "+enemyPos.getX()+" Y "+enemyPos.getY());
//			}
			energyDropCorrection = 0.0;
			collision = false;
			enemyBulletPower = bpower;
			return bpower;
		}

		public double getBearing() {
			return enemyBearing;
		}

		public double getHeading() {
			return enemyHeading;
		}

		public double getHeadingChange() {
			return enemyHeadingChange;
		}

		public double getSpeed() {
			return enemySpeed;
		}

		public double getPrevSpeed() {
			return enemyPrevSpeed;
		}

		public double getEnergy( ) {
			return enemyEnergy;
		}

		public double getDistance( ) {
			return enemyDistance;
		}

		public Point2D.Double getPos( ) {
			return enemyPos;
		}

		public Point2D.Double getNextPos( ) {
			return enemyNextPos;
		}

		public Point2D.Double getPrevPos( ) {
			return enemyPrevPos;
		}
	}

	/**
	 * Frame: Keep exact PM-values for playback instead of simplified ones
     *
     * the key has the following layout:(v0.8-v0.9fix)
     * ---xxxxx x--xxxxx
     *      l   l    L- speed + 8  (0-16)
	 *      l   L------ roundbreak notification
	 *      L---------- angle + 10 (0-20)
     *
     * the key has the following layout:(v0.10-     )
     * ----xxxx x---xxxx
     *      l   l    L- abs speed   (0-8)
	 *      l   L------ roundbreak notification
	 *      L---------- abs angle   (0-10)
	 */
	class Frame {
		double speed;
		double headingChange;
		int    key;				// number for quick reference
		int    speedfac = 0;	// speedfactor -1, 0 or 1
		int    anglefac = 0;	// anglefactor -1, 0 or 1

	    Frame( ) {
		}

		public void setFrame(double s, double h) {
	        speed = s;
	        headingChange = h;
			key = (int)Math.round(Math.abs(Math.toDegrees(h)));
			if (key != 0) {
				if (h < 0) anglefac = -1;
				else anglefac = 1;
			}
			int tmpkey = (int)Math.round(Math.abs(s));
			if (tmpkey != 0) {
				if (s < 0) speedfac = -1;
				else speedfac = 1;
			}
			key = (key << 8) + tmpkey;
//			key = (int)Math.round(( Math.round(Math.toDegrees(h)+10) * 256) + s + MAXSPEED);
	  	}

		public int getKey( ) {
			return key;
		}

		public int getSpeedFac( ) {
			return speedfac;
		}

		public int getAngleFac( ) {
			return anglefac;
		}

		public double getSpeed( ) {
			return speed;
		}

		public double getHeadingChange( ) {
			return headingChange;
		}
	}	

	/**
	 * Candidate: Candidate for expected future targetlocation
     *
	 */
//	class Candidate {
//		private int	startFrame = 0;
//		private Point2D.Double expectedPos	= new Point2D.Double();
//		private int speedfac = 0;	// speedfactor -1, 0 or 1
//		private int anglefac = 0;	// anglefactor -1, 0 or 1
//		private int length = 0;
//		private double bpower;
//
//	    Candidate( ) {
//		}
//
//		public void setCandidate( int sf, int sfac, int afac, int len) {
//	        startFrame = sf;
//	        speedfac = sfac;
//			anglefac = afac;
//			length = len;
//	  	}
//
//		public void setResult( Point2D.Double exppos, double bpow) {
//			expectedPos = exppos;
//			bpower = bpow;
//		}
//
//		public Point2D.Double getResult( ) {
//			return expectedPos;
//		}
//
//		public double getBPower( ) {
//			return bpower;
//		}
//	}	
}
//=================================
				