package ag.battledata;

import ag.movement.GravEllipse;
import ag.movement.GravLine;

/**
 * Virtual bullet for dodging
 * @author Andree Grosse
 *
 */
public class VirtualBullet {

	// The original position of the attacker
	private DataVector mOriginalPos;

	// The direction the bullet moves
	private DataVector mDirection;
	private GravEllipse mPosition;
	private long mTickFired;
	
	private GravLine mGravLine;
	
	/**
	 * Creates a virtual bullet
	 * @param bulletvelocity The bullet velocity
	 * @param originalpos Original position of Gir
	 * @param originaldiff Orginal difference vector from Gir to the target - must
	 * be normalized
	 * @param dataset TrainDataSet in which the input is stored
	 */
	public VirtualBullet(long tickfired, DataVector originalpos, 
			double bulletvelocity, DataVector direction){
		
		
		mDirection = direction.multiple(bulletvelocity);

		mOriginalPos = originalpos;
		mTickFired = tickfired;
		
		mGravLine = new GravLine(mOriginalPos, 10, 0.3, 30, direction.rotate(Math.PI * 0.5));
		mPosition = new GravEllipse(mOriginalPos, 3 + (bulletvelocity - 11) * 0.35, 300,
				direction, 200, direction.rotate(Math.PI * 0.5), 400);
	
	}
	
	/**
	 * Returns the position
	 * @return the position
	 */
	public DataVector getPosition(){
		return mPosition;
	}
	
	/**
	 * Returns the original positionthe original position
	 * @return
	 */
	public DataVector getOriginalPosition(){
		return mOriginalPos;
	}
	
	/**
	 * Returns the tick this virtual bullet was fired
	 * @return the tick this virtual bullet was fired
	 */
	public long getTickFired(){
		return mTickFired;
	}
		
	/**
	 * Returns the bullets' velocity
	 * @return the bullets' velocity
	 */
	public double getVelocity(){
		return mDirection.length();
	}
	
	/**
	 * Returns the distance from this bullet to its original position
	 * @return the distance from this bullet to its original position
	 */ 
	public double getDistance(){
		return mPosition.minus(mOriginalPos).length();
	}

	/**
	 * Returns the force the GravPoint of this bullet emits on a point
	 * @param point the point for which we want to calculate the received force
	 * @return A DataVector representing the force
	 */
	public DataVector getForceOn(DataVector point){
		
		return mPosition.getForceOn(point).plus(
					mGravLine.getForceOn(point));
		
	}
	
	/**
	 * Moves the virtual bullet
	 */
	public void move(){
		mPosition.x += mDirection.x;
		mPosition.y += mDirection.y;		
	}
	
	// ########################################################################
	

}
