/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* This movement is a kind of random movement; random movement
 * takes place in sectors. It should be used on rather small
 * arenas like <=1000x1000.
 * The battlefield is divided into 4 sectors like this:
 *  _________________
 * |        |        |
 * |   3    |   0    |
 * |________|________|
 * |        |        |
 * |   2    |   1    |
 * |________|________|
 * 
 * Each sector represents one quarter of the battlefield and is
 * numbered according to quarter angles around the center of the
 * battlefield.
 *
 * This movement strives to place the robot, so that it moves
 * in the sector on the side of the arena opposite to the enemy.
 * So if the enemy is in the sector x, the robot should move around
 * sector (x+2)%4. Movement inside the sector is random in a way that
 * each step the robot chooses randomly a point inside the sector
 * and upon reaching the point it selects another one, and so on.
 */
 
public class SectorMovement extends Movement
{

   // margin from the boundaries of each sector beyond which the robot does not move
   static final double sectorMargin = 20;
   // maximum and minimum length of single goTo move when changing sectors
   static final double maximumBorderMove = 300;
   static final double minimumBorderMove = 100;

   // robot for which this movement is used
   AdvancedRobot robot;
   // robot's body
   Body body;
   // enemy to avoid
   Enemy enemy;
   
   // data about the battlefield
   Vector2d arenaSize;
   Vector2d arenaCenter;
   Vector2d[][] sectorBorders;  //< 4 sections (2 points each) acting as boundaries between sectors
   Vector2d sectorSize;         // size of a single sector
   Vector2d sectorPositions[];  // coordinates of bottom left corners of each sector
   
   // if this value is set to true, next run method will select a new destination point
   boolean selectNewDestination = true;
   
   /* at construction provide robot, its body
    * and enemy in one-on-one battle
    */
   public SectorMovement( AdvancedRobot robot, Body body, Enemy enemy )
   {
      this.robot = robot;
      this.body = body;
      this.enemy = enemy;
      initialize();
   }

   /* prepare data about the battlefield
    */
   void initialize()
   {
      arenaSize = new Vector2d( robot.getBattleFieldWidth(), robot.getBattleFieldHeight() );
      arenaCenter = new Vector2d( arenaSize );
      arenaCenter.div( 2 );
      
      /* sector borders are 4 sections; section x separates sectors
       * x and (x+1)%4; 
       * when changing sectors the robot should go to some point on this line;
       */
      sectorBorders = new Vector2d[4][2];
      
      // border separating sector 0 and 1
      sectorBorders[0][0] = new Vector2d( arenaCenter.x + sectorMargin, arenaCenter.y );
      sectorBorders[0][1] = new Vector2d( arenaSize.x - sectorMargin, arenaCenter.y );
      
      // border separating sector 1 and 2
      sectorBorders[1][0] = new Vector2d( arenaCenter.x, arenaCenter.y - sectorMargin );
      sectorBorders[1][1] = new Vector2d( arenaCenter.x, sectorMargin );      
      
      // border separating sector 2 and 3
      sectorBorders[2][0] = new Vector2d( arenaCenter.x - sectorMargin, arenaCenter.y );
      sectorBorders[2][1] = new Vector2d( sectorMargin, arenaCenter.y );   
      
      // border separating sector 3 and 0
      sectorBorders[3][0] = new Vector2d( arenaCenter.x, arenaCenter.y + sectorMargin );
      sectorBorders[3][1] = new Vector2d( arenaCenter.x, arenaSize.y - sectorMargin );   
      
      /* prepare sector positions with margin;
       * positions are bottom left corners of each sector
       */
      sectorPositions = new Vector2d[] {
         new Vector2d( arenaCenter.x + sectorMargin, arenaCenter.y + sectorMargin ),
         new Vector2d( arenaCenter.x + sectorMargin, sectorMargin ),
         new Vector2d( sectorMargin, sectorMargin ),
         new Vector2d( sectorMargin, arenaCenter.y + sectorMargin )
      };

      // sector size is an arena quarter cut by sector margins at each side      
      sectorSize = new Vector2d( arenaCenter.x - 2*sectorMargin, arenaCenter.y - 2*sectorMargin );
   }
   
   /* prepare orders
    */
   public void run()
   {
         
      // allow robot to finish it's previous move
      if ( body.destinationClose() )
         selectNewDestination = true;
         
      // let the robot reach its previous destination
      if ( !selectNewDestination )
         return;
         
      // check if enemy position is known
      if ( enemy.getLogLength() == 0 ) {
         // enemy not scanned at all; move to random sector
         moveInSector( (int)Util.randomLong( 0, 3 ) );
         // do not select new destination next frame
         selectNewDestination = false;
         return;
      }
      
      // get the latest known enemy position
      Vector2d enemyPosition = new Vector2d( enemy.getData( 0 ).position );
      // get the robot position
      Vector2d robotPosition = new Vector2d( robot.getX(), robot.getY() );
      
      // determine at which sectors the robot and the enemy are located
      double enemyAngle = getAngleToCenter( enemyPosition );
      double robotAngle = getAngleToCenter( robotPosition );
      int enemySector = getSectorFromAngle( enemyAngle );
      int robotSector = getSectorFromAngle( robotAngle );
      
      // calculate at which sector the robot would be safest
      int desiredSector = (enemySector+2) & 3;
      
      if ( robotSector != desiredSector ) {  // need to change sector
         // determine in which direction the robot should move
         double direction = Util.normalRelativeAngle( robotAngle - enemyAngle );
         int destinationSector;
         if ( direction < 0 )  // move to previoud sector
            destinationSector = (robotSector + 3) & 3;
         else
            destinationSector = (robotSector + 1) & 3;
         
         // order to change sector
         changeSector( robotSector, destinationSector );
      } else {    // sector is ok, perform goto random movement inside sector
         moveInSector( robotSector );
      }
      
      // do not select new destination next frame
      selectNewDestination = false;
   }
   
   /* when hit enemy, change destination
    */
   public void onHitRobot( HitRobotEvent e )
   {
      selectNewDestination = true;      
   }
   
   /* assuming that robot is at currentSector, moves the robot to the
    * destinationSector; the robot will move through a random point
    * at sector border between sectors;
    * abs(destinationSector-currentSector) must be 1!
    */
   void changeSector( int currentSector, int destinationSector )
   {
      Logger.log( "secmov", "changing sector from " + Integer.toString( currentSector )
                               + " to " + Integer.toString( destinationSector ) );
      // determine sector border to which to go to
      int border;
      if ( currentSector < destinationSector ) {
         if ( currentSector == 0 && destinationSector == 3 )
            border = 3;
         else
            border = currentSector;
      } else {
         if ( currentSector == 3 && destinationSector == 0 )
            border = 3;
         else
            border = destinationSector;
      }
      
      Logger.log( "secmov", "border chosen " + Integer.toString( border ) );
      
      /* the robot will have to move to any point in border section;
       * i will determine angle span to both section ends and
       * choose a random angle between these two angles;
       * then choose a random run lenght and run goTo
       */
       
      // determine section ends' coordinates relative to robot position
      Vector2d robotPosition = new Vector2d( robot.getX(), robot.getY() );
      Vector2d end0 = new Vector2d( sectorBorders[border][0] );
      Vector2d end1 = new Vector2d( sectorBorders[border][1] );
      
      Logger.log( "secmov", "border section " + end0.toString() + ", " + end1.toString() );
      
      end0.sub( robotPosition );
      end1.sub( robotPosition );
      
      // get angles to section ends'
      double angle0 = end0.angle();
      double angle1 = end1.angle();
      
      Logger.log( "secmov", "border angles " + Double.toString( Math.toDegrees( angle0  ) ) + ", "
                            + Double.toString( Math.toDegrees( angle1 ) ) );
      
      // choose random angle between these two
      double randomAngle = angle0 + 
                   Util.randomDouble( 0, Util.normalRelativeAngle( angle1 - angle0 ) );
      
      Logger.log( "secmov", "randomAngle chosen " + Double.toString( Math.toDegrees( randomAngle ) ) );
      
      // choose random move length
      double length = Util.randomDouble( minimumBorderMove, maximumBorderMove );
      
      Logger.log( "secmov", "random length chosen " + Double.toString( length ) );
      
      // calculate the point to go to
      Vector2d goToPoint = new Vector2d();
      goToPoint.setFromAngleDistance( randomAngle, length );
      goToPoint.add( robotPosition );
      
      // make order
      body.goTo( goToPoint );
      
   }
   
   /* move to a random point inside given sector
    */
   void moveInSector( int sector )
   {
      // choose a random point inside the sector
      Vector2d destination = new Vector2d( 
         Util.randomDouble( 
            sectorPositions[sector].x, sectorPositions[sector].x + sectorSize.x ),
         Util.randomDouble( 
            sectorPositions[sector].y, sectorPositions[sector].y + sectorSize.y )
      );
      
      // make order
      body.goTo( destination );
   }

   /* given point position on the arena determines the angle to center of arena
    */
   double getAngleToCenter( Vector2d point )
   {
      // calculate point position relative to arena center
      Vector2d pointRelative = new Vector2d( point );
      pointRelative.sub( arenaCenter );
      // determine angle from the center
      return pointRelative.angle();
   }
   
   /* given point's angle to center of arena, determines which sector (quarter)
    * it is in;
    */
   int getSectorFromAngle( double angle )
   {
      return ( (int)Math.floor( 2 * angle / Math.PI ) ) & 3;
   }
}