package tzu.intel;

import tzu.util.*;
import robocode.*;

/**
 * Stores basic data on a robot.
 */
public class BasicBot implements Comparable, Constants {

    String          name;               // Unique identifier.
    double          energy;             // Energy points remaining.
    double          bearing;            // Bearing from us in degrees.
    double          distance;           // Distance to us.
    boolean         isTeammate;         // True if this robot is on our team.
    int             teamType;           // TeamRobot, Leader, or Droid.

    double          gameSpeed;          // Speed reported by the game.
    double          prevGameSpeed;      // Previous speed reported by game.
    double          speed;              // Always positive.  Range 0 to 8.
    double          prevSpeed;          // Previous speed.  Range 0 to 8.
    double          chgSpeed;           // Acceleration/deacceleration.
    double          prevChgSpeed;       // Previous acceleration/deacceleration.

    double          gameHeading;        // Direction front is pointing in.
    double          prevGameHeading;    // Prev direction front pointed in.
    double          heading;            // Direction of driving.
    double          prevHeading;        // Previous heading.
    double          chgHeading;         // Turning rate in +/- degrees.

    double          x;                  // Current x coordinate.
    double          y;                  // Current y coordinate.

    long            lastUpdate;         // Time of last update.
    int             updateCount;        // Number of times updated.

    AdvancedRobot   myRobot;            // Reference to your robot.

    /**
     * Creates a basic bot object.
     * @param e     The ScannedRobotEvent.
     * @param ar    Your AdvancedRobot.
     */
    public BasicBot(ScannedRobotEvent e, AdvancedRobot ar) {

        myRobot             = ar;

        name                = e.getName();
        energy              = e.getEnergy();
        bearing             = BotMath.zeroTo360(
                              myRobot.getHeading() +
                              e.getBearing());
        distance            = e.getDistance();

        gameSpeed           = e.getVelocity();
        prevGameSpeed       = 0;
        speed               = BotMath.abs(gameSpeed);
        prevSpeed           = 0;
        chgSpeed            = speed - prevSpeed / e.getTime();
        prevChgSpeed        = chgSpeed;
        gameHeading         = e.getHeading();
        heading             = gameHeading;
        if (gameSpeed < 0) heading = (heading + A_180) % A_360;
        prevHeading         = heading;
        prevGameHeading     = gameHeading;
        chgHeading          = 0;

        x                   = myRobot.getX() + BotMath.sin(bearing) * distance;
        y                   = myRobot.getY() + BotMath.cos(bearing) * distance;

        lastUpdate          = e.getTime();
        updateCount++;

        if (myRobot instanceof TeamRobot) {
            isTeammate = (((TeamRobot)myRobot).isTeammate(name));
            if (energy > 130) {
                teamType = LEADER;
            } else if (energy > 110) {
                teamType = DROID;
            } else {
                teamType = TEAMROBOT;
            }

        } else {
            isTeammate = false;
            teamType = TEAMROBOT;
        }
    }

    /**
     * A method to reinitialize a bot on the second and subsequent
     * rounds of a battle.
     */
    public void reinitialize() {

        energy                  = 100;
        bearing                 = 0.0;
        distance                = 0.0;

        speed                   = 0.0;
        prevSpeed               = 0.0;
        gameSpeed               = 0.0;
        prevGameSpeed           = 0.0;
        chgSpeed                = 0.0;
        prevChgSpeed            = 0.0;

        heading                 = 0.0;
        gameHeading             = 0.0;
        prevHeading             = 0.0;
        prevGameHeading         = 0.0;
        chgHeading              = 0.0;

        x                       = 0.0;
        y                       = 0.0;

        lastUpdate              = 0;
    }

    public double getX()                { return x; }
    public double getY()                { return y; }
    public long getLastUpdate()         { return lastUpdate; }
    public double getEnergy()           { return energy; }
    public double getBearing()          { return bearing; }
    public double getDistance()         { return distance; }
    public double getChgHeading()       { return chgHeading; }
    public double getChgSpeed()         { return chgSpeed; }
    public double getPrevChgSpeed()     { return prevChgSpeed; }
    public double getHeading()          { return heading; }
    public double getPrevHeading()      { return prevHeading; }
    public String getName()             { return name; }
    public double getPrevSpeed()        { return prevSpeed; }
    public double getSpeed()            { return speed; }
    public boolean isTeammate()         { return isTeammate; }
    public int getTeamType()            { return teamType; }
    public boolean isDead()             { return energy == -1; }
    public AdvancedRobot getMyRobot()   { return myRobot; }

    /**
     * Update this BasicBot using a ScannedRobotEvent.
     * @param e     The ScannedRobotEvent.
     */
    public void update(ScannedRobotEvent e) {

        updateCount++;
        energy          = e.getEnergy();
        bearing         = BotMath.zeroTo360(
                          myRobot.getHeading() +
                          e.getBearing());
        distance        = e.getDistance();

        prevGameSpeed   = gameSpeed;
        gameSpeed       = e.getVelocity();
        prevSpeed       = speed;
        speed           = BotMath.abs(gameSpeed);
        prevChgSpeed    = chgSpeed;
        chgSpeed        = (speed - prevSpeed) / (e.getTime() - lastUpdate);
        if (prevGameSpeed * gameSpeed < 0) chgSpeed = ROBOT_ACCELERATION;

        prevGameHeading = gameHeading;
        gameHeading     = e.getHeading();
        prevHeading     = heading;
        heading         = (gameSpeed < 0) ?
                          (gameHeading + A_180) % A_360 :
                          gameHeading;
        chgHeading      = BotMath.plusMinus180(
                          prevGameHeading, gameHeading) /
                          (e.getTime() - lastUpdate);

        x               = myRobot.getX() + BotMath.sin(bearing) * distance;
        y               = myRobot.getY() + BotMath.cos(bearing) * distance;
        lastUpdate      = e.getTime();
    }

    /**
     * A convienience method for marking a robot as dead.
     */
    public void updateDead() {
        energy = -1;
    }

    /**
     * Compare this bot to another bot.  If it's the same
     * object, or has the same name, it's considered the
     * same bot.
     */
    public int compareTo(Object o) {
        if (o instanceof String) {
            return name.compareTo((String)o);
        } else {
            return (o == this ? 0 : name.compareTo(((Bot)o).name));
        }
    }

    /** Return true if the robot name is the same; false otherwise. */
    public boolean equals(Bot b) {
        return b.getName().equals(name);
    }

    /** Return a text representation of this robot. */
    public String toString() {

        return
            name                                              +
            "   energy:"      + BotMath.round2(energy)        +
            "   bearing:"     + BotMath.round2(bearing)       +
            "   distance:"    + BotMath.round2(distance)      +
            "   heading:"     + BotMath.round2(heading)       +
            "   turning:"     + BotMath.round2(chgHeading)    +
            "   speed:"       + BotMath.round2(gameSpeed)     +
            "   accel.:"      + BotMath.round2(chgSpeed)      +
            "   x:"           + BotMath.round2(x)             +
            "   y:"           + BotMath.round2(y)             ;
    }
}
