package romz.robot.base;

import static romz.math.RobotMath.abs;
import static romz.math.RobotMath.atan;
import static romz.math.RobotMath.cos;
import static romz.math.RobotMath.direction;
import static romz.math.RobotMath.heading;
import static romz.math.RobotMath.min;
import static romz.math.RobotMath.sin;

import java.awt.Color;
import java.util.List;

import robocode.AdvancedRobot;
import robocode.Bullet;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;
import romz.component.movement.Movement;
import romz.component.radar.Radar;
import romz.component.targeting.Targeting;
import romz.model.Situation;
import romz.model.advice.Advice;
import romz.model.advice.MovementAdvice;
import romz.model.advice.RadarAdvice;
import romz.model.advice.TargetingAdvice;
import romz.model.robot.Enemy;
import romz.model.robot.Hero;
import romz.model.robot.Robot;
import romz.model.wave.Wave;

public abstract class BaseRobot extends AdvancedRobot {

	private Radar radar;
	private Movement movement;
	private Targeting targeting;
	
	private Situation situation = new Situation();
	
	public void onScannedRobot(ScannedRobotEvent e) {
		Enemy last = situation.enemy;
		situation.enemy = new Enemy();
		situation.enemy.scanned = true;
		situation.enemy.timeScanned = getTime();
		situation.enemy.energy = e.getEnergy();
		situation.enemy.relativeBearing = e.getBearing();
		situation.enemy.absoluteBearing = Utils.normalAbsoluteAngleDegrees(situation.hero.last.heading + situation.enemy.relativeBearing);
		situation.enemy.distance = e.getDistance();
		situation.enemy.x = situation.hero.last.x + situation.enemy.distance * sin(situation.enemy.absoluteBearing);
		situation.enemy.y = situation.hero.last.y + situation.enemy.distance * cos(situation.enemy.absoluteBearing);
		situation.enemy.heading = e.getHeading();
		situation.enemy.velocity = e.getVelocity();
		situation.enemy.direction = direction(situation.enemy.heading, situation.enemy.absoluteBearing);
		situation.enemy.last = last;
		if (situation.enemy.last != null) {
			long timeDiff = situation.enemy.timeScanned - situation.enemy.last.timeScanned;
			situation.enemy.acceleration = (situation.enemy.velocity - situation.enemy.last.velocity) / timeDiff;
			situation.enemy.turnRate = Utils.normalRelativeAngleDegrees(situation.enemy.heading - situation.enemy.last.heading) / timeDiff;
			double adjustedLastAbsoluteBearing = heading(situation.hero.x, situation.hero.y, situation.enemy.last.x, situation.enemy.last.y);
			situation.enemy.angularSpeed = abs(situation.enemy.absoluteBearing - adjustedLastAbsoluteBearing) / timeDiff;
		}
		System.out.println("--- Enemy scanned ---");
		System.out.println("enemy.angularSpeed : " + situation.enemy.angularSpeed);
		System.out.println("---------------------\n");
	}
	
	public void run() {
		init();
		while(true) {
			updateData();
			Advice advice = getAdvice();
			takeAction(advice);
			execute();
		}
	}

	private void init() {
		radar = getRadar();
		movement = getMovement();
		targeting = getTargeting();
		setColors(getBodyColor(), getGunColor(), getRadarColor());
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		situation.battleField.height = getBattleFieldHeight();
		situation.battleField.width = getBattleFieldWidth();
	}
	
	private void updateData() {
		situation.time = getTime();
		Hero last = situation.hero;
		situation.hero = new Hero();
		situation.hero.energy = getEnergy();
		situation.hero.x = getX();
		situation.hero.y = getY();
		situation.hero.heading = getHeading();
		situation.hero.radarHeading = getRadarHeading();
		situation.hero.radarTurnRemaining = getRadarTurnRemaining();
		situation.hero.gunHeading = getGunHeading();
		situation.hero.gunHeat = getGunHeat();
		situation.hero.distanceRemaining = getDistanceRemaining();
		situation.hero.last = last;
		if (situation.enemy != null) {
			situation.enemy.absoluteBearing = Utils.normalAbsoluteAngleDegrees(situation.hero.heading + situation.enemy.relativeBearing);
		}
		List<Wave> hittingWaves = situation.updateWaves();
		targeting.handleWaveHit(hittingWaves);
	}
	
	private Advice getAdvice() {
		Advice advice = new Advice();
		advice.radarAdvice = radar.getRadarAdvice(situation);
		advice.movementAdvice = movement.getMovementAdvice(situation);
		advice.targetingAdvice = targeting.getTargetingAdvice(situation);
		return advice;
	}
	
	private void takeAction(Advice advice) {
		scan(advice.radarAdvice);
		move(advice.movementAdvice);
		target(advice.targetingAdvice);
	}
	
	void scan(RadarAdvice radarAdvice) {
		if (radarAdvice.radarTurn != 0) {
			setTurnRadarRight(radarAdvice.radarTurn);
			if (radarAdvice.fullScan) {
				setTurnRight(radarAdvice.radarTurn);
				setTurnGunRight(radarAdvice.radarTurn);
			}
		}
	}

	void move(MovementAdvice movementAdvice) {
		double turn = Utils.normalRelativeAngleDegrees(movementAdvice.heading - situation.hero.heading);
		setTurnRight(turn);
		if (movementAdvice.distance != 0) {
			if (movementAdvice.direction > 0) {
				setAhead(movementAdvice.distance);
			} else {
				setBack(movementAdvice.distance);
			}
		}
	}

	void target(TargetingAdvice targetingAdvice) {
		double gunTurn = Utils.normalRelativeAngleDegrees(targetingAdvice.fireAngle - situation.hero.gunHeading);
		setTurnGunRight(gunTurn);
		if (targetingAdvice.fire && fireAllowed(gunTurn)) {
			double firePower = targetingAdvice.firePower;
			fireMonitoredBullet(firePower);
		}
	}

	private boolean fireAllowed(double gunTurn) {
		return gunIsAdjusted(gunTurn) && situation.hero.gunHeat == 0 && situation.hero.energy > 0.1;
	}
	
	private boolean gunIsAdjusted(double gunTurn) {
		return gunTurn < atan(Robot.HALF_ROBOT_SIZE / situation.enemy.distance);
	}
	
	private void fireMonitoredBullet(double firePower) {
		Bullet bullet = setFireBullet(firePower);
		situation.addWave(bullet);
	}
	
	public abstract Radar getRadar();
	public abstract Movement getMovement();
	public abstract Targeting getTargeting();
	
	public abstract Color getBodyColor();
	public abstract Color getGunColor();
	public abstract Color getRadarColor();
}
