package m3thos;
import robocode.*;
import java.io.Serializable;
/**
 * Enemy data structure!
 * @author  Miguel Sousa Filipe
 */
public class Enemy implements Serializable
{
    private String name;             // the present target name.
    private double distance;         // the present distance from target.
    private double avgSpeed;         // the target's Average speed.
    private double bearing;          // ABSOLUTE bearing!
    private double heading;
    private double energy;           // to calculate things like firepower.
    private float lastX, lastY;     // last known coords

    private long lastScan;             // the time we last seen the target.
    private float inflictedDamage;     // damage inflicted by target on us
    private float inflictedShots;      // shots inflicted by enemy on us
    private float damage;           // damage infliced by us on target
    private float shots, hits;      // shots and hits by us on the enemy

    public boolean isDead = true;

    public String getName() { return name; }
    public double getDistance() { return distance; }
    public double getSpeed() { return avgSpeed; }
    public double getBearing() { return bearing; }
    public double getHeading() { return heading; }
    public double getEnergy() { return energy; }
    public double getLastScan() { return lastScan; }
    public int getShots() { return (int) shots; }
    public int getHits() { return (int) hits; }

    public double getX() { return lastX; }
    public double getY() { return lastY; }

    /** Creates a new instance of Enemy */
    public Enemy( String name)
    { 
        this.name = name;
        this.shots = 0;
        this.hits = 0;
        this.inflictedShots = 0;
        this.damage = 0;
        this.inflictedDamage = 0;
        this.bearing = 0;
        this.heading = 0;
        this.energy = 999;
        this.avgSpeed = 0;
        this.distance = 99999;
    }

    public void updateEnemy( 
            ScannedRobotEvent e, 
            double my_x, 
            double my_y, 
            double my_heading, 
            long time) 
    {

		this.updateSpeed(e.getVelocity());
		this.distance = e.getDistance();
		this.energy = e.getEnergy();
		this.bearing = my_heading + e.getBearingRadians();
		this.heading = e.getHeadingRadians();
		this.lastScan = time;
        this.isDead = false;

        //calculate X,Y based on my_pos+heading
        lastX = (float) ( my_x + getDistance() * Math.sin(bearing) );
        lastY = (float) ( my_y + getDistance() * Math.cos(bearing) );
    }

    public void updateEnemy(HitRobotEvent e, long time)
    {
		this.distance = 30; // we give a 10 pixel radius to find our target
		this.bearing = e.getBearingRadians();
		this.energy = e.getEnergy();
        this.lastScan = time;
    }

    private void updateSpeed(double speed)
    {
        if( avgSpeed != 0)
            avgSpeed = avgSpeed * 0.70 + speed * 0.30;
        else
            avgSpeed = speed;
    }

    public void addHit(double power)
    {
        damage += calcDamageFromPower(power);
        if( hits < shots)
            hits++;
    }

    public void incShots() { shots++; }

    public void addInflictedShot(double power)
    {
        inflictedDamage += calcDamageFromPower(power);
        inflictedShots++;
    }

    public double getDamageRatio()
    { 
        if(inflictedDamage == 0)
            return 0; 
        return damage - inflictedDamage;
    }

    public double getHitRatio()
    {
        if( shots == 0)
            return 1;
        return hits / shots;
    }
    
    public void printFightStats()
    {
        System.out.println("Fighting statistics for: "+ name);
        System.out.println("\tHit Ratio: " + getHitRatio());
        System.out.println("\tDamage ratio: "+ getDamageRatio());
    }

    /**
     * calcFirePower: Method to calculate the fire power for shooting this target
     *  it uses, our energy, distance and energy of the target.
     */
    public double calcFirePower(double myEnergy) {
        double power, damage;
        power = 4*getHitRatio() + (300 / distance);

        // reduce bullet power if low on energy 
        damage = myEnergy / 8;
        if( damage < 1 )
            power *= damage;
        
        if (power > 3) 
            power = 3;
        damage =  calcDamageFromPower(power);
        if (energy <= damage)                                                    
            power =  calcPowerFromDamage(energy+0.1);
        return power;
    }
    

	/**
	 * isBetterTarget chooses between current target and a given enemy the best
	 * target.
	 */
	public boolean isBetterTarget(Enemy e) 
    {
        if( e == null)
            return false;
        if( this.isDead )
            return true;
        if( e.isDead )
            return false;

        double rel_dist = e.getDistance() / getDistance();
        if( rel_dist < 0.75)
            return true;
        double rel_ene = e.getEnergy() / getEnergy();

        /* 10% further away at most, 70% the energy of current target */
        if( rel_dist < 1.1 && rel_ene < 0.70)
            return true;

        return false;
	}

    private double calcDamageFromPower(double power)
    {
        return ((power > 1)? ((6 * power) -2):(4 * power)); 
    }

    private double calcPowerFromDamage(double damage)
    {
        return ((energy/4 > 1)? ((energy + 2)/6):(energy / 4));
    }
}
