package ar.horizon.components.movement;

import static java.lang.Math.*;
import static ar.horizon.util.Util.*;

import java.awt.Color;
import java.awt.geom.*;
import java.util.*;

import ar.horizon.Component;
import ar.horizon.stats.*;
import ar.horizon.util.*;
import ar.horizon.util.graphics.ColoredShape;
import ar.horizon.util.graphics.DrawingUtil;
import robocode.*;

/**
 * @author Aaron Rotenberg
 */
public class SolarWindMovement extends Component {
	private static final boolean ALLOW_ENABLED_FLATTENER = true;
	private static final int ALLOW_ENABLED_FLATTENER_BULLETS = 200;
	private static final double FLATTENER_MIN_HIT_RATE = 0.09;
	private static final double FLATTENER_WEIGHT = 0.3;
	
	/**
	 * The number of waves to look at when surfing. Higher numbers provide more
	 * accuracy but are much, much slower.
	 */
	private static final int SURF_WAVES = 2;
	/**
	 * The distance from the wall to start turning at when wall smoothing.
	 * Higher numbers are safer but provide less freedom of movement; lower
	 * numbers allow a chance of colliding with the wall.
	 */
	private static final double WALL_STICK = 160.0;
	// How did I come up with these numbers? Trial and error, with a little
	// bit of linear equation solving. Specifically:
	// 
	// preferred distance = -(0.5/-0.001) = 500
	private static final double ORBIT_DISTANCE_BASE_FACTOR = 0.5;
	private static final double ORBIT_DISTANCE_ADJUSTMENT_FACTOR = -0.001;
	/**
	 * Determines how much distance from the enemy is weighted when calculating
	 * position danger.
	 */
	private static final double DANGER_DISTANCE_MULTIPLIER = 75.0;
	private static final double DANGER_GUESS_FACTOR_WEIGHT = 1.0;
	/**
	 * Determines how much less a wave is valued when calculating danger if it
	 * will hit after other waves.
	 */
	private static final double DANGER_WAVE_ORDER_IMPORTANCE_FACTOR = 0.25;

	private static final boolean PRINT_GUESS_FACTORS = false;
	/**
	 * Indicates whether this component attempts to paint graphical debugging
	 * info.
	 */
	private static final boolean GRAPHICAL_DEBUGGING = true;

	private static StatBuffer<FireRecording, Double> surfStats =
			new DynamicClusteringStatBuffer<FireRecording, Double>(
					new SolarWindBufferParams());
	private static StatBuffer<FireRecording, Double> surfStatsWithFlattener =
			new DynamicClusteringStatBuffer<FireRecording, Double>(
					new SolarWindBufferParams());
	private static int enemyBulletsFired = 0;
	private static int enemyBulletsHit = 0;
	private static boolean flattenerEnabled = false;
	private static int flattenerEnabledRoundCount = 0;

	private double enemyEnergy = 100.0;
	private double enemyGunHeat = GUN_HEAT_AT_START_OF_ROUND;
	private List<MovementWave> enemyWaves = new ArrayList<MovementWave>();
	/**
	 * This field is only used in Krabb's Optimization Part 2 in
	 * {@link #doMovement}. It is not actually required for the movement to
	 * function.
	 */
	private int currentDirection = 0;

	@Override
	public void run() {
		// We don't need to recalculate the cached GF arrays in the waves
		// because the flattener is never toggled mid-round.
		flattenerEnabled =
				ALLOW_ENABLED_FLATTENER
						&& enemyBulletsFired > ALLOW_ENABLED_FLATTENER_BULLETS
						&& (double) enemyBulletsHit
								/ (double) enemyBulletsFired > FLATTENER_MIN_HIT_RATE;

		if (flattenerEnabled) {
			flattenerEnabledRoundCount++;
		}

		robot.debugPrintLine("Flattener "
				+ (flattenerEnabled ? "enabled" : "disabled") + "; enabled "
				+ flattenerEnabledRoundCount + " / "
				+ (robot.getRoundNum() + 1) + " rounds.");
	}

	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		// Don't create a wave if the enemy could not possibly have fired on
		// this tick. The extra getGunCoolingRate() is a fudge factor - we may
		// be off by a turn in rare cases.
		if (enemyGunHeat <= robot.getGunCoolingRate()) {
			// Wall damage calculation adapted from Dookious, see
			// http://robowiki.net/wiki/Dookious
			double wallDamage =
					max(abs(enemyRecording.getAcceleration()) / 2 - 1, 0.0);
			double enemyBulletPower =
					(enemyEnergy - e.getEnergy()) - wallDamage;

			if (enemyBulletPower > 0.09 && enemyBulletPower < 3.01) {
				RobotRecording myPastRecording = myLog.getEntry(2);
				addWave(myPastRecording, enemyBulletPower);
				enemyGunHeat = getGunHeat(enemyBulletPower);
			}
		}

		enemyEnergy = e.getEnergy();
	}

	private Wave addWave(RobotRecording myPastRecording, double enemyBulletPower) {
		FireRecording fireRecording = new FireRecording(myPastRecording);
		MovementWave newWave =
				new MovementWave(fireRecording,
						myPastRecording.getRoundTime() + 1, enemyBulletPower);
		recalculateCachedGFArray(newWave);

		// Performance Enhancing Bug: technically we should be precise
		// predicting our MEAs here. But, as it turns out, that degrades overall
		// performance, because there are so many guns using the simple
		// calculation.
		double maxEscapeAngle =
				maxEscapeAngle(getBulletSpeed(enemyBulletPower));
		newWave.setMaxEscapeAngleClockwise(maxEscapeAngle);
		newWave.setMaxEscapeAngleCounterclockwise(maxEscapeAngle);

		enemyWaves.add(newWave);
		enemyBulletsFired++;
		return newWave;
	}

	@Override
	public void onBulletHit(BulletHitEvent e) {
		enemyEnergy = e.getEnergy();
	}

	@Override
	public void onHitByBullet(HitByBulletEvent e) {
		enemyBulletsHit++;
		logEnemyBullet(e.getBullet(), "Ouch! I got hit!   ");
	}

	@Override
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		logEnemyBullet(e.getHitBullet(), "Bullet hit bullet! ");
	}

	private void logEnemyBullet(Bullet bullet, String message) {
		Point2D.Double location =
				new Point2D.Double(bullet.getX(), bullet.getY());
		double bulletPower = bullet.getPower();

		for (int i = 0; i < enemyWaves.size(); i++) {
			Wave enemyWave = enemyWaves.get(i);
			double distance =
					location.distance(enemyWave.getFirerRecording().getLocation());

			// Does this wave correspond to the bullet we're looking for?
			if (round(getBulletSpeed(bulletPower) * 10) == round(getBulletSpeed(enemyWave.getBulletPower()) * 10)
					&& abs(distance - enemyWave.getDistanceTraveled()) < 30) {
				// This is a real hit, so set the realBullet value to true.
				enemyWave.getFireRecording().setRealBullet(true);

				logWaveHit(surfStats, enemyWave, location, message);
				logWaveHit(surfStatsWithFlattener, enemyWave, location, message);

				enemyWaves.remove(i);
				return;
			}
		}

		if (PRINT_GUESS_FACTORS) {
			robot.debugPrintLine(message
					+ "(GAH! I can't find a wave corresponding to the opponent's "
					+ "bullet!)");
		}
	}

	private void logWaveHit(StatBuffer<FireRecording, Double> statBuffer,
			Wave enemyWave, Point2D.Double location, String message) {
		double distance =
				location.distance(enemyWave.getFirerRecording().getLocation());

		// We want to discard data from bullets the opponent just fired.
		// Bullets very near to the opponent can make for inaccurate
		// guess factor calculations.
		// TODO: Is this actually necessary? Why did I put this in in
		// the first place?
		if (distance < 40) {
			if (PRINT_GUESS_FACTORS) {
				robot.debugPrintLine(message
						+ "(distance traveled by opponent's bullet is too "
						+ "short to provide good data; ignoring)");
			}
		} else {
			// Add it to the stat buffer.
			final double guessFactor = enemyWave.getGuessFactor(location);
			statBuffer.add(new Pair<FireRecording, Double>(
					enemyWave.getFireRecording(), guessFactor));

			if (PRINT_GUESS_FACTORS) {
				robot.debugPrintLine(message
						+ String.format("(guess factor: %1$+01.2f)",
								guessFactor));
			}

			// Since we updated the stat buffer, we need to recalculate
			// every wave's cached GF array.
			for (MovementWave cachedWave : enemyWaves) {
				recalculateCachedGFArray(cachedWave);
			}
		}
	}

	private void recalculateCachedGFArray(MovementWave wave) {
		StatBuffer<FireRecording, Double> statBuffer =
				(flattenerEnabled ? surfStatsWithFlattener : surfStats);
		wave.setGuessFactorArray(new GuessFactorArray(
				statBuffer.getNearestNeighbors(wave.getFireRecording()),
				FLATTENER_WEIGHT, flattenerEnabled, myRecording.getTotalTime()));
	}

	@Override
	public void onTickEnd() {
		enemyGunHeat -= robot.getGunCoolingRate();
		updateWaves();

		// Here's the actual movement code.
		List<MovementWave> surfWaves = getSurfWaves();
		debugDrawWaves(surfWaves);
		doMovement(surfWaves);
	}

	private void updateWaves() {
		for (int i = 0; i < enemyWaves.size(); i++) {
			MovementWave enemyWave = enemyWaves.get(i);

			enemyWave.advance(robot.getTime());

			if (enemyWave.getPassedMePoint() == null
					&& enemyWave.isHit(myRecording.getLocation())) {
				enemyWave.setPassedMePoint(myRecording.getLocation());
			}

			if (enemyWave.isHitPassive(myRecording.getLocation())) {
				// This is a flattener hit, so set the real bullet value to
				// false.
				enemyWave.getFireRecording().setRealBullet(false);

				logWaveHit(surfStatsWithFlattener, enemyWave,
						enemyWave.getPassedMePoint(), "Wave passed me.    ");
				enemyWaves.remove(i);
				i--;
			}
		}
	}

	/**
	 * @return A list of the waves to surf this tick. These are the <i>n</i>
	 *         waves that will hit us first, where <i>n</i> =
	 *         {@link #SURF_WAVES}.
	 */
	private List<MovementWave> getSurfWaves() {
		// There are typically so few waves in the air at a time that it would
		// be slower to use a priority queue.
		List<MovementWave> unusedEnemyWaves =
				new LinkedList<MovementWave>(enemyWaves);
		List<MovementWave> surfWaves = new ArrayList<MovementWave>(SURF_WAVES);

		for (int i = 0; unusedEnemyWaves.size() > 0 && i < SURF_WAVES; i++) {
			int closestWave = -1;
			double closestToHitTime = Double.POSITIVE_INFINITY;

			for (int j = 0; j < unusedEnemyWaves.size(); j++) {
				MovementWave enemyWave = unusedEnemyWaves.get(j);
				double toHitTime =
						(myRecording.getLocation().distance(
								enemyWave.getFirerRecording().getLocation()) - enemyWave.getDistanceTraveled())
								/ getBulletSpeed(enemyWave.getBulletPower());

				// TODO: Changing this 1.0 to -ROBOT_CENTER may or may not
				// improve performance.
				if (toHitTime > 1.0 && toHitTime < closestToHitTime) {
					closestWave = j;
					closestToHitTime = toHitTime;
				}
			}

			// Condition is necessary because distance will sometimes be less
			// than 1.0 for all waves.
			if (closestWave > -1) {
				surfWaves.add(unusedEnemyWaves.get(closestWave));
				unusedEnemyWaves.remove(closestWave);
			}
		}

		return surfWaves;
	}

	/**
	 * Draws the waves corresponding to the opponent's bullets. Colored waves
	 * are being used for surfing; gray waves aren't.
	 */
	private void debugDrawWaves(List<MovementWave> surfWaves) {
		if (GRAPHICAL_DEBUGGING) {
			for (MovementWave enemyWave : enemyWaves) {
				boolean isSurfWave = surfWaves.contains(enemyWave);

				Collection<ColoredShape> shapes =
						DrawingUtil.drawGuessFactorArray(
								isSurfWave ? enemyWave.getGuessFactorArray()
										: null,
								enemyWave.getFirerRecording().getLocation(),
								enemyWave.getDistanceTraveled(),
								enemyWave.getFirerRecording().getEnemyAbsoluteBearing(),
								enemyWave.getTargetRecording().getLateralDirection(),
								enemyWave.getMaxEscapeAngleClockwise(),
								enemyWave.getMaxEscapeAngleCounterclockwise());

				for (ColoredShape shape : shapes) {
					robot.debugFillShape(shape);
				}
			}
		}
	}

	private void doMovement(List<MovementWave> surfWaves) {
		// Don't move until we've scanned the enemy at least once.
		if (enemyRecording != null) {
			int minimumDangerDirection =
					getMinimumDangerDirection(new PredictedState(myRecording),
							surfWaves, 0).getDirection();

			// If we're surfing a wave, maximize our escape angles by orbiting
			// the location that the enemy bullet was fired from. If there are
			// no waves to surf, orbit the enemy's current position.
			Point2D.Double orbitLocation =
					surfWaves.size() > 0 ? surfWaves.get(0).getFirerRecording().getLocation()
							: enemyRecording.getLocation();
			double goAngle =
					MyGoAngleGenerator.INSTANCE.getGoAngle(
							myRecording.getLocation(), orbitLocation,
							minimumDangerDirection);

			robot.setMaxVelocity(minimumDangerDirection == 0 ? 0 : MAX_VELOCITY);
			robot.setBackAsFront(wallSmoothing(myRecording.getLocation(),
					goAngle, minimumDangerDirection, WALL_STICK, fieldRectangle));

			currentDirection = minimumDangerDirection;
		}
	}

	private DangerPrediction getMinimumDangerDirection(
			PredictedState startPosition, List<MovementWave> surfWaves,
			int waveIndex) {
		boolean surfWaveAvailable = (waveIndex < surfWaves.size());
		if (!surfWaveAvailable && surfWaves.size() > 0) {
			// Recursive lower bound: return 0 danger if we have no more waves
			// left to predict against.
			return new DangerPrediction(null, 0);
		}

		List<DangerPrediction> dangerPredictions =
				new LinkedList<DangerPrediction>();

		// Krabb's Optimization Part 2: calculate the danger of our current
		// direction first.
		int[] directions;
		if (currentDirection == 1) {
			directions = new int[] { 1, 0, -1 };
		} else if (currentDirection == -1) {
			directions = new int[] { -1, 0, 1 };
		} else {
			directions = new int[] { 0, 1, -1 };
		}

		if (surfWaveAvailable) {
			Wave enemyWave = surfWaves.get(waveIndex);

			for (int direction : directions) {
				// Predict how far we can move before this wave hits.
				List<PredictedState> predictedPositions =
						predictPosition(startPosition, enemyWave,
								enemyWave.getFirerRecording().getLocation(),
								direction, MyGoAngleGenerator.INSTANCE,
								WALL_STICK, fieldRectangle);
				for (PredictedState state : predictedPositions) {
					dangerPredictions.add(new DangerPrediction(state, direction));
				}
			}
		} else {
			for (int direction : directions) {
				// No waves to surf; predict dangers by distance only.
				List<PredictedState> predictedPositions =
						predictPosition(startPosition, null,
								enemyRecording.getLocation(), direction,
								MyGoAngleGenerator.INSTANCE, WALL_STICK,
								fieldRectangle);
				for (PredictedState state : predictedPositions) {
					dangerPredictions.add(new DangerPrediction(state, direction));
				}
			}
		}

		for (DangerPrediction prediction : dangerPredictions) {
			// Start with the danger from enemy distance.
			prediction.addDanger(DANGER_DISTANCE_MULTIPLIER
					/ prediction.getState().getLocation().distance(
							enemyRecording.getLocation()));
		}

		DangerPrediction currentMinDanger = null;
		if (surfWaveAvailable) {
			MovementWave enemyWave = surfWaves.get(waveIndex);

			for (DangerPrediction prediction : dangerPredictions) {
				// Add in the guess factor danger.
				double guessFactor =
						enemyWave.getGuessFactor(prediction.getState().getLocation());
				int guessFactorIndex =
						GuessFactorArray.getIndexFromGuessFactor(guessFactor);
				double thisDanger =
						enemyWave.getGuessFactorArray().getAtIndex(
								guessFactorIndex);
				prediction.addDanger(DANGER_GUESS_FACTOR_WEIGHT * thisDanger);

				// Krabb's Optimization Part 1: don't even bother recursing if
				// we already have a higher danger than the current minimum.
				if (currentMinDanger == null
						|| prediction.getDanger() < currentMinDanger.getDanger()) {
					// Recursively add the danger from future waves.
					prediction.addDanger(getMinimumDangerDirection(
							prediction.getState(), surfWaves, waveIndex + 1).getDanger()
							* DANGER_WAVE_ORDER_IMPORTANCE_FACTOR);

					if (currentMinDanger == null
							|| prediction.getDanger() < currentMinDanger.getDanger()) {
						currentMinDanger = prediction;
					}
				}
			}
		} else {
			for (DangerPrediction prediction : dangerPredictions) {
				if (currentMinDanger == null
						|| prediction.getDanger() < currentMinDanger.getDanger()) {
					currentMinDanger = prediction;
				}
			}
		}

		// Only draw debugging data for the first wave, so as not to clutter up
		// the battlefield.
		if (waveIndex == 0) {
			debugDrawPredictedPositions(dangerPredictions);
		}

		return currentMinDanger;
	}

	private static class MyGoAngleGenerator implements GoAngleGenerator {
		public static final MyGoAngleGenerator INSTANCE =
				new MyGoAngleGenerator();

		private MyGoAngleGenerator() {
		}

		public double getGoAngle(Point2D.Double myPosition,
				Point2D.Double enemyPosition, int direction) {
			int nonzeroDirection = sign(direction);
			double goAngle = absoluteBearing(enemyPosition, myPosition);
			double moveAwayFactor =
					PI
							/ 2
							- limit(
									-(PI / 4 - 0.1),
									ORBIT_DISTANCE_ADJUSTMENT_FACTOR
											* enemyPosition.distance(myPosition)
											+ ORBIT_DISTANCE_BASE_FACTOR,
									PI / 4 - 0.1);
			return goAngle + (nonzeroDirection * moveAwayFactor);
		}
	}

	private void debugDrawPredictedPositions(List<DangerPrediction> predictions) {
		if (GRAPHICAL_DEBUGGING) {
			int i = 0;
			for (DangerPrediction prediction : predictions) {
				PredictedState state = prediction.getState();
				double danger = prediction.getDanger();

				Ellipse2D.Double shape =
						new Ellipse2D.Double(state.getLocation().x - 2,
								state.getLocation().y - 2, 4, 4);
				Color color = DrawingUtil.getIntensityColor(danger - 0.1);
				robot.debugFillShape(new ColoredShape(shape, color));

				i++;
			}
		}
	}

	@Override
	public void onRoundEnd() {
		robot.debugPrintLine("Enemy bullets hit: " + enemyBulletsHit + " / "
				+ enemyBulletsFired);
	}
}
