//--------------------------------------------------------------------------
// $Id: Wave.java,v 1.1 2003/08/29 09:18:58 erning Exp $
//--------------------------------------------------------------------------
// Copyright (c) 2000-2003 Dragon Software Corp. All rights reserved.
//
// Please refer to COPYRIGHT for notices regarding the confidential
// and proprietary nature of this source code.
//--------------------------------------------------------------------------

package net.dragonsoft.robocoding.lindada;

import net.dragonsoft.robocoding.util.Position;
import net.dragonsoft.robocoding.Enemy;
import net.dragonsoft.robocoding.EnemyState;

public class Wave
{
    public final static int STATUS_TRAVELING = 0;
    public final static int STATUS_HIT       = 1;
    public final static int STATUS_SKIPPED   = 2;
    public final static int STATUS_EXPIRED   = 3;

    /**
     * Call by WaveTracer
     * @param enemy
     * @param power
     */
    public Wave(Enemy enemy, double power)
    {
        _enemy = enemy;
        _ses = enemy.getCurrentState();
        //if (power < 0.1d) power = 0.1d; else if (power > 3d) power = 3d;
        _power = power < 0.1d ? 0.1d : power > 3d ? 3d : power;
        _velocity = 20d - 3d * power;
        _turnAngle = Double.POSITIVE_INFINITY;
        _radius = 0d;
        _status = STATUS_TRAVELING;
    }

    /**
     * Calling by WaveTracer
     * @param enemy
     */
    public void update(Enemy enemy)
    {
        // is it the correct one?
        if (enemy != _enemy) return;
        // need test?
        if (_status != STATUS_TRAVELING) return;

        EnemyState es = enemy.getCurrentState();

        // distance from enemy's current position to the wave's center
        double distance = Position.calcDistance(_ses.myX, _ses.myY, es.x, es.y);
        // travel time
        _travelTime = es.time - _ses.time;
        // current radius of the wave
        _radius = _velocity * _travelTime;
        if (Math.abs(distance - _radius) < 18d)
        {   // hit the enemy ?
            // absolute from wave's center to the enemy's current position
            double angle = Position.calcAngle(_ses.myX, _ses.myY, es.x, es.y);
            double turnAngle = robocode.util.Utils.normalRelativeAngle(angle - _ses.angle);
            _turnAngle = turnAngle;
            // fix the radius
            _radius = distance;
            if (Math.abs(turnAngle) < Math.toRadians(20d * _travelTime))
            {
                _status = STATUS_HIT;
                _hes = es;
                /*
                System.out.print(_ses.myX + "," + _ses.myY + " - ");
                System.out.print(_hes.x + "," + _hes.y + " - ");
                System.out.print(distance + " - " + _power + " - ");
                System.out.println(_hes.time - _ses.time);
                */
            }
            else
            {   // too large angle to turn, so skip it
                _status = STATUS_SKIPPED;
                _hes = es;
            }
        }
        else if (_radius > distance)
        {
            _status = STATUS_EXPIRED;
        }
    }

    /**
     * Returns current status of the wave
     */
    public int getStatus()
    {
        return _status;
    }

    /**
     * Returns the power of the wave (0.1 <= power <= 3.0)
     */
    public double getPower()
    {
        return _power;
    }

    /**
     * Returns the velocity of the wave (20 - 3 * power)
     */
    public double getVelocity()
    {
        return 20d - 3d *_power;
    }

    /**
     * Returns the tick passes of the wave. It will increase while the wave's
     * status is STATUS_FLYING
     */
    public long getTravelTime()
    {
        return _travelTime;
    }

    /**
     * Returns the current radius of the wave. It will increase while the wave's
     * status is STATUS_FLYING
     */
    public double getRadius()
    {
        return _radius;
    }

    /**
     * Returns the current angle changed
     */
    public double getTurnAngle()
    {
        return _turnAngle;
    }

    /**
     * Returns the enemy object that the wave target
     */
    public Enemy getEnemy()
    {
        return _enemy;
    }

    /**
     * Returns the enemy state when the wave was sent
     */
    public EnemyState getSentEnemyState()
    {
        return _ses;
    }

    /**
     * Returns the enemy state when the wave hit the enemy.
     * If wave's status is not STATUS_HIT, it will returns null.
     */
    public EnemyState getHitEnemyState()
    {
        return _hes;
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append("Wave: [").append(_enemy.getName()).append("]");
        sb.append("[status=").append(_status == 0 ? "flying" : _status == 1 ? "hit" : _status == 2 ? "skipped" : _status == 3 ? "expired" : "unknown").append("]");
        sb.append("[radius=").append(Math.round(_radius)).append("]");
        sb.append("[power=").append(Math.round(_power * 10) / 10).append("]");
        sb.append("[sent=").append(_ses).append("]");
        sb.append("[hit=").append(_hes).append("]");

        return sb.toString();
    }

    /* target enemy */
    private Enemy _enemy;

    /* the enemy state when wave was sent */
    private EnemyState _ses;

    /* the enemy state when wave hit the enemy */
    private EnemyState _hes;

    /* power of the wave (same as bullet) */
    private double _power;

    /* velocity of the wave (same as bullet = 20d / 3d * power) */
    private double _velocity;

    /* current radius */
    private double _radius;

    /* tick traveled */
    private long _travelTime;

    /* angle changed */
    private double _turnAngle;

    /* status of the wave */
    private int _status;
}
