package wiki;

import robocode.*;
import java.util.*;

/**
   RobotLog - Use this to record everything that happens.
 */
public class RobotLog {
    static int logLength = 1500;
    static int currentTime=0;
    static Hashtable logs=new Hashtable();
    Hashtable livingRobots=new Hashtable();
    static Hashtable lastUpdateTimes=new Hashtable();
    
    static AdvancedRobot robot;
    static double robotHeadingRadians;
    static double robotX, robotY;
    static double enemyDistance;
    
    public RobotLog(AdvancedRobot r) {
        init(r,1500);
    }
    
    public RobotLog(AdvancedRobot r, int size) {
        init(r,size);
    }
    
    private void init(AdvancedRobot r, int size) {
        robot = r;
        r.addCustomEvent(new Condition("RobotLog") {
            public boolean test() {
                RobotLog.this.record();
                return true;
            }
        });
        logLength = size;
    }
    
    public int getCurrentTime() {
        return currentTime;
    }
    
    public void record() {
        currentTime++;
        
        robotHeadingRadians = robot.getHeadingRadians();
        robotX = robot.getX();
        robotY = robot.getY();
        
        // Clear out current log entry
        Object[] botArray = (livingRobots.values().toArray());
        for (int botIndex = 0; botIndex<botArray.length; botIndex++) {
            String botName = (String)botArray[botIndex];
            get(botName,currentTime).hit = 0;
            get(botName,currentTime).recharge = 0;
        }
        
        // Log hit bullets
        // This is used to help detect if the bot has shot.
        Vector shots = robot.getBulletHitEvents();
        for (int i=0;i<shots.size();i++){
            BulletHitEvent shot = (BulletHitEvent)shots.get(i);
            double damage = shot.getBullet().getPower();
            if (damage<1) damage*=4;
            else damage=(damage*6)-2;
            get(shot.getBullet().getVictim(),currentTime).hit = damage;
        }
        // Log hit by bullets
        // This is used to help detect if the bot has shot.
        Vector hits = robot.getHitByBulletEvents();
        for (int i=0;i<hits.size();i++){
            HitByBulletEvent shot = (HitByBulletEvent)hits.get(i);
            double recharge = shot.getBullet().getPower()*3;
            get(shot.getName(),currentTime).recharge = recharge;
        }
        
        // Log hit robots
        // Scanned events will overwrite hit events because they are more accurate.
        Vector bots = robot.getHitRobotEvents();
        for (int i=0;i<bots.size();i++){
            HitRobotEvent bot = (HitRobotEvent)bots.get(i);
            String name = bot.getName();
            get(name,currentTime).x = getBotX(bot);
            get(name,currentTime).y = getBotY(bot);
            get(name,currentTime).angle = 0;
            get(name,currentTime).v = 0;
            get(name,currentTime).life = bot.getEnergy();
            setLastUpdateTime(name,currentTime);
        }
        
        
        // Log scanned robots
        bots = robot.getScannedRobotEvents();
        for (int i=0;i<bots.size();i++){
            ScannedRobotEvent bot = (ScannedRobotEvent)bots.get(i);
            String name = bot.getName();
            enemyDistance = bot.getDistance();
            get(name, currentTime).time = currentTime;
            get(name, currentTime).x = getBotX(bot);
            get(name, currentTime).y = getBotY(bot);
            get(name, currentTime).angle = bot.getHeadingRadians();
            get(name, currentTime).v = bot.getVelocity();
            get(name, currentTime).life = bot.getEnergy();
            get(name, currentTime).shot = Math.max(0,
                (get(name, currentTime-1).life -
                get(name, currentTime).life) -
                get(name, currentTime).hit-0.0001);
            setLastUpdateTime(name,currentTime);
            
            // If we see it it must be alive
            livingRobots.put(name, bot.getName());
        }
        
        bots = robot.getRobotDeathEvents();
        for (int i=0;i<bots.size();i++){
            RobotDeathEvent bot = (RobotDeathEvent)bots.get(i);
            
            // Remove the dead ones
            livingRobots.remove(bot.getName());
        }
        // If the round is over, everyone will be restored
        // I could remove them so that no ghost logs are accessed, but for now
        // dont remove anyone so that the name list is more complete:
        // Until everything is scanned, having a full name list is better then
        // having an accruate location list.  Will this lead to wolverine
        // shooting at ghosts??
    }
    
    public String [] getLivingBotNames() {
        String [] names = new String[livingRobots.size()];
        Object[] botArray = livingRobots.values().toArray();
        
        for (int i=0;i<livingRobots.size();i++) {
            names[i] = (String) botArray[i];
        }
        return names;
    }
    
    public RobotModel [] getLivingBots() {
        RobotModel [] b = new RobotModel[livingRobots.size()];
        Object[] botArray = livingRobots.values().toArray();
        
        for (int i=0;i<livingRobots.size();i++) {
            b[i] = get((String)botArray[i]);
        }
        return b;
    }
    
    public boolean isEveryoneScanned() {
        return robot.getOthers() == livingRobots.size();
    }
    
    
    public boolean isBotRecorded(String botName, int time) {
        return get(botName,time).time == time;
    }
    
    public RobotModel get(String botName) {
        return get(botName, getLastUpdateTime(botName));
    }
    
    public RobotModel get(String botName, int time) {
        RobotModel[] log = (RobotModel[]) logs.get(botName);
        if (logs.get(botName)==null) {
            // This is the first time this bot has been spotted
            log = new RobotModel[logLength]; {
                for(int i=0;i<log.length;i++){
                    log[i] = new RobotModel();
                    log[i].v = 99999;
                }
            }
            logs.put(botName, log);
        }
        while (time >= log.length) time-=log.length;
        while (time < 0) time+=log.length;
        return log[time];
    }
    
    public int getLastUpdateTime(String botName) {
        Integer lastUpdateTime = (Integer) lastUpdateTimes.get(botName);
        if (lastUpdateTime!=null)
            return lastUpdateTime.intValue();
        else
            return 0;
    }
    
    public static void setLastUpdateTime(String botName, int time) {
        lastUpdateTimes.put(botName, new Integer(time));
    }
    
    public double getBotX(HitRobotEvent e) {
        double dist = 40;
        double angle = robotHeadingRadians + e.getBearingRadians();
        return robotX + Math.sin(angle)*dist;
    }
    public double getBotY(HitRobotEvent e) {
        double dist = 40;
        double angle = robotHeadingRadians + e.getBearingRadians();
        return robotY + Math.cos(angle)*dist;
    }
    public double getBotX(ScannedRobotEvent e) {
        double dist = enemyDistance;
        double angle = robotHeadingRadians + e.getBearingRadians();
        return robotX + Math.sin(angle)*dist;
    }
    public double getBotY(ScannedRobotEvent e) {
        double dist = enemyDistance;
        double angle = robotHeadingRadians + e.getBearingRadians();
        return robotY + Math.cos(angle)*dist;
    }
    double getBotAV(String botName, int time){
        return angleDiff(get(botName,time).angle,get(botName,time-1).angle);
    }
    public static double angleDiff(double a1, double a2) {
        return normalizeAngle(a1-a2);
    }
    public static double normalizeAngle(double d) {
        while (d>Math.PI) d-=Math.PI*2;
        while (d<-Math.PI) d+=Math.PI*2;
        return d;
    }
}
