/*
 * Created on 2004-9-26
 */
package tide.pear;


import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;

import tide.BulletHitBulletEvent;
import tide.EventListener;
import tide.HitByBulletEvent;
import tide.RobotConsole;
import tide.ScannedRobotEvent;
import tide.pear.statistic.MyStatistManager2;
import tide.util.BotInfo;
import tide.util.BotInfoManager;
import tide.util.MathUtils;
import tide.util.Statist;
import tide.util.StatistUtils;
import tide.util.Wave;
import tide.util.WaveManager;

/**
 * 1v1 statist enemy fire analyzer
 * @author iiley(Chen Jing)
 */
public class StatistEnemyFireAnalyzer extends EventListener implements BulletAnalyzer {
	
	private BotInfoManager infoManager;
	private MyStatistManager2 statistManager;
	private WaveManager    waveManager;
	private List           bullets;
	
	private String enemyName;
	private double lastEnemyFirePower = 2d;
	private double lastEnemyFireTime  = 0d;
	private int    bulletIndex, bearingIndex, lastVIndex, wallIndex, distIndex, velocityIndex, moveTimeIndex, accIndex;
	
	public static final int MOVETIME_GAP = 10;
	private static final int FACTORS = 31;
	private int moveTime;	
	
	/**
	 * @param console
	 */
	public StatistEnemyFireAnalyzer(RobotConsole console) {
		super(console);
		this.infoManager = Pear.getBotInfoRecorder().getBotInfoManager();
		statistManager = new MyStatistManager2(FACTORS);
		waveManager = new WaveManager();
		bullets = new ArrayList();
	}
	
	/* (non-Javadoc)
	 * @see tide.EventListener#onWork()
	 */
	public void onWork() {
		Point2D myPos = new Point2D.Double(console.getX(), console.getY());
		waveManager.test(console.getTime(), myPos);
		
		BotInfo enemyFireTimeInfo = infoManager.getInfoMustHaveOne(enemyName, 
				console.getTime() - 2);
		BotInfo myInfoOfEnemyFireTime = infoManager.getInfoMustHaveOne(console.getName()+enemyName, 
				console.getTime() - 2);
		if(enemyFireTimeInfo == null || myInfoOfEnemyFireTime == null){
			//return;
		}
		double enemyFiredPower = infoManager.firedPower(enemyName, console.getTime() - 1);
		if(enemyFiredPower > 0d){
			lastEnemyFirePower = enemyFiredPower;
		}
		
		updateIndexs();
		
		Statist statist = statistManager.getStatists(
				bulletIndex,
				distIndex, 
				velocityIndex, 
				bearingIndex, 
				wallIndex, 
				moveTimeIndex, accIndex, enemyFiredPower > 0d);
		
		Wave wave = new Wave(statist, 
				enemyFireTimeInfo.getLocation(), 
				lastEnemyFirePower, 
				myInfoOfEnemyFireTime.getDirection(),
				myInfoOfEnemyFireTime.getAbsBearing(), 
				myInfoOfEnemyFireTime.getTime());
		if(enemyFiredPower > 0d){
			lastEnemyFireTime = myInfoOfEnemyFireTime.getTime();
			wave.setRateTimes(2);
			wave.setRealBullet(true);
		}
		
		//if enemy do not fire
		if((console.getTime() - lastEnemyFireTime) < 35){
			waveManager.add(wave);
		}
	}
	
	/* (non-Javadoc)
	 * @see tide.BulletAnalyzer#getBullets()
	 */
	public List getBullets() {
		return bullets;
	}
	
	public List getWaves(){
		return waveManager.realWaves();
	}
	
	public void onRoundBegin() {
		bullets.clear();
		waveManager.clear();
	}
	
	public void onScannedRobot(ScannedRobotEvent event) {
		if(enemyName == null){
			enemyName = event.getName();
		}
	}
    public void onHitByBullet(HitByBulletEvent e){
    	//correct the wave's statist
    	Point2D hitedPos = new Point2D.Double(e.getX(), e.getY());
		Wave wave = waveManager.getHitedWave(e.getTime(), hitedPos);
		if(wave != null){
			wave.rateHit(hitedPos);
		}
	}
    
    
	/* (non-Javadoc)
	 * @see tide.EventListener#onBulletHitBullet(tide.BulletHitBulletEvent)
	 */
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		//correct the wave's statist
    	Point2D hittedPos = new Point2D.Double(e.getX(), e.getY());
		Wave wave = waveManager.getHitedWave(e.getTime(), hittedPos);
		if(wave != null){
			wave.rateHit(hittedPos);
		}
	}
	//-----------------------------------------------------------

	public double distanceToWall(Point2D p){
		return Math.min(Math.min(p.getX(), console.getMapWidth()-p.getX()),
				Math.min(p.getY(), console.getMapHeight()-p.getY()));
	}
	
	/**
	 * update all statist buffer's indexs
	 */
	private void updateIndexs() {
		BotInfo myInfoOfEnemyFireTime = infoManager.getInfoMustHaveOne(console.getName()+enemyName, 
				console.getTime() - 2);
		BotInfo enemyOfEnemyFireTime = infoManager.getInfoMustHaveOne(enemyName, 
				console.getTime() - 2);
		BotInfo myInfoBeforeEnemyFireTime = infoManager.getInfoMustHaveOne(console.getName()+enemyName, 
				console.getTime() - 3);

		double moveHeading = myInfoOfEnemyFireTime.getVelocity() >= 0 ? myInfoOfEnemyFireTime.getHeading() :
			MathUtils.anglePI(myInfoOfEnemyFireTime.getHeading() + Math.PI);
		bulletIndex = getBulletIndex(lastEnemyFirePower);
		bearingIndex = getBearingIndex(myInfoOfEnemyFireTime.getAbsBearing() + Math.PI/2d - moveHeading);
		lastVIndex = getVelocityIndex(myInfoBeforeEnemyFireTime.getVelocity());
		wallIndex = getWallIndex(enemyOfEnemyFireTime, myInfoOfEnemyFireTime);
		distIndex = getDistIndex(myInfoOfEnemyFireTime.getDistance());
		velocityIndex = getVelocityIndex(myInfoOfEnemyFireTime.getVelocity());
		
		//update moveTime index
		double acc = Math.round(Math.abs(myInfoOfEnemyFireTime.getVelocity())-Math.abs(myInfoBeforeEnemyFireTime.getVelocity()));
		int div = (int)Math.ceil((double)MOVETIME_GAP*(myInfoOfEnemyFireTime.getDistance()/400));
		if (acc < -0.5d){
			moveTime = div;
		}else if(acc > 0.5d){
			moveTime = 0;//div*3 - 1;
		//}else if(myInfoOfEnemyFireTime.getVelocity() == 0d){
			//moveTime = 0;
		}else{
			if(moveTime < div*3 - 1){
				moveTime = div*3 - 1;
			}
			moveTime++;
		}
		moveTimeIndex = moveTime / div;
		if(moveTimeIndex > statistManager.MOVETIME_D - 1){
			moveTimeIndex = statistManager.MOVETIME_D-1;
		}
		accIndex = Math.min(moveTimeIndex, 2);
	}
    
	public int getBulletIndex(double power){
		return getSegIndex(power, statistManager.BULLET_SEG);
	}
	
	public static int getSegIndex(double value, double[] segs){
		int index = 0;
		for(;index < segs.length; index++){
			if(value<segs[index]){
				break;
			}
		}
		return index;
	}
	
	public int getDistIndex(double dist){
		/*int index = 0;
		if(dist > 240){
			index = 1 + ((int)dist - 240)/230;
		}
		if(index > statistManager.DIST_D - 1){
			index = statistManager.DIST_D - 1;
		}
		return index;
		*/
		return getSegIndex(dist, statistManager.DIST_SEG);
	}
	
	public int getBearingIndex(double moveBearing){
		double b = MathUtils.anglePI(moveBearing);
		if(b>Math.PI/2d || b<-Math.PI/2d){
			b = MathUtils.anglePI(Math.PI - b);
		}
		b = Math.toDegrees(b);
		double sign = b<0d ? -1: 1;
		b = Math.abs(b);
		int index = 0;
		if(b < 10){ 
			index = 0;
		}/*else if(b < 25){
			index = 1;
		}*/else{
			index = 1;
		}
		index *= sign;
		index += 1;
		return index;
	}
	
	public int getWallIndex(BotInfo info, BotInfo targetInfo){
		int index = 0;
		double step = StatistUtils.maxEscapAngle(lastEnemyFirePower)/statistManager.WALL_D;
		do{
			double distToWall = distanceToWall(MathUtils.nextPoint(
					info.getLocation(),
					targetInfo.getAbsBearing() + targetInfo.getDirection()*(index+1)*step,
					targetInfo.getDistance()
					));
			if(distToWall < 18){
				break;
			}else{
				index ++;
			}
		}while(index < statistManager.WALL_D);
		return Math.min(index, statistManager.WALL_D -1);
	}
	
	public int getWallIndex(Point2D pos, BotInfo info){
		double moveDir = info.getVelocity() >= 0 ? info.getHeading() :
			MathUtils.anglePI(info.getHeading() + Math.PI);
		double dist = 100;
		Point2D p1 = MathUtils.nextPoint(pos, moveDir, dist);
		Point2D p2 = MathUtils.nextPoint(pos, moveDir, -dist);
		double p1DistToWall = distanceToWall(p1);
		double p2DistToWall = distanceToWall(p2);
		if(p1DistToWall<0 && p2DistToWall<0){
			return 0;
		}else if(p1DistToWall>0 && p2DistToWall<0){
			return 1;
		}else if(p1DistToWall<0 && p2DistToWall>0){
			return 2;
		}else{
			p1 = MathUtils.nextPoint(pos, moveDir, dist*2);
			p2 = MathUtils.nextPoint(pos, moveDir, -dist*2);
			p1DistToWall = distanceToWall(p1);
			p2DistToWall = distanceToWall(p2);
			if(p1DistToWall<0 && p2DistToWall<0){
				return 3;
			}else if(p1DistToWall>0 && p2DistToWall<0){
				return 4;
			}else if(p1DistToWall<0 && p2DistToWall>0){
				return 5;
			}else{
				return 6;
			}
		}
	}
	
	public int getVelocityIndex(double vel){
		/*double absV = Math.abs(Math.abs(vel) - 0.0);
		int index = 0;
		index = (int)Math.floor(absV/(8d/statistManager.VELOCITY_D));
		if(index > statistManager.VELOCITY_D - 1){
			index = statistManager.VELOCITY_D - 1;
		}
		return index;
		*/
		return getSegIndex(Math.abs(vel), statistManager.VELOCITY_SEG);
	}
}
