package gh.mini;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;
import java.util.*;

/**
 * Gruwel - a minibot by Gert Heijenk (GrubbmGait)
 *
 * This is the result of the mating of one GrubbmOgre and four GrubbmGrunts
 * It has the muscles of its fathers and the broad view of its mother
 * Any child of close relatives has more chance of DNA-defects,
 * in this case it lacks communicative and contactual skills.
 * This bot is reasonable in one-on-one, quite well in melee, and completely unfit for a team
 *
 * Revision information:
 * v0.1 20050325 Initial version. Lovechild of GrubbmOgre 0.4 and GrubbmGrunt 0.6
 * v0.2 20050705 Improved radarhandling
 * v0.3 20051123 Tweaked codesize (-250 bytes)
 *               Adding (simplified) StopNGo to boost one-on-one performance (+375 bytes)
 *				 Keep backup-enemy in melee to prevent selection of harmless enemies (+75 bytes)
 * v0.4 20051201 Use 'slow with turboboost' movement instead of StopNGo, I just like that movement
 * v0.5 20051221 Do not keep backup-enemy
 *               Use preferred distance with trick-movement (slow with turboboost)
 * v0.6 20051230 Bugfixing wallavoidance and 'old' movement by moving some code from mainloop to ScannedRobotEvent
 * v0.7 20060102 Bugfixing trick-movement
 * v0.8 20060109 Bugfixing dive-in protection trick-movement
 * v0.9 20060314 Introducing distance dependent dive-in protection on both movements
 */

public class Gruwel extends AdvancedRobot {

	// movement variables
	private final static double	DIVECONSTANT	= Math.PI / 3600;
	private final static double DEFDIVEANGLE	= 8 * Math.PI / 24;	// 60.0 degrees
	private final static double WALLAVOID		= 35;	// the area to avoid near walls
	private final static double BLINDMANSTICK	= 130;	// the distance when to check wallavoidance
	private final static double DEFDISTANCE		= 420;	// the default distance to keep from enemy
	private static boolean	robothit;					// true if I hit other robot
	private static double	myDirection = 1;			// my current direction
	private static double	myPosX, myPosY;				// my current position
	private static double	myCheckDistance;			// my distance to check for wallavoidance (blindmansstick)
	private static long		currTime;					// the current time (mainloop)
    private static double	shortturn = 1;				// shortturn -1 meaning turning 90 degrees instead of 270

	// scanning and locking variables
	private static boolean	radarScan;				// if true, busy with a radar scan

	// some info about the current locked opponent
	private static String	eName;		// enemy name
	private static double	eAttrac;	// enemy attractivity (less is better)
	private static double	oldHeading;	// the previous heading of the selected opponent

	// specific stuff for StopNGo movement
	private static final double FULLSTOP		= 0.0;	// the absolute minimum speed
	private static final double MINSPEED		= 2.0;	// the minimum speed (sharp turn)
	private static final double MAXSPEED		= 8.0;	// the maximum speed
	private static boolean	stopNGoTrick = true;
	private static double	enemyDistance = DEFDISTANCE;
	private static double	enemyEnergy;
	private static double	enemyPrevEnergy;
	private static double	energyDropCorrection;
	private static double	enemySpeed;
	private static double	enemyPrevSpeed;
	private static double	enemyBearing;
	private static double	perpAngle;

	// some general info
	private static double	fieldWidth;		// the width of the battlefield
	private static double	fieldHeight;	// the heigth of the battlefield
    private static RoundRectangle2D.Double playField; // the playable battlefield
	private static Rectangle2D.Double fireField;	// the shootable battlefield

	//global declarations:
//	static int[] finishes;

	/**
	 * run: Gruwel's default behavior
	 */
	public void run() {
		double  newposx, newposy;	// my future (end)position
		double	powerFired;			// the power the enemy fired at me
		double	adjustAngle;		// the angle to adjust for wallavoidance
		double	distDiff;
		double	myHeading;
		double	myDistance;
		long	defSpeedTime = 0;		// the time when I have to be on default speed again

//		if (finishes == null)
//			finishes = new int[getOthers()+1];

		// Give the robot an appealing look
		setColors( Color.red.darker(), Color.red, Color.red.brighter());

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        playField = new RoundRectangle2D.Double( WALLAVOID, WALLAVOID,
	    			fieldWidth - 2 * WALLAVOID, fieldHeight - 2 * WALLAVOID, 50, 50);
        fireField = new Rectangle2D.Double( 17, 17, fieldWidth - 34, fieldHeight - 34);

		// if melee, disable StopNGo
		if (getOthers() > 1) {
			stopNGoTrick = false;
		}

		// Start with a very unattractive target
		eAttrac = Double.POSITIVE_INFINITY;

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);
		setTurnRadarRight( Double.POSITIVE_INFINITY);	// start scanning

		// mainloop
		while( true ) {
			currTime = getTime();
//			if (stopNGoTrick == true)
//				out.println(currTime+" "+getVelocity());
			// Check if nearly at end of movement
			myDistance = Math.abs( getDistanceRemaining());
			if (myDistance < 1) {
				if (stopNGoTrick == false) {
					myDistance = (Math.sin( currTime/11) * Math.cos( currTime/29))*270 + (Math.random()*100 - 50);
					myDirection = (myDistance > 0 ? 1 : -1);
					myDistance = Math.abs(myDistance);
					if (robothit == true) {
						myDirection = -myDirection;
						robothit = false;
					}
				}
				else {
					myDistance = Double.POSITIVE_INFINITY;
					myDirection = -myDirection;
				}
			}

			// Now keep a preferred distance, stay at approx DEFDISTANCE.
			if (stopNGoTrick == true) {
				distDiff = enemyDistance - DEFDISTANCE;
//			if (Math.abs( distDiff) > 5) {
				adjustAngle = shortturn * (distDiff < 0 ? (Math.sqrt(Math.abs( distDiff)) / 25) : -(Math.sqrt(Math.abs( distDiff)) / 60));
//					if (distDiff < 0) {
//						adjustAngle = (shortturn * (Math.sqrt(Math.abs( distDiff)) / 25));
//						adjustAngle = (shortturn * 0.4);
//					}
//					else {
//						adjustAngle = -(shortturn * (Math.sqrt(Math.abs( distDiff)) / 60));
//						adjustAngle = (shortturn * -0.4);
//					}
				setTurnRightRadians( perpAngle + adjustAngle);
			}

			// perform some wall-avoidance, try to slide away if moving near wall
			myHeading = getHeadingRadians();
			myCheckDistance = Math.min( BLINDMANSTICK, myDistance) * myDirection;
			newposx = (myPosX = getX()) + Math.sin( myHeading) * myCheckDistance;	// calculate end-of-stick position
			newposy = (myPosY = getY()) + Math.cos( myHeading) * myCheckDistance;
			// if endposition not in field, adapt
			if (!playField.contains( newposx, newposy)) {
				adjustAngle = Math.atan( Math.tan( calcAngleToWall( myHeading)));
				// if adaption is too big relative to perpendicular (dive-in protection), just bounce
//				if ((stopNGoTrick == true) && (getVelocity() <= MINSPEED)) {
//					if (Math.PI / 2.7 < Math.abs(adjustAngle - perpAngle)) {
//						myDirection = -myDirection;			// switch direction
//						adjustAngle = 0;
//					}
//				}
				setTurnRightRadians( adjustAngle);
			}

			// perform some dive-in protection here
			adjustAngle = DEFDIVEANGLE - (enemyDistance * DIVECONSTANT);	// maxDiveAngle
//			out.println("mDA: "+Math.toDegrees(adjustAngle));
			if (((Math.abs(enemyBearing - getTurnRemainingRadians()) < adjustAngle) && (myDirection == 1)) ||
				((Math.abs(enemyBearing - getTurnRemainingRadians()) > (Math.PI - adjustAngle)) && (myDirection == -1))) {				
					myDirection = -myDirection;
					setTurnRightRadians( perpAngle);
			}

			if (stopNGoTrick == true) {
				powerFired = doCheckEnemyFired();

				// Minimumspeed movement with turboboost when enemy fired
				// This is the opposite of the handbrake, my first idea
				// I like it better than StopNGo, but it may not be as successfull
				if (powerFired > 0.0) {
					defSpeedTime = currTime + 10 + (long)(powerFired * 2) - 4;
					setMaxVelocity( MAXSPEED);
				}
				// check if I have to go back to default MinimumSpeed again
				if (defSpeedTime - currTime <= 0) {
					setMaxVelocity( MINSPEED);
				}
			}
			// set the distance and direction
			setAhead( myDistance * myDirection);

			// perform all actions previously set
			execute();
		}
	}

	/**
	 * calcAngleToWall: Calculate the wall'smoothing' (or something that looks like it)
	 */
	public double calcAngleToWall( double myhead) {
		double di;

		for (di = 0; di < Math.PI; di += Math.PI/36) {
			if (checkInPlayField( myhead + di) == true) {
				return( di);
			}
			if (checkInPlayField( myhead - di) == true) {
				return( -di);
			}
		}
		return ( di);
	}

	/**
	 * checkInPlayField: Check if my future position is in the playField
	 */
	public boolean checkInPlayField( double thead) {
		return (playField.contains( myPosX + Math.sin( thead) * myCheckDistance, myPosY + Math.cos( thead) * myCheckDistance));
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

		double tmpBearing = getHeadingRadians() + (enemyBearing = e.getBearingRadians());
		double power;
		double battrac;
		double dist = e.getDistance();
		double myX = getX();
		double myY = getY();

		// calculate 'attractivity' of this enemy
		battrac = (e.getEnergy() < 20 ? dist * 0.75 : dist);
		if (!(e.getName().equals( eName))) {
			if ((battrac < (eAttrac * 0.8)) && (eAttrac > 80) && (dist < 1200)) {
				eName = e.getName();	// new more attractive enemy
			}
			else {
				return;		// new not very appealing enemy
			}
		}
		// target approved, set attractivity
		eAttrac = battrac;

		// keep some information for the StopNGo movement
		enemyPrevSpeed = enemySpeed;
		enemySpeed = e.getVelocity();
		enemyEnergy = e.getEnergy();
		enemyDistance = dist;

		// Stay perpendicular to the opponent when in lock-mode
		power = Utils.normalRelativeAngle(e.getBearingRadians() + Math.PI/2);
		setTurnRightRadians( perpAngle = Math.atan( Math.tan( power)));
		shortturn = (power != perpAngle ? -myDirection : myDirection);

		// At start of round, always perform two radarsweeps without locking
		if (currTime < 16) return;

		// set the power of the bullet, including energymanagement
		power = ( dist > 850 ? 0.1 : (dist > 700 ? 0.49 : (dist > 250 ? 1.9 : 3.0)));
//		power = 0.1;
		power = Math.min( getEnergy()/5, Math.min( (e.getEnergy()/4) + 0.2, power));

		// perform lineair and circular targetting
		long deltahittime;
		Point2D.Double point = new Point2D.Double();
		double head, chead, bspeed;
		double tmpx, tmpy;

		// perform an iteration to find a reasonable accurate expected position
		tmpx = myX + Math.sin( tmpBearing) * dist;
		tmpy = myY + Math.cos( tmpBearing) * dist;
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		oldHeading = e.getHeadingRadians();
		point.setLocation( tmpx, tmpy);
		deltahittime = 0;
		do {
			tmpx += Math.sin( head) * e.getVelocity();
			tmpy +=	Math.cos( head) * e.getVelocity();
			head += chead;
			deltahittime++;
			// if position not in field, adapt
			if (!fireField.contains( tmpx, tmpy)) {
				bspeed = point.distance( myX, myY) / deltahittime;
				power = Math.max( Math.min( (20 - bspeed) / 3.0, 3.0), 0.1);
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (point.distance( myX, myY) - 18) / (20 - (3 * power))) > deltahittime);

		// If the gun is too hot or not aligned, don't bother trying to fire.
		tmpBearing = 2.2 * Utils.normalRelativeAngle( tmpBearing - getRadarHeadingRadians());
		if ((getGunHeat() == 0.0) && (getGunTurnRemaining() == 0.0) && (power > 0.0) && (getEnergy() > 0.1) && (radarScan == false)) {
			// Only fire the gun when it is locked and loaded, do a radarsweep afterwards.
			setFire( power);
			setTurnRadarRightRadians( Double.NEGATIVE_INFINITY * tmpBearing);
			radarScan = true;
		}
		else {
			// when not firing, lock on target.
			setTurnRadarRightRadians( tmpBearing);
			radarScan = false;
		}
		// always lock when tricking simple targetters
		if (stopNGoTrick == true) {
			setTurnRadarRightRadians( tmpBearing);
		}
		// Turn gun after eventual firing
		setTurnGunRightRadians( Utils.normalRelativeAngle(((Math.PI / 2) - Math.atan2( point.y - myY, point.x - myX)) - getGunHeadingRadians()));
	}

	/**
	 * doCheckEnemyFired: Check if the enemy has fired a bullet. If so, remember it
	 */
	double doCheckEnemyFired() {

		double bpower = 0.0;
		if (enemyEnergy != enemyPrevEnergy) {
			if ((enemySpeed == 0) && (Math.abs( enemySpeed - enemyPrevSpeed) > 2)) {
				energyDropCorrection += ((enemyPrevSpeed / 2) - 1);
//				out.println(currTime+" Enemy hit the wall");
			}
			bpower = enemyPrevEnergy - (enemyEnergy + energyDropCorrection);
			if ((bpower < 0.09) || (bpower > 3.0)) bpower = 0.0;
		}
		energyDropCorrection = 0.0;
		enemyPrevEnergy = enemyEnergy;
		return bpower;
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
	public void onRobotDeath(RobotDeathEvent e) {
		// If current target dies, force to find another
		if (eName.equals( e.getName())) {
			radarScan = true;
			setTurnRadarRightRadians( eAttrac = Double.POSITIVE_INFINITY);
		}
	}

	/**
	 * onHitRobot: Bounce off !
	 */	
	public void onHitRobot(HitRobotEvent e) {
		// if I hit someone, just bounce off, ramming is not a good idea when in team or melee.
		if (e.isMyFault()) {
			robothit = true;
		}
	}

	/**
	 * onHitWall: Do nothing !
	 */	
//	public void onHitWall(HitWallEvent e) {
//		// if I hit someone, just bounce off, ramming is not a good idea when in team or melee.
//		wallhits++;
//	}

	/**
	 * onBulletHit: Yes, I hit somebody
	 */
	public void onBulletHit(BulletHitEvent e)
	{
		energyDropCorrection += (4 * e.getBullet().getPower() + 2 * Math.max( e.getBullet().getPower() - 1, 0));
	}

	/**
	 * onHitByBullet:  I am hit, handle some (movement) statistics.
	 */
	public void onHitByBullet(HitByBulletEvent e)
	{
		energyDropCorrection -= (3 * e.getPower());
	}

	/**
	 * onWin: Show my private victory dance
	 */
//	public void onWin(WinEvent e)
//	{
//		//Victory dance	
//		setTurnGunRight( Double.POSITIVE_INFINITY);
//		setTurnRadarLeft( Double.POSITIVE_INFINITY);
//		setTurnRight(10);
//		ahead(20);
//		printStats();
//		waitFor(new RadarTurnCompleteCondition(this));
//	}

	/**
	 * onDeath: Show some statistics
	 */
	public void onDeath(DeathEvent e)
	{
		// Disable stopNGoTrick if I die fast	
		if (getRoundNum() < 4) stopNGoTrick = false;
//		printStats();
	}

	/**
	 * printStats: Print statistics on end of round
	 */
//	public void printStats( )
//	{
//		finishes[getOthers()]++;
//		for (int i=0; i<finishes.length; i++)
//			out.print(finishes[i] + " ");
//		out.println();
//		out.println("Wallhits: "+wallhits);
//		out.println("Robothits:"+robothits);
//	}
}

