/*
 * Copyright (C) 2012 Vadym Timofeyev
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Vadym Timofeyev - initial implementation
 */
package tvv.micro;

import java.awt.Color;
import java.awt.geom.Rectangle2D;

import robocode.AdvancedRobot;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;

/**
 * Robot name: Antares.
 *
 * @author Vadym Timofeyev
 * @since JDK 1.5
 * @version 1.1.1
 */
public final class Antares extends AdvancedRobot {
    /**
     * Antares star, color index: B−V 1.83, spectral type: M1.5Iab-b
     * RGB color: 0xFFCA8A
     *
     * @see <a aref="http://en.wikipedia.org/wiki/Antares">Wiki</a>
     * @see <a href="http://vendian.org/mncharity/dir3/starcolor/">StarColor</a>
     */
    private static final int ANTARES_RGB = 0xFFCA8A;

    /**
     * Constants.
     */
    private static final double MAX_FIRE_DISTANCE      = 400.0;
    private static final double MIN_MOVE_DISTANCE      = 120.0;
    private static final double MOVE_DISTANCE          = 180.0;
    private static final double NEAR_WALL_DISTANCE     = 50.0;
    private static final double NEAR_WALL_DISTANCE_DBL = 150.0;

    /**
     * Predefined calculations.
     */
    private static final double PI_DIV_2    = StrictMath.PI / 2.0;
    private static final double PI_DIV_8    = StrictMath.PI / 8.0;
    private static final double PI_DIV_12   = StrictMath.PI / 12.0;
    private static final double PI_DIV_1920 = StrictMath.PI / 1920.0;

    /**
     * Movement distance and direction.
     */
    private double m_direction;

    /**
     * Battlefield without spaces near walls.
     */
    private Rectangle2D.Double m_internalBattlefield;

    /**
     * The main method. Overridden to set up robot's basic behavior.
     */
    @Override
    public void run() {
        m_direction = MOVE_DISTANCE;

        m_internalBattlefield = new Rectangle2D.Double(
            NEAR_WALL_DISTANCE
          , NEAR_WALL_DISTANCE
          , getBattleFieldWidth() - NEAR_WALL_DISTANCE_DBL
          , getBattleFieldHeight() - NEAR_WALL_DISTANCE_DBL
        );

        setAllColors(new Color(ANTARES_RGB));

        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
        setAdjustRadarForRobotTurn(true);

        setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
    }

    /**
     * This method is called when robot sees an another robot, i.e. when the
     * robot's radar scan "hits" another robot.
     *
     * @param event the scanned-robot event set by the game
     */
    @Override
    public void onScannedRobot(final ScannedRobotEvent event) {
        setTurnRadarLeftRadians(getRadarTurnRemainingRadians());
        manageGun(event);
        manageFire(event);
        manageMovement(event);

        clearAllEvents();
    }

    /**
     * Manages shooting and energy.
     *
     * Fire only if:
     * - the robot has enough energy
     * - distance less than N
     * - enemy is not disabled or it is not a duel
     * - gun is not heat
     * - the gun is not turning
     *
     * @param event the scanned-robot event set by the game
     */
    private void manageFire(final ScannedRobotEvent event) {
        if (getEnergy() > Rules.MIN_BULLET_POWER
         && event.getDistance() < MAX_FIRE_DISTANCE
         && (event.getEnergy() > 0.0 || getOthers() > 1)
         && Utils.isNear(getGunHeat(), 0.0)
         && Math.abs(getGunTurnRemainingRadians()) < PI_DIV_12) {
              setFire(
                  StrictMath.max(
                      StrictMath.min(
                          StrictMath.min(
                              MAX_FIRE_DISTANCE / event.getDistance()
                            , Rules.MAX_BULLET_POWER)
                        , StrictMath.min(
                              getEnergy()
                            , event.getEnergy() / 4.0 + Rules.MIN_BULLET_POWER)
                      )
                    , Rules.MIN_BULLET_POWER
                  )
              );
        }
    }

    /**
     * Manages gun.
     *
     * @param event the scanned-robot event set by the game
     */
    private void manageGun(final ScannedRobotEvent event) {
        final double bearing  = getHeadingRadians() + event.getBearingRadians();

        setTurnGunRightRadians(Utils.normalRelativeAngle(
            bearing - getGunHeadingRadians()
          + StrictMath.signum(event.getEnergy())
          * (StrictMath.random() - 0.5) * PI_DIV_8
          + StrictMath.sin(event.getHeadingRadians() - bearing)
          * event.getVelocity() / 16.0
        ));
    }

    /**
     * Manages robot turning and movement.
     *
     * @param event the scanned-robot event set by the game
     */
    private void manageMovement(final ScannedRobotEvent event) {
        if (!m_internalBattlefield.contains(getX(), getY())) {
            moveWallsMode();
        } else if (Utils.isNear(event.getEnergy(), 0.0)) {
            moveRamMode(event);
        } else {
            moveMeleeMode(event);
        }
    }

    /**
     * Moves in "melee mode".
     *
     * @param event the scanned-robot event set by the game
     */
    private void moveMeleeMode(final ScannedRobotEvent event) {
        setTurnRightRadians(Utils.normalRelativeAngle(
            event.getBearingRadians() + PI_DIV_2 - m_direction * PI_DIV_1920
        ));

        if (Utils.isNear(getVelocity(), 0.0)) {
            setAhead(m_direction = -m_direction);
        }
    }

    /**
     * Moves in "ram mode".
     *
     * @param event the scanned-robot event set by the game
     */
    private void moveRamMode(final ScannedRobotEvent event) {
        if (Utils.isNear(getVelocity(), 0.0)) {
            setTurnRightRadians(event.getBearingRadians());
            setAhead(event.getDistance());
        }
    }

    /**
     * Moves in "walls-avoidance mode".
     */
    private void moveWallsMode() {
        final double angle = Utils.normalRelativeAngle(
            StrictMath.atan2(m_internalBattlefield.getCenterX() - getX()
                           , m_internalBattlefield.getCenterY() - getY()
            ) - getHeadingRadians()
        );

        setTurnRightRadians(angle);

        setAhead(MIN_MOVE_DISTANCE
              * (StrictMath.abs(angle) > PI_DIV_2 ? -1.0 : 1.0));
    }
}
