package ar.horizon.util;

import static java.lang.Math.*;

import java.awt.geom.*;
import java.util.*;

/**
 * Utility class, most of it copied from robocode.Rules and robocode.util.Utils
 * to allow for older versions of Robocode.
 * 
 * @author Aaron Rotenberg
 */
public final class Util {
	private Util() {
	}

	// robocode.Rules begins here.

	public static final double ACCELERATION = 1;
	public static final double DECELERATION = 2;
	public static final double MAX_VELOCITY = 8;
	public static final double RADAR_SCAN_RADIUS = 1200;
	public static final double MIN_BULLET_POWER = 0.1;
	public static final double MAX_BULLET_POWER = 3;
	public static final double MAX_TURN_RATE = 10;
	public static final double MAX_TURN_RATE_RADIANS = toRadians(MAX_TURN_RATE);
	public static final double GUN_TURN_RATE = 20;
	public static final double GUN_TURN_RATE_RADIANS = toRadians(GUN_TURN_RATE);
	public static final double RADAR_TURN_RATE = 45;
	public static final double RADAR_TURN_RATE_RADIANS =
			toRadians(RADAR_TURN_RATE);
	public static final double ROBOT_HIT_DAMAGE = 0.6;
	public static final double ROBOT_HIT_BONUS = 1.2;

	public static double getTurnRate(double velocity) {
		return MAX_TURN_RATE - 0.75 * velocity;
	}

	public static double getTurnRateRadians(double velocity) {
		return toRadians(getTurnRate(velocity));
	}

	public static double getWallHitDamage(double velocity) {
		return max(abs(velocity) / 2 - 1, 0);
	}

	public static double getBulletDamage(double bulletPower) {
		double damage = 4 * bulletPower;

		if (bulletPower > 1) {
			damage += 2 * (bulletPower - 1);
		}
		return damage;
	}

	public static double getBulletHitBonus(double bulletPower) {
		return 3 * bulletPower;
	}

	public static double getBulletSpeed(double bulletPower) {
		return 20 - 3 * bulletPower;
	}

	public static double getGunHeat(double bulletPower) {
		return 1 + (bulletPower / 5);
	}

	// robocode.Rules ends here.

	// robocode.util.Utils begins here.

	public static final double TWO_PI = 2 * PI;
	public static final double THREE_PI_OVER_TWO = 3 * PI / 2;
	public static final double PI_OVER_TWO = PI / 2;

	public static double normalAbsoluteAngle(double angle) {
		return (angle %= TWO_PI) >= 0 ? angle : (angle + TWO_PI);
	}

	public static double normalRelativeAngle(double angle) {
		return (angle %= TWO_PI) >= 0 ? (angle < PI) ? angle : angle - TWO_PI
				: (angle >= -PI) ? angle : angle + TWO_PI;
	}

	public static double normalNearAbsoluteAngle(double angle) {
		angle = (angle %= TWO_PI) >= 0 ? angle : (angle + TWO_PI);

		if (isNear(angle, PI)) {
			return PI;
		} else if (angle < PI) {
			if (isNear(angle, 0)) {
				return 0;
			} else if (isNear(angle, PI_OVER_TWO)) {
				return PI_OVER_TWO;
			}
		} else {
			if (isNear(angle, THREE_PI_OVER_TWO)) {
				return THREE_PI_OVER_TWO;
			} else if (isNear(angle, TWO_PI)) {
				return 0;
			}
		}
		return angle;
	}

	public static boolean isNear(double angle1, double angle2) {
		return (abs(angle1 - angle2) < .00001);
	}

	// robocode.util.Utils ends here.

	public static final double ROBOT_WIDTH = 36.0;
	public static final double ROBOT_CENTER = ROBOT_WIDTH / 2;
	public static final double GUN_HEAT_AT_START_OF_ROUND = 3.0;

	public static Point2D.Double project(Point2D.Double sourceLocation,
			double angle, double distance) {
		return new Point2D.Double(sourceLocation.x + sin(angle) * distance,
				sourceLocation.y + cos(angle) * distance);
	}

	public static double absoluteBearing(Point2D.Double from, Point2D.Double to) {
		return atan2(to.x - from.x, to.y - from.y);
	}

	public static double angleDifference(double angle1, double angle2) {
		return normalRelativeAngle(angle1 - angle2);
	}

	/**
	 * Constrains a value to be within a given range.
	 */
	public static double limit(double min, double value, double max) {
		return max(min, min(value, max));
	}

	public static int sign(double value) {
		return value < 0 ? -1 : 1;
	}

	/**
	 * @return An <em>upper bound</em> on the max escape angle a robot can
	 *         achieve given the supplied bullet velocity. For more precise MEA
	 *         calculations, use precise prediction.
	 */
	public static double maxEscapeAngle(double bulletSpeed) {
		return asin(8.0 / bulletSpeed);
	}

	/**
	 * @see #getBearingOffsetFromGuessFactor(double, int, double, double)
	 */
	public static double getGuessFactorFromBearingOffset(double bearingOffset,
			int lateralDirection, double maxEscapeAngleClockwise,
			double maxEscapeAngleCounterclockwise) {
		double maxEscapeAngle =
				(bearingOffset > 0) ? maxEscapeAngleClockwise
						: maxEscapeAngleCounterclockwise;
		double directedGuessFactor = bearingOffset / maxEscapeAngle;
		double guessFactor = directedGuessFactor * lateralDirection;
		return guessFactor;
	}

	/**
	 * @see #getGuessFactorFromBearingOffset(double, int, double, double)
	 */
	public static double getBearingOffsetFromGuessFactor(double guessFactor,
			int lateralDirection, double maxEscapeAngleClockwise,
			double maxEscapeAngleCounterclockwise) {
		double directedGuessFactor = guessFactor * lateralDirection;
		double maxEscapeAngle =
				(directedGuessFactor > 0) ? maxEscapeAngleClockwise
						: maxEscapeAngleCounterclockwise;
		double bearingOffset = directedGuessFactor * maxEscapeAngle;
		return bearingOffset;
	}

	/**
	 * Fast wall smoothing by Simonton, see <a href=
	 * "http://old.robowiki.net/cgi-bin/robowiki?WallSmoothing/NonIterative">the
	 * old wiki page on noniterative wall smoothing</a>.
	 */
	public static double wallSmoothing(Point2D.Double botLocation,
			double angle, int direction, double wallStick,
			Rectangle2D.Double fieldRectangle) {
		int nonzeroDirection = sign(direction);

		angle =
				smoothWest(fieldRectangle.height - botLocation.y, angle
						- PI_OVER_TWO, nonzeroDirection, wallStick)
						+ PI_OVER_TWO;
		angle =
				smoothWest(fieldRectangle.width - botLocation.x, angle + PI,
						nonzeroDirection, wallStick)
						- PI;
		angle =
				smoothWest(botLocation.y - fieldRectangle.y, angle
						+ PI_OVER_TWO, nonzeroDirection, wallStick)
						- PI_OVER_TWO;
		angle =
				smoothWest(botLocation.x - fieldRectangle.x, angle,
						nonzeroDirection, wallStick);
		angle =
				smoothWest(botLocation.y - fieldRectangle.y, angle
						+ PI_OVER_TWO, nonzeroDirection, wallStick)
						- PI_OVER_TWO;
		angle =
				smoothWest(fieldRectangle.width - botLocation.x, angle + PI,
						nonzeroDirection, wallStick)
						- PI;
		angle =
				smoothWest(fieldRectangle.height - botLocation.y, angle
						- PI_OVER_TWO, nonzeroDirection, wallStick)
						+ PI_OVER_TWO;

		return angle;
	}

	private static double smoothWest(double distance, double angle,
			int orientation, double wallStick) {
		if (distance < -wallStick * sin(angle)) {
			return acos(orientation * distance / wallStick)
					- (orientation * PI_OVER_TWO);
		}

		return angle;
	}

	/**
	 * Fast wall distance by Simonton, see <a
	 * href="http://old.robowiki.net/cgi-bin/robowiki?WallDistance">the old wiki
	 * page on wall distance</a>.
	 */
	public static double getWallDistance(RobotRecording entry, int direction,
			Rectangle2D.Double fieldRectangle) {
		Point2D.Double location = entry.getLocation();
		double distance = entry.getEnemyDistance();
		double absoluteBearing =
				entry.getEnemyRecording().getEnemyAbsoluteBearing();

		return min(min(min(distanceWest(fieldRectangle.height - location.y,
				distance, absoluteBearing - PI_OVER_TWO, direction),
				distanceWest(fieldRectangle.width - location.x, distance,
						absoluteBearing + PI, direction)), distanceWest(
				location.y - fieldRectangle.y, distance, absoluteBearing
						+ PI_OVER_TWO, direction)), distanceWest(location.x
				- fieldRectangle.x, distance, absoluteBearing, direction));
	}

	private static double distanceWest(double toWall, double eDist,
			double eAngle, int oDir) {
		if (eDist <= toWall) {
			return Double.POSITIVE_INFINITY;
		}

		double wallAngle = acos(-oDir * toWall / eDist) + oDir * PI_OVER_TWO;
		return normalAbsoluteAngle(oDir * (wallAngle - eAngle));
	}

	public static interface GoAngleGenerator {
		/**
		 * @return The absolute angle the robot will try to move at, given the
		 *         specified information.
		 */
		double getGoAngle(Point2D.Double myPosition,
				Point2D.Double enemyPosition, int direction);
	}

	/**
	 * Precise prediction, a.k.a. "that really complicated function that nobody
	 * likes to think about too much".
	 * 
	 * @param wave
	 *            The wave that is being predicted against. If a valid wave is
	 *            passed, the function predicts where the bot will be when it
	 *            hits. If <code>null</code> is passed, the function
	 *            auto-predicts a small time into the future.
	 * @param orbitPosition
	 *            The orbit position that is passed to the go angle generator.
	 *            This is independent of the wave, but if a wave is available,
	 *            <code>wave.recording.getEnemyRecording().getLocation()</code>
	 *            is probably what you want to pass for this parameter.
	 */
	public static LinkedList<PredictedState> predictPosition(
			PredictedState startPosition, Wave wave,
			Point2D.Double orbitPosition, int direction,
			GoAngleGenerator goAngleGenerator, double wallStick,
			Rectangle2D.Double fieldRectangle) {
		LinkedList<PredictedState> predictions =
				new LinkedList<PredictedState>();
		boolean waveMode = (wave != null);

		Point2D.Double predictedPosition =
				(Point2D.Double) startPosition.getLocation().clone();
		double predictedVelocity = startPosition.getVelocity();
		double predictedHeading = startPosition.getHeading();
		double maxTurning, moveAngle, moveDir;

		int counter = startPosition.getTicks();
		boolean intercepted = false;
		int maxTicks = (waveMode ? 500 : 10);

		do {
			double goAngle =
					goAngleGenerator.getGoAngle(predictedPosition,
							orbitPosition, direction);
			moveAngle =
					wallSmoothing(predictedPosition, goAngle, direction,
							wallStick, fieldRectangle)
							- predictedHeading;
			moveDir = 1;

			if (cos(moveAngle) < 0) {
				moveAngle += PI;
				moveDir = -1;
			}

			moveAngle = normalRelativeAngle(moveAngle);

			maxTurning = getTurnRateRadians(abs(predictedVelocity));
			predictedHeading =
					normalRelativeAngle(predictedHeading
							+ limit(-maxTurning, moveAngle, maxTurning));

			if (direction == 0) {
				predictedVelocity =
						(predictedVelocity < 0 ? limit(-MAX_VELOCITY,
								predictedVelocity + DECELERATION, 0) : limit(0,
								predictedVelocity - DECELERATION, MAX_VELOCITY));
			} else {
				predictedVelocity +=
						(predictedVelocity * moveDir < 0 ? DECELERATION
								* moveDir : ACCELERATION * moveDir);
				predictedVelocity =
						limit(-MAX_VELOCITY, predictedVelocity, MAX_VELOCITY);
			}

			predictedPosition =
					project(predictedPosition, predictedHeading,
							predictedVelocity);
			predictions.add(new PredictedState(predictedPosition,
					predictedVelocity, predictedHeading, counter));
			counter++;

			if (waveMode) {
				if (predictedPosition.distance(wave.getFirerRecording().getLocation()) < wave.getDistanceTraveled()
						+ (counter + 1) * getBulletSpeed(wave.getBulletPower())) {
					intercepted = true;
				}
			}
		} while (!intercepted && counter < maxTicks);

		return predictions;
	}
}
