package nat.ether.utils;

import java.awt.geom.Point2D;
import java.io.Serializable;
import java.util.ArrayList;

import robocode.util.Utils;
import robocode.*;

public final class Wave implements Serializable, Cloneable {
	private static final long serialVersionUID = 6826380621990735922L;
	public static final int INTERSECTION = 3, PASSED = 1, ACTIVE = 2;

	// Wave state
	public final double bulletVelocity;
	public final double bulletPower;
	public Point2D fireLocation;
	public final double fireHeading, fireVelocity;
	public final Range maxEscapedAngle;
	public final double headOnAngle;
	public final double lateralDirection;
	public final double fireAngle;
	public final long fireTime;
	public final Store scan;

	// Wave status
	public int status;
	public double distanceTraveled;
	public Range hitRange = null;
	public Point2D hitLocation = null;
	public Point2D bulletLocation = null;

	public boolean isHit = false;
	public boolean bulletHitBullet = false;
	public boolean isFiringWave;

	// unrelated information
	public float[] paintBucket;

	// Clone constructor
	private Wave(double distanceTraveled, double bulletVelocity, Point2D fireLocation, double fireHeading, double fireVelocity, Range maxEscapedAngle, double headOnAngle, double lateralDirection, double bulletPower, double fireAngle, long fireTime, int status, boolean isFiringWave, boolean isHit, boolean bulletHitBullet, Range hitRange, Store scan, float[] paintBucket) {
		this.distanceTraveled = distanceTraveled;
		this.bulletVelocity = bulletVelocity;
		this.fireLocation = fireLocation;
		this.fireHeading = fireHeading;
		this.fireVelocity = fireVelocity;
		this.maxEscapedAngle = maxEscapedAngle;
		this.headOnAngle = headOnAngle;
		this.lateralDirection = lateralDirection;
		this.bulletPower = bulletPower;
		this.fireAngle = this.headOnAngle;
		this.fireTime = fireTime;
		this.status = status;
		this.isFiringWave = isFiringWave;
		this.isHit = isHit;
		this.bulletHitBullet = bulletHitBullet;
		this.hitRange = hitRange;
		this.scan = scan;
		this.paintBucket = paintBucket;
	}

	public Wave(double bulletPower, Point2D location, double fireHeading, double fireVelocity, double angle, Store scan, long time) {
		this.distanceTraveled = 0;
		this.bulletVelocity = M.getBulletSpeed(bulletPower);
		this.fireLocation = location;
		this.fireHeading = fireHeading;
		this.fireVelocity = fireVelocity;
		this.headOnAngle = angle;
		this.lateralDirection = scan.dir;
		this.bulletPower = bulletPower;
		this.fireAngle = this.headOnAngle;
		this.fireTime = time;
		this.status = ACTIVE;
		this.isFiringWave = false;
		this.isHit = false;
		this.bulletHitBullet = false;
		this.hitRange = null;
		this.scan = scan;
		this.paintBucket = null;

		this.maxEscapedAngle = getMaximumEscapedAngle();
	}

	public final Wave clone() {
		return new Wave(distanceTraveled, bulletVelocity, fireLocation, fireHeading, fireVelocity, maxEscapedAngle, headOnAngle, lateralDirection, bulletPower, fireAngle, fireTime, status, isFiringWave, isHit, bulletHitBullet, hitRange, scan, paintBucket);
	}

	public final double getRelativeAngle(double angle) {
		return Utils.normalRelativeAngle(angle - this.headOnAngle) * lateralDirection;
	}

	public final double getGuessFactor(double angle) {
		double angleOffset = getRelativeAngle(angle);
		return angleOffset / getMaximumEscapedAngle(angleOffset);
	}

	public final double getGuessFactor(int bin, int middle) {
		return ((double) (bin - middle) / (double) middle);
	}

	public final double getAngle(double GF) {
		return Utils.normalAbsoluteAngle((GF * getMaximumEscapedAngle(GF) * lateralDirection) + headOnAngle);
	}

	public final void advanceWave() {
		distanceTraveled += bulletVelocity;
	}

	public final void update(Point2D enemyLocation) {
		advanceWave();
		status = getStatus(enemyLocation);

		// get precise intersection range
		if (status == INTERSECTION) {
			Range currentIntersection = getCurrentIntersectionRange(enemyLocation);
			if (hitRange == null)
				hitRange = currentIntersection;
			else
				hitRange = hitRange.grow(currentIntersection);
		}

		// get static hit position
		if (distanceTraveled * distanceTraveled > enemyLocation.distanceSq(fireLocation) && hitLocation == null) {
			hitLocation = (Point2D) enemyLocation.clone();
		}
	}

	private final double getMaximumEscapedAngle(double value) {
		return value < 0 ? maxEscapedAngle.lower : maxEscapedAngle.upper;
	}

	private final Range getMaximumEscapedAngle() {
		Range MEA = null;
		double traditionalMEA = Math.asin(Rules.MAX_VELOCITY / bulletVelocity);
		MEA = new Range(traditionalMEA, traditionalMEA);

		return MEA;
	}

	// /////////////////////////////////
	// Precise Wave Intersection code
	// CREDIT: Skilgannon
	private final int getStatus(Point2D botLocation) {
		double[] enemyCorner = new double[] { fireLocation.distanceSq(botLocation.getX() - 18, botLocation.getY() + 18), fireLocation.distanceSq(botLocation.getX() + 18, botLocation.getY() + 18), fireLocation.distanceSq(botLocation.getX() + 18, botLocation.getY() - 18), fireLocation.distanceSq(botLocation.getX() - 18, botLocation.getY() - 18) };

		// score, for speed
		int score = 0;

		// reached
		if (distanceTraveled * distanceTraveled > Math.min(Math.min(enemyCorner[0], enemyCorner[1]), Math.min(enemyCorner[2], enemyCorner[3])))
			score++;

		// passed
		if ((distanceTraveled - bulletVelocity) * (distanceTraveled - bulletVelocity) < Math.max(Math.max(enemyCorner[0], enemyCorner[1]), Math.max(enemyCorner[2], enemyCorner[3])))
			score += 2;

		return score;
	}

	private final Range getCurrentIntersectionRange(Point2D botLocation) {
		double[] robotYBounds = new double[] { botLocation.getY() - 18, botLocation.getY() + 18 };
		double[] robotXBounds = new double[] { botLocation.getX() - 18, botLocation.getX() + 18 };

		double[] radius = new double[] { distanceTraveled, distanceTraveled - bulletVelocity };

		ArrayList<Point2D> intersectPoint2Ds = new ArrayList<Point2D>();

		// loop through each robot bound
		for (int i = 0; i < 2; i++) {
			for (int j = 0; j < 2; j++) {

				// get the vertical intersection point
				Point2D[] vertIntersectPoint2Ds = getVerticalIntersectionPoint2Ds(fireLocation.getX(), fireLocation.getY(), radius[i], robotXBounds[j]);

				for (int k = 0; k < vertIntersectPoint2Ds.length; k++) {
					if (inRange(vertIntersectPoint2Ds[k].getY(), robotYBounds))
						intersectPoint2Ds.add(vertIntersectPoint2Ds[k]);
				}

				// horizontal intersection
				Point2D[] horizIntersectPoint2Ds = getHorizontalIntersectionPoint2Ds(fireLocation.getX(), fireLocation.getY(), radius[i], robotYBounds[j]);

				for (int k = 0; k < horizIntersectPoint2Ds.length; k++) {
					if (inRange(horizIntersectPoint2Ds[k].getX(), robotXBounds))
						intersectPoint2Ds.add(horizIntersectPoint2Ds[k]);
				}

				// corner
				Point2D testCorner = new Point2D.Double(robotXBounds[i], robotYBounds[j]);
				double distSq = testCorner.distanceSq(fireLocation);
				if (distSq <= radius[0] * radius[0] && distSq > radius[1] * radius[1])
					intersectPoint2Ds.add(testCorner);
			}
		}

		double lowerGF = 1;
		double upperGF = -1;

		for (Point2D p : intersectPoint2Ds) {
			double currentGF = getGuessFactor(getAngle(fireLocation, p));
			lowerGF = Math.min(lowerGF, currentGF);
			upperGF = Math.max(upperGF, currentGF);
		}

		return new Range(lowerGF, upperGF);
	}

	private static Point2D[] getVerticalIntersectionPoint2Ds(double centerX, double centerY, double r, double intersectX) {
		double deltaX = centerX - intersectX;
		double sqrtVal = r * r - deltaX * deltaX;

		if (sqrtVal < 0)
			return new Point2D[] {};

		sqrtVal = Math.sqrt(sqrtVal);

		return new Point2D[] { new Point2D.Double(intersectX, centerY + sqrtVal), new Point2D.Double(intersectX, centerY - sqrtVal) };
	}

	private final static Point2D[] getHorizontalIntersectionPoint2Ds(double centerX, double centerY, double r, double intersectY) {
		double deltaY = centerY - intersectY;
		double sqrtVal = r * r - deltaY * deltaY;

		if (sqrtVal < 0)
			return new Point2D[] {};

		sqrtVal = Math.sqrt(sqrtVal);

		return new Point2D[] { new Point2D.Double(centerX + sqrtVal, intersectY), new Point2D.Double(centerX - sqrtVal, intersectY) };
	}

	// End Precise Wave intersection code
	// ////////////////////////////////////

	public static final boolean inRange(double q, double[] bounds) {
		return bounds[0] <= q && q <= bounds[1];
	}

	public static final double getAngle(Point2D source, Point2D target) {
		return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
	}
}
