/*
 * BlotBot - attempt at a melee bot (originally based on DotBot)
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.blot;

import robocode.*;

import java.util.HashMap;
import java.util.Iterator;
import java.util.HashSet;
import java.io.*;
import java.awt.*;

import qohnil.util.*;
import qohnil.neural.TrainingPattern;
import qohnil.neural.NeuralNet;

/**
 * BlotBot
 *
 * KNOWN BUGS:
 * 1-on-1:
 *
 * melee:
 *      statistics on firing are not really accurate
 *
 * TO DO:
 * 1-on-1:
 *      add calls to LinearEnemyBullet .create(), .update() and .adapt()
 *
 *      correct pattern matching statistics
 *      correct firing mode selection
 *      deactivated neural network, reactivate?
 *      make hitting statistics more short-term?
 *      feed own moving data to neural net inputs?
 *      feed last known fire reaction angles into the matrix!
 *      first evade alwasy with stopTime == 3, only if hit: switch,
 *      keep track of evasion statistics?
 *      two neural networks (one for short range, one for long range)?
 *
 * melee:
 *      correct firing mode statistics (virtual bullets?)
 *
 * TO CHECK:
 * 1-on-1:
 *      check pattern matcher
 *      check for border collision, adapt evasion direction accordingly
 *      removed neural net statistics again
 *      take firepower into account for hitting statistics
 *
 */
public class BlotBot extends AdvancedRobot {

    public static final double FIRE_NEURALLY_THRESHOLD = 0.02;
    public static final int MIN_STOP_TIME = 3; // 3
    public static final int MAX_STOP_TIME = 10; // 8
    public static final int NUM_ITERATIONS = 1;

    // for stop-and-go-movement
    public static final int MOVE = 0;
    public static final int STOP = 1;

    int maxOthers = 10;
    String gunTarget = null;
    String evadeTarget = null;
    double gunTargetDistance = Double.MAX_VALUE;
    int wiggleStep = 100;
    HashMap robots = new HashMap(20);
    static HashMap statistics = new HashMap(500);
    int firingMode = 1;
    MovableData targetBot = null;
    static int numRounds = 0;
    static double[] targetDecisionParams = new double[10];
    long[] numFired = new long[3];
    long[] numHit = new long[3];
    int meleeAhead = 1;

    int ahead = 2; // can be -2 or 2. -2 for backward movement, 2 for forward movement.
    int stopTime = 3; // how much before next fire we should start to brake
    long nextFireTime = 999; // when the enemy will fire the next shot (earliest possible tick)
    long lastFireTime = -999; // when the enemy has fired the last time
    long turnsToMoveAhead = 0; // how many turns we want to keep moving
    double dEnergy = 0.0; // firePower of last shot
    double lastEnergy = 100.0; // energy our opponent had on last tick
    int moveMode = STOP; // our current movement mode
    double maxVelocity = 8.0; // the maximum velocity for 1-on-1 movement
    double avgVelocity = 0.0; // our average velocity over the last few turns

    boolean has1on1Target = false; // whether our 1-on-1 radar has a target
    String target1on1 = null; // the name of the current 1-on-1 target
    boolean hasShot = false; // whether we already fired, so that we don't need to execute() anymore

    static double[] num1on1Hit = new double[VirtualBlotBullet.NUM_FIRING_MODES]; // the success rate statistics for 1-on-1
    static long num1on1Fired = 0; // we need only one as we always "fire" all virtual bullets (and at the same time)
    static long num1on1BulletHit = 0; // how often we have hit the opponent (total)
    static long num1on1PatternFired = 0;
    static long num1on1GotFiredAt = 0; // how often the opponent has fired at us
    static long num1on1GotHit = 0; // ... and how often she actually hit us

    static long[] numStopTimeUsed = new long[MAX_STOP_TIME + 1]; // how often we used a specific stop time for evading (1-on-1)
    static long[] numStopTimeHit = new long[MAX_STOP_TIME + 1]; // how often we got hit using a specific stop time for evading (1-on-1)

    static MultimodeStatistics turnStats = new MultimodeStatistics(2, true); // how often we get hit after turning / not turning
    static MultimodeStatistics turnStatsFar = new MultimodeStatistics(2, true); // how often we get hit after turning / not turning when opponent is far away
    static MultimodeStatistics turnLengthStats = new MultimodeStatistics(2, true); // how often we get hit with long/short moves
    boolean haveTurned = false; // whether we have turned last time or not
    boolean longTurn = true;

    HashSet bulletSet = new HashSet(50); // contains all virtual bullets we fire
    HashSet realBulletSet = new HashSet(20); // contains all real bullets we fire
    HashSet enemyBullets = new HashSet(20); // contains the bullets the enemy fires at us

    ObjectBuffer firePatterns = new ObjectBuffer(30); // contains all current fire reaction patterns

    ObjectBuffer targetHistory = new ObjectBuffer(1600); // contains the last 5000 positions of our 1-on-1 opponent
    ObjectBuffer myHistory = new ObjectBuffer(1000); // contains the last 1000 positions of ourselves (for feeding into the patterned bullet evasion)

    static ObjectBuffer trainingPatterns = new ObjectBuffer(150);

    static PatternAnalyzer patternAnalyzer = new PatternAnalyzer(16, 800);
    static PatternAnalyzer patternAnalyzer32 = new PatternAnalyzer(32, 500);

    static PatternAnalyzer patternAnalyzerSelf = new PatternAnalyzer(16, 300);

    static NeuralPatternAnalyzer neuralAnalyzer = new NeuralPatternAnalyzer();

    double dodgePosX = 0.0;
    double dodgePosY = 0.0;

    private static BlotBot instance = null;

    public BlotBot() {
        super();
        BlotBot.instance = this;
    }

    public static BlotBot getInstance() {
        return instance;
    }

    /**
     * reinitialization stuff necessary when in melee in a round after being
     * in a 1-on-1 the round before...
     */
    private void reinit() {
        System.out.println("oh dear, need to reinit!");
        targetHistory.clear();
        trainingPatterns.clear();
        myHistory.clear();
    }

    public void run() {

        // initialize
        Logger.init("blotbot.log", Logger.WARN, this);
        loadStatistics();
        maxOthers = getOthers();
        loadParams();
        numRounds++;
        setColors(new Color(255, 127, 63), new Color(255, 127, 63), new Color(63, 255, 127));
        /*
        if (numRounds > 20) {
            patternAnalyzer.setSampleSize(32);
            patternAnalyzer.setMaxCapacity(700);
        }
        */
        setAdjustGunForRobotTurn(true);
		setAdjustRadarForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
        setTurnRadarLeftRadians(Double.MAX_VALUE);
        if (getOthers() > 1) {
            reinit();
        } else {
            patternAnalyzer.addHistory(targetHistory);
            patternAnalyzer32.addHistory(targetHistory);
            patternAnalyzerSelf.addHistory(myHistory);
            System.out.println("# PATTERNS: " + patternAnalyzer.getNumPatterns());
            neuralAnalyzer.addHistory(targetHistory);
        }

		// main thread
        while (true) {
            hasShot = false;
            if (getOthers() == 1) {
                run1on1();
            } else if (getOthers() > 1) {
                runMelee();
            }
            // execute turn
            if (!hasShot) {
                execute();
            }
        }
	}

    private void run1on1() {
        // if we have no 1on1 target yet, simply search one
        if (!has1on1Target) {
            setTurnRadarRightRadians(Math.PI / 4.0);
        }
        has1on1Target = false;

        // update moving average of velocity
        avgVelocity = avgVelocity * 0.8 + getVelocity() * 0.2;

        // now for the interesting stuff
        updateBotHistory();
        updateVirtualBullets();
//        neuralAnalyzer.update();
//        neuralAnalyzer.train(NUM_ITERATIONS);

        calculateAhead();
        calculateTurnAngle();

        calculateGun();
    }

    private void updateBotHistory() {
        if (myHistory.size() >= 1) {
            myHistory.add(new MovableData(this, (MovableData)myHistory.get()));
        } else {
            myHistory.add(new MovableData(this, null));
        }
    }

    private void updateFiringMode() {
        if (num1on1Fired < 5) {
            // by default, use linear fire
            firingMode = VirtualBlotBullet.LINEAR_FIRE;
        } else {
            // calculate sum of all hits
            int numConventionalModes = VirtualBlotBullet.NUM_FIRING_MODES;
            double[] success = new double[numConventionalModes];

            double sum = 0.0;
            for (int i = 0; i < numConventionalModes; i++) {
                success[i] = num1on1Hit[i] / num1on1Fired;
                success[i] *= success[i] * success[i];
                sum += success[i];
            }

            // debug output
/*
            System.out.print("firing mode ratios: ");
            for (int i = 0; i < VirtualBlotBullet.NUM_FIRING_MODES; i++) {
                System.out.print(Math.round(success[i] / sum * 100.0) + " ");
            }
            System.out.println();
*/

            // choose firing mode by weighted random
            double rnd = BotMath.random.nextDouble() * sum;
            double accumulate = 0.0;
            for (int i = 0; i < VirtualBlotBullet.NUM_FIRING_MODES; i++) {
                accumulate += success[i];
                if (rnd <= accumulate) {
                    firingMode = i;
                    // System.out.println("switched to " + i);
                    break;
                }
            }
        }
    }

    private void updateVirtualBullets() {

        // retrieve current target
        if (targetBot == null) {
            return;
        }

        updateOwnBullets();
        updateEnemyBullets();


        /* keeping track of the real bullets we fired
         * just needed for debugging purposes
         *//*
        iter = realBulletSet.iterator();
        while (iter.hasNext()) {
            Bullet b = (Bullet)iter.next();
        }
        */
    }

    private void updateOwnBullets() {
        HashSet bulletSetCopy = new HashSet(bulletSet);
        Iterator iter = bulletSetCopy.iterator();
        while (iter.hasNext()) {
            VirtualBlotBullet b = (VirtualBlotBullet)iter.next();
            b.update(getTime());
            if (b.hasHit(targetBot.coords)) {
                num1on1Hit[b.getFiringMode()] += 1.0;  //b.getFirePower();
                bulletSet.remove(b);
            } else if (b.hasPassed(targetBot.coords, getBattleFieldWidth(), getBattleFieldHeight())) {
                bulletSet.remove(b);
            }
        }
    }

    private void updateEnemyBullets() {
        HashSet bulletSetCopy = new HashSet(enemyBullets);
        Iterator iter = bulletSetCopy.iterator();
        while (iter.hasNext()) {
            VirtualBullet b = (VirtualBullet)iter.next();
            b.update(getTime());
            //addDot(b.coords.getX(), b.coords.getY(), 2.0);
            if (b.hasPassed(new Coords(getX(), getY()),
                    getBattleFieldWidth(), getBattleFieldHeight())) {
                enemyBullets.remove(b);
            }
        }
    }

    /**
     * Wolverine-style moving. Thanks to graygoo for his excellent ideas!
     */
    private void calculateAhead() {

        if (getOthers() == 0) {
            return;
        }

        // if we don't do any special dodging, we just slowly glide...
        // (in preparation for enemy fire, trying to irritate her)
        long timeUntilNextFire = nextFireTime - getTime();
        long timeFromLastFire = getTime() - lastFireTime;

        if (target1on1 == null || targetBot == null) {
            setAhead(ahead);
            return;
        }
        MovableData bot = targetBot;

        if (timeFromLastFire == 0 && turnsToMoveAhead <= 0) {

            // check if we need a new stop time
            int time = (int)BotMath.getBulletFlightTime(dEnergy, targetBot.distance);

            if (turnLengthStats.decideMode(2) == 0) {
                // works badly against LoneDragon
                turnsToMoveAhead = BotMath.random.nextInt(Math.max(1, time - 11)) + 8;
                turnLengthStats.add(0);
                longTurn = true;
            } else {
                // works badly against One et al
                turnsToMoveAhead = BotMath.random.nextInt(Math.max(time - 3, 1)) + 0;
                turnLengthStats.add(1);
                longTurn = false;
            }

            // reverse direction for dodging
            if (bot.distance < 100.0 || (bot.distance <= 150 && getEnergy() - bot.energy > 48.0)) {
                // never turn if too near (costs too much time)
                haveTurned = false;

            } else if (bot.distance < 350) {
                if (turnStats.decideMode(2) == 0) {
                    ahead = -ahead;
                    haveTurned = true;
                } else {
                    // we didn't turn
                    haveTurned = false;
                }
                System.out.println("turn decision (turn / don't turn): "
                        + turnStats.getNumTotal(0) + " (" + turnStats.getHitRatio(0)
                        + ") / " + turnStats.getNumTotal(1) + " ("
                        + turnStats.getHitRatio(1) + ")");
            } else {
                if (turnStats.decideMode(2) == 0) {
                    ahead = -ahead;
                    haveTurned = true;
                } else {
                    haveTurned = false;
                }
            }

            // check if we would hit a virtual bullet, if so: reverse
            Coords ourNextPositionNoTurn = BotMath.polarToCoords(
                    BotMath.normalizeAbsoluteAngle(getHeadingRadians()
                        + getPerpendicularCorrection1on1(bot, Math.PI / 3.0)), // approximate new direction
                    turnsToMoveAhead * 2.0 * ahead,  new Coords(getX(), getY()));
            //addDot(ourNextPositionNoTurn.getX(), ourNextPositionNoTurn.getY(), 22.5);
            ahead = -ahead;
            Coords ourNextPositionTurn = BotMath.polarToCoords(
                    BotMath.normalizeAbsoluteAngle(getHeadingRadians()
                        + getPerpendicularCorrection1on1(bot, Math.PI / 3.0)), // approximate new direction
                    turnsToMoveAhead * 2.0 * ahead,
                    new Coords(getX(), getY()));
            ahead = -ahead;
            //addDot(ourNextPositionTurn.getX(), ourNextPositionTurn.getY(), 11.25);

            Iterator iter = enemyBullets.iterator();
            double noTurnHitProb = 0.0;
            double turnHitProb = 0.0;
            while (iter.hasNext()) {
                Object obj = iter.next();
                VirtualBullet bullet = null;
                if (obj instanceof LinearEnemyBullet) {
                    bullet = (LinearEnemyBullet)obj;
                } else {
                    bullet = (PatternEnemyBullet)obj;
                }

                double rel2 = bullet.getReliability() * bullet.getReliability();
                if ((bot.distance < 300 &&  rel2 > 0.4) || rel2 > 0.66) {
                    noTurnHitProb += rel2 * bullet.willHitProbability(ourNextPositionNoTurn, 10);
                    turnHitProb += rel2 * bullet.willHitProbability(ourNextPositionTurn, 10);
                }
            }

            System.out.println("turn: " + turnHitProb + " / no turn: " + noTurnHitProb);

            if (noTurnHitProb > 2.0 * turnHitProb) {
                System.out.println("bullet would probably hit, reverse!");
                ahead = -ahead;
                haveTurned = !haveTurned;
            }

            // check if we have a border problem, if so: reverse
            Coords c = BotMath.polarToCoords(getHeadingRadians(), ahead * 25.0,
                    new Coords(getX(), getY()));
            if (c.getX() < 25.0 || c.getX() > getBattleFieldWidth() - 25.0
                    || c.getY() < 25.0 || c.getY() > getBattleFieldHeight() - 25.0) {
                System.out.println("avoiding border!");
                ahead = -ahead;
                haveTurned = !haveTurned;
            }

            // update turn statistics accordingly
            if (bot.distance < 350) {
                if (haveTurned) {
                    turnStats.add(0);
                } else {
                    turnStats.add(1);
                }
            } else {
                if (haveTurned) {
                    turnStatsFar.add(0);
                } else {
                    turnStatsFar.add(1);
                }
            }

            moveMode = MOVE;
            maxVelocity = 7.0 + BotMath.random.nextDouble();
        }

        if (moveMode == MOVE) {
            // MOVE MODE
            setMaxVelocity(maxVelocity + BotMath.random.nextDouble() - 0.5);
            setAhead(ahead * 1000);
            turnsToMoveAhead--;
            if (turnsToMoveAhead == 0) {
                moveMode = STOP;
                maxVelocity = 0.1 + BotMath.random.nextDouble() * 1.0;
            }

        } else {
            // STOP MODE
            setMaxVelocity(maxVelocity + BotMath.random.nextDouble() - 0.5);
            setAhead(ahead * 1000);
        }
    }

    private void calculateTurnAngle() {
        if (targetBot == null) {
            return;
        }
        MovableData bot = targetBot;

        // okay, we have a known target, evade!
        double wishedAngle = Math.PI / 2.0; // 90 degrees
        if (bot.distance > 375.0) {
            wishedAngle = Math.PI / 3.5; // 45 degrees (was: 45 degrees, now also 60 degrees)
        } else if (bot.distance > 150.0) {
            wishedAngle = Math.PI / 3.0; // 60 degrees

        } else if (getEnergy() - bot.energy > 48.0) { // near and huge energy avantage
            // go even nearer, ramming
            wishedAngle = Math.PI / 4.0; // 45 degrees
        } else if (bot.distance < 100.0) {
            // flee
            wishedAngle = 3.0 * Math.PI / 4.0; // 135 degrees
        }

        double rndAngle = 0.0;
        if (moveMode == STOP) {
            rndAngle = BotMath.random.nextDouble() * (Math.PI / 4.0) + Math.PI / 8.0;
        } else {
            rndAngle = BotMath.random.nextDouble() * (Math.PI / 8.0);
        }
        if (BotMath.random.nextBoolean()) {
            rndAngle = -rndAngle;
        }
        wishedAngle += rndAngle;

        double angle = getPerpendicularCorrection1on1(bot, wishedAngle);
        angle += Math.sin((double)getTime() * Math.PI / 3.5) * Math.PI / 18.0;
        angle += (BotMath.random.nextDouble() - 0.5) * Math.PI / 18.0;

        setTurnRightRadians(BotMath.normalizeRelativeAngle(angle));
    }

    private void calculateGun() {
        MovableData bot = targetBot;
        if (bot == null) {
            return;
        }

        // okay, we have a known target

        // simple direct fire shooting
        FireParameters direct = new FireParameters(bot.bearing, bot.distance,
                compute1on1Firepower(bot.distance, bot));

        // linear prediction
        FireParameters linear = new FireParameters(direct);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getLinearPredictedCoords(
                    Math.round(linear.getDistance()
                            / getBulletVelocity(linear.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            linear = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(linear.getDistance(), bot));
        }

        // circular prediction
        FireParameters circular = new FireParameters(linear);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getCircularPredictedCoords(
                    Math.round(circular.getDistance()
                            / getBulletVelocity(circular.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            circular = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(circular.getDistance(), bot));
        }

        // remember real velocity for after the inverse predictions
        double storeVelocity = bot.velocity;
        double storeTurnVelocity = bot.turnVelocity;

        // compute averages
        double avgVelocity = 0.0;
        double avgAbsVelocity = 0.0;
        double avgTurnVelocity = 0.0;
        double avgAbsTurnVelocity = 0.0;
        int numAvgValues = Math.min(targetHistory.size(), 32);
        for (int i = 0; i < numAvgValues; i++) {
            MovableData temp = (MovableData)targetHistory.get(i);
            avgVelocity += temp.velocity;
            avgAbsVelocity += Math.abs(temp.velocity);
            avgTurnVelocity += temp.turnVelocity;
            avgAbsTurnVelocity += Math.abs(temp.turnVelocity);
        }
        avgVelocity /= numAvgValues;
        avgAbsVelocity /= numAvgValues;
        avgTurnVelocity /= numAvgValues;
        avgAbsTurnVelocity /= numAvgValues;

        // for averaged aiming
        bot.velocity = avgVelocity;

        // averaged linear prediction
        FireParameters avgLinear = new FireParameters(direct);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getLinearPredictedCoords(
                    Math.round(avgLinear.getDistance()
                            / getBulletVelocity(avgLinear.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            avgLinear = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(avgLinear.getDistance(), bot));
        }

        // for absolute averaged aiming
        bot.velocity = avgAbsVelocity;
        if (storeVelocity < 0.0) {
            bot.velocity = - bot.velocity;
        }

        // averaged linear prediction
        FireParameters avgAbsLinear = new FireParameters(direct);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getLinearPredictedCoords(
                    Math.round(avgAbsLinear.getDistance()
                            / getBulletVelocity(avgAbsLinear.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            avgAbsLinear = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(avgAbsLinear.getDistance(), bot));
        }

        // for averaged aiming
        bot.velocity = avgVelocity;
        bot.turnVelocity = avgTurnVelocity;

        // averaged circular prediction
        FireParameters avgCircular = new FireParameters(avgLinear);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getCircularPredictedCoords(
                    Math.round(avgCircular.getDistance()
                            / getBulletVelocity(avgCircular.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            avgCircular = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(avgCircular.getDistance(), bot));
        }

        // for averaged aiming
        bot.velocity = avgAbsVelocity;
        bot.turnVelocity = avgAbsVelocity;
        if (storeVelocity < 0.0) {
            bot.velocity = - bot.velocity;
        }
        if (storeTurnVelocity < 0.0) {
            bot.turnVelocity = - bot.turnVelocity;
        }

        // absolute averaged circular prediction
        FireParameters avgAbsCircular = new FireParameters(avgAbsLinear);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getCircularPredictedCoords(
                    Math.round(avgAbsCircular.getDistance()
                            / getBulletVelocity(avgAbsCircular.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            avgAbsCircular = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(avgAbsCircular.getDistance(), bot));
        }

        // restore real turn velocity
        bot.turnVelocity = storeTurnVelocity;

        // for inverse aiming
        bot.velocity = - 0.6 * storeVelocity;

        // inverse linear prediction
        FireParameters invLinear = new FireParameters(direct);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getLinearPredictedCoords(
                    Math.round(invLinear.getDistance()
                            / getBulletVelocity(invLinear.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            invLinear = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(invLinear.getDistance(), bot));
        }

        // inverse circular prediction
        FireParameters invCircular = new FireParameters(invLinear);
        for (int i = 0; i < 3; i++) {
            // predict target position after that time
            Coords predictedCoords = bot.getCircularPredictedCoords(
                    Math.round(invCircular.getDistance()
                            / getBulletVelocity(invCircular.getFirePower())),
                    getTime());

            // update predictions
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            invCircular = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(invCircular.getDistance(), bot));
        }

        // restore real velocity
        bot.velocity = storeVelocity;

        // pattern matching prediction
        FireParameters pattern = new FireParameters(direct);
        boolean shortRange = (pattern.getDistance() < 300)
                || (patternAnalyzer32.getNumPatterns() < 500);
        if (shortRange) {
            patternAnalyzer.findSimilarIdx((int)Math.round(
                    (direct.getDistance() + 200.0) / getBulletVelocity(direct.getFirePower())));
        } else {
            patternAnalyzer32.findSimilarIdx((int)Math.round(
                    (direct.getDistance() + 200.0) / getBulletVelocity(direct.getFirePower())));
        }
        for (int i = 0; i < 3; i++) {
            int patternTime = (int)Math.round(
                pattern.getDistance() / getBulletVelocity(pattern.getFirePower()));
            Polar reaction = null;
            if (shortRange) {
                reaction = patternAnalyzer.getPrediction(patternTime);
            } else {
                reaction = patternAnalyzer32.getPrediction(patternTime);
            }
            if (reaction == null) {
                pattern = new FireParameters(direct);
                break;
            }

            // calculate updated bot position
            Coords predictedCoords = BotMath.polarToCoords(
                    BotMath.normalizeAbsoluteAngle(bot.heading + reaction.getBearing()),
                    reaction.getDistance(), bot.coords);

            // clip at battlefield borders
            predictedCoords = BotMath.clip(predictedCoords, bot.coords,
                    new Coords(getBattleFieldWidth(), getBattleFieldHeight()));

            // reconvert to polar
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            pattern = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(pattern.getDistance(), bot));
        }

        // neural prediction
        FireParameters neural = new FireParameters(direct);
/*
        for (int i = 0; i < 3; i++) {
            int neuralTime = (int)Math.round(
                neural.getDistance() / getBulletVelocity(neural.getFirePower()));
            Polar reaction = neuralAnalyzer.getPrediction(neuralTime);

            // calculate updated bot position
            Coords predictedCoords = BotMath.polarToCoords(
                    BotMath.normalizeAbsoluteAngle(bot.heading + reaction.getBearing()),
                    reaction.getDistance(), bot.coords);

            // clip at battlefield borders
            predictedCoords = BotMath.clip(predictedCoords, bot.coords,
                    new Coords(getBattleFieldWidth(), getBattleFieldHeight()));

            // reconvert to polar
            Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
            neural = new FireParameters(polar.getBearing(), polar.getDistance(),
                    compute1on1Firepower(neural.getDistance(), bot));
        }
*/

        // selected angle to actually shoot at according to firing mode
        FireParameters predicted = new FireParameters(
                0.0, Double.POSITIVE_INFINITY, 0.0);

        switch (firingMode) {
            case VirtualBlotBullet.DIRECT_FIRE:
                predicted = direct;
                break;

            case VirtualBlotBullet.LINEAR_FIRE:
                predicted = linear;
                break;

            case VirtualBlotBullet.CIRCULAR_FIRE:
                predicted = circular;
                break;

            case VirtualBlotBullet.INVERSE_LINEAR_FIRE:
                predicted = invLinear;
                break;

            case VirtualBlotBullet.INVERSE_CIRCULAR_FIRE:
                predicted = invCircular;
                break;

            case VirtualBlotBullet.AVG_LINEAR_FIRE:
                predicted = avgLinear;
                break;

            case VirtualBlotBullet.AVG_CIRCULAR_FIRE:
                predicted = avgCircular;
                break;

            case VirtualBlotBullet.AVG_ABS_LINEAR_FIRE:
                predicted = avgAbsLinear;
                break;

            case VirtualBlotBullet.AVG_ABS_CIRCULAR_FIRE:
                predicted = avgAbsCircular;
                break;

/*
            case VirtualBlotBullet.PATTERN_MATCHING_FIRE:
                predicted = pattern;
                break;
                */
            default:
                Logger.error("INVALID FIRING MODE: " + firingMode);
        }

        // fire neurally if possible
        if (neuralAnalyzer.getError() < 0.04) {
            System.out.println("fire neurally, error = " + neuralAnalyzer.getError());
            predicted = neural;

        // otherwise, fire pattern-matched
        } else if (patternAnalyzer.getError() < 0.1) {
            predicted = pattern;
        }

        if (predicted.getDistance() < 60.0) {
            predicted = direct;
        }

        double turnAngle = BotMath.normalizeRelativeAngle(
                predicted.getAngle() - getGunHeadingRadians()
                + (BotMath.random.nextDouble() - 0.5) * (Math.PI / 90.0)); // slightly randomize
        setTurnGunRightRadians(turnAngle);

        // ok, gun is heading in the right direction, shoot!
        if (Math.abs(turnAngle)
                < Math.max(22.5 / predicted.getDistance(),
                        Math.PI / 360.0)
                && getGunHeat() <= 0.0) {

            if (getEnergy() > 0.2) {
                // create virtual bullets for all firing modes used...
                // create them only when we actually fired, because we want
                // to factor in the firing reaction of the opponent

                // direct fire
                VirtualBlotBullet bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.DIRECT_FIRE,
                        direct.getAngle(), direct.getFirePower(), this);
                bulletSet.add(bullet);

                // linear fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.LINEAR_FIRE,
                        linear.getAngle(), linear.getFirePower(), this);
                bulletSet.add(bullet);

                // circular fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.CIRCULAR_FIRE,
                        circular.getAngle(), circular.getFirePower(), this);
                bulletSet.add(bullet);

                // avg linear fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.AVG_LINEAR_FIRE,
                        avgLinear.getAngle(), avgLinear.getFirePower(), this);
                bulletSet.add(bullet);

                // avg circular fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.AVG_CIRCULAR_FIRE,
                        avgCircular.getAngle(), avgCircular.getFirePower(), this);
                bulletSet.add(bullet);

                // avg abs linear fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.AVG_ABS_LINEAR_FIRE,
                        avgAbsLinear.getAngle(), avgAbsLinear.getFirePower(), this);
                bulletSet.add(bullet);

                // avg abs circular fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.AVG_ABS_CIRCULAR_FIRE,
                        avgAbsCircular.getAngle(), avgAbsCircular.getFirePower(), this);
                bulletSet.add(bullet);

                // linear fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.INVERSE_LINEAR_FIRE,
                        invLinear.getAngle(), invLinear.getFirePower(), this);
                bulletSet.add(bullet);

                // circular fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.INVERSE_CIRCULAR_FIRE,
                        invCircular.getAngle(), invCircular.getFirePower(), this);
                bulletSet.add(bullet);

/*
                // pattern matching fire
                bullet = new VirtualBlotBullet(
                        VirtualBlotBullet.PATTERN_MATCHING_FIRE,
                        pattern.getAngle(), pattern.getFirePower(), this);
                bulletSet.add(bullet);
                */

                num1on1Fired++;
                updateFiringMode();

                fire(Math.min(predicted.getFirePower(), getEnergy() - 0.1));
                hasShot = true;
            }
        }
    }

    private void storeFirePattern(double directFirePower, double directDistance,
                                  MovableData bot, double directAngle,
                                  double linearAngle, double circularAngle) {

        // calculate time the bullet would hit in direct fire mode
        double timeUntilImpact = BotMath.getBulletFlightTime(
                directFirePower, directDistance);


    }

    private void runMelee() {

        setMaxVelocity(8.0);

        double angle = BotMath.random.nextDouble() * 2.0 * Math.PI;

        // evade the current evasion point (enemy bot that last fired at us)
        if (evadeTarget != null) {
            angle = getPerpendicularCorrection(
                    (MovableData)robots.get(evadeTarget), Math.PI / 2.0);
        }

        // move according to antigravity
        double antigrav = calculateAntigravity() - getHeadingRadians();
        if (ahead < 0) {
            // when moving in the reverse direction, we also need to reverse
            // the antigravity force...
            antigrav = Math.PI + antigrav;
        }

        angle = 0.67 * angle + 0.33 * BotMath.normalizeRelativeAngle(antigrav);

        // wiggle a bit
        angle += Math.sin((double)getTime() * 0.785398163) * 0.174532925;

        // turning
        setTurnRightRadians(BotMath.normalizeRelativeAngle(angle));

        // update movement
        if (getDistanceRemaining() == 0.0) {
            double ahead = BotMath.random.nextDouble() * wiggleStep * 0.75 + wiggleStep * 0.25;
            if (BotMath.random.nextBoolean()) {
                meleeAhead = -meleeAhead;
            }
            setAhead(meleeAhead * ahead);
        }

        // decide gun target if none yet or anymore
        if (gunTarget == null) {
            decideGunTarget();
        }

        // update gun
        if (gunTarget != null) {
            targetBot = (MovableData)robots.get(gunTarget);
            if (targetBot != null) {

                // do simple direct hit angle approximation
                double predictedAngle = targetBot.bearing;
                double predictedDistance = targetBot.distance;
                double predictedFirePower = computeFirepower(predictedDistance, targetBot);

                if (firingMode != 0) {
                    for (int i = 0; i < 5; i++) {
                        // predict target position after that time
                        Coords predictedCoords;
                        if (firingMode == 2) {
                            predictedCoords = targetBot.getCircularPredictedCoords(
                                    Math.round(predictedDistance / getBulletVelocity(predictedFirePower)), getTime());
                        } else {
                            predictedCoords = targetBot.getLinearPredictedCoords(
                                    Math.round(predictedDistance / getBulletVelocity(predictedFirePower)), getTime());
                        }

                        // update predictions
                        Polar polar = BotMath.coordsToPolar(predictedCoords, new Coords(getX(), getY()));
                        predictedDistance = polar.getDistance();
                        predictedAngle = polar.getBearing();
                        predictedFirePower = computeFirepower(predictedDistance, targetBot);
                    }
                }

                double turnAngle = BotMath.normalizeRelativeAngle(
                                predictedAngle - getGunHeadingRadians());
                setTurnGunRightRadians(turnAngle);
                if (Math.abs(turnAngle)
                        < Math.max(34.90658504
                                / predictedDistance, 8.72664626e-3)
                        && getGunHeat() <= 0.0) {
                    StaticBotData targetStats = (StaticBotData)statistics.get(gunTarget);
                    if (targetStats == null) {
                        targetStats = new StaticBotData();
                    }
                    targetStats.numFired[firingMode]++;
                    numFired[firingMode]++;
                    statistics.put(gunTarget, targetStats);
                    firingMode++;
                    firingMode %= 3;
                    if (targetStats.numFired[0] >= 5 && targetStats.numFired[1] >= 5 && targetStats.numFired[2] >= 5) {
                        double v = BotMath.random.nextDouble();
                        double d1 = (double)targetStats.numHit[0] / targetStats.numFired[0] + 0.025;
                        double d2 = (double)targetStats.numHit[1] / targetStats.numFired[1] + 0.025;
                        double d3 = (double)targetStats.numHit[2] / targetStats.numFired[2] + 0.025;
                        d1 *= d1;
                        d2 *= d2;
                        d3 *= d3;
                        v *= d1 + d2 + d3;
                        if (v <= d1) {
                            firingMode = 0;
                        } else if (v <= d1 + d2) {
                            firingMode = 1;
                        } else {
                            firingMode = 2;
                        }
                    } else if (numFired[0] >= 5 && numFired[1] >= 5 && numFired[2] >= 5) {
                        double v = BotMath.random.nextDouble();
                        double d1 = (double)numHit[0] / numFired[0] + 0.025;
                        double d2 = (double)numHit[1] / numFired[1] + 0.025;
                        double d3 = (double)numHit[2] / numFired[2] + 0.025;
                        d1 *= d1;
                        d2 *= d2;
                        d3 *= d3;
                        v *= d1 + d2 + d3;
                        if (v <= d1) {
                            firingMode = 0;
                        } else if (v <= d1 + d2) {
                            firingMode = 1;
                        } else {
                            firingMode = 2;
                        }
                    }
                    if (getEnergy() > 0.2) {
                        fire(Math.min(predictedFirePower, getEnergy() - 0.1));
                        // decide new gun target after firing
                        decideGunTarget();
                        hasShot = true;
                    }
                }
            }
        }
    }

    public double calculateAntigravity() {
        Coords grav = new Coords(0.0, 0.0);
        double myX = getX();
        double myY = getY();

        // hide from all other bots
        Iterator iter = robots.values().iterator();
        while (iter.hasNext()) {
            MovableData bot = (MovableData)iter.next();
            if (bot == null || !bot.isAlive) {
                continue;
            }
            double x = bot.coords.getX() - myX;
            double y = bot.coords.getY() - myY;
            double d = x * x + y * y;
            d = -1000000.0 / d;
            grav.add(new Coords(x * d, y * d));
        }

        // center avoidance
        double x = getBattleFieldWidth() / 2.0 - myX;
        double y = getBattleFieldHeight() / 2.0 - myY;
        double d = Math.sqrt(x * x + y * y);
        d = -2500.0 / d;
        grav.add(new Coords(x * d, y * d));

        // corner attraction
        // upper left
        /*
        d = 2500.0 / Math.sqrt(myX * myX + myY * myY);
        grav.add(new Coords(myX * d, myY * d));
        // upper right
        x = getBattleFieldWidth() - myX;
        d = 2500.0 / Math.sqrt(x * x + myY * myY);
        grav.add(new Coords(x * d, myY * d));
        // lower left
        y = getBattleFieldHeight() - myY;
        d = 2500.0 / Math.sqrt(myX * myX + y * y);
        grav.add(new Coords(myX * d, y * d));
        // lower right
        x = getBattleFieldWidth() - myX;
        y = getBattleFieldHeight() - myY;
        d = 2500.0 / Math.sqrt(x * x + y * y);
        grav.add(new Coords(x * d, y * d));
        */

        // wall avoidance
        // bottom
        double gravY = - getY();
        gravY *= -25000000.0 / Math.abs(gravY * gravY * gravY);
        grav.add(new Coords(0, gravY));
        // left
        double gravX = - getX();
        gravX *= -25000000.0 / Math.abs(gravX * gravX * gravX);
        grav.add(new Coords(gravX, 0));
        // top
        gravY = getBattleFieldHeight() - getY();
        gravY *= -25000000.0 / Math.abs(gravY * gravY * gravY);
        grav.add(new Coords(0, gravY));
        // right
        gravX = getBattleFieldWidth() - getX();
        gravX *= -25000000.0 / Math.abs(gravX * gravX * gravX);
        grav.add(new Coords(gravX, 0));

        return Math.atan2(grav.getX(), grav.getY());
    }

    public void loadParams() {
        File f = getDataFile("params.cfg");
        try {
            BufferedReader br = new BufferedReader(new FileReader(f));
            for (int i = 0; i < targetDecisionParams.length; i++) {
                targetDecisionParams[i] = parseParam(br.readLine());
            }
            br.close();
        } catch (IOException e) {
            Logger.fatal(e.getMessage());
        }
    }

    private double parseParam(String line) {
        try {
            return Double.parseDouble(line);
        } catch(NumberFormatException e) {
            Logger.fatal(e.getMessage());
            return 0.0;
        }
    }

    public void decideGunTarget() {
        // evaluate each bot
        double maxEval = -1000.0;
        Iterator iter = robots.keySet().iterator();
        String lastTarget = gunTarget;
        while (iter.hasNext()) {
            String botName = (String)iter.next();
            MovableData bot = (MovableData)robots.get(botName);
            if (bot == null || !bot.isAlive) {
                continue;
            }

            // distance factor
            double distanceEval =  40.0 / bot.coords.distance(getX(), getY());

            // energy factor
            double energyEval = (200.0 - bot.energy) / 200.0;

            // angle factor
            double angle = Math.abs(BotMath.normalizeRelativeAngle(
                    bot.heading - getGunHeadingRadians()));
            if (angle > BotMath.DEG90) {
                angle = BotMath.DEG90 - angle;
            }
            double angleEval = 1.0 - angle / BotMath.DEG90;

            // last attack time factor
            long timeDiff = getTime() - bot.lastAttackTime;
            double attacksMeEval = 0.0;
            if (timeDiff <= 80.0) {
                attacksMeEval = 0.5;
            }
            if (timeDiff <= 40.0) {
                attacksMeEval = 1.0;
            }

            // wasAlreadyAttackedFactor
            // if we already attacked this enemy, no reservations for
            // attacking him again.
            if (bot.wasAlreadyAttacked) {
                Logger.info("I ALREADY ATTACKED " + botName);
            }
            double alreadyAttackedEval = (bot.wasAlreadyAttacked) ? 1.0 : 0.0;

            // continuity factor: try not to switch targets too often
            double continuityEval = (botName.equals(lastTarget)) ? 1.0 : 0.0;

            StaticBotData stats = (StaticBotData)statistics.get(botName);

            double hitRatioEval = 0.0;
            double gotMeKilledEval = 0.0;
            double damageDealtEval = 0.0;
            double damageTakenEval = 0.0;

            if (stats != null) {
                // hit ratio factor
                long numHit = stats.numHit[0] + stats.numHit[1] + stats.numHit[2];
                long numFired = stats.numFired[0] + stats.numFired[1] + stats.numFired[2];
                hitRatioEval = (numFired > 0)
                        ? (double)numHit / numFired : 0.0;

                // got killed factor
                if (numRounds > 2) {
                    gotMeKilledEval = 1.0 - (double)stats.numKilledMe / (numRounds - 1);
                }

                // damage dealt factor - how much the current bot has damaged us
                // the more the more urgent it gets to kill'em
                damageDealtEval = stats.damageDealt / numRounds / 150.0;

                // damage taken factor - how much we have damaged the current bot
                // in previous rounds - the more the better
                damageTakenEval = stats.damageTaken / numRounds / 150.0;
            }

            // combine to total evaluation
            double totalEval
                    = distanceEval * targetDecisionParams[0] // 1.5
                    + energyEval * targetDecisionParams[1] // 0.25
                    + angleEval * distanceEval * targetDecisionParams[2] // 0.25
                    + attacksMeEval * targetDecisionParams[3] // 0.5
                    + alreadyAttackedEval * targetDecisionParams[4] // 0.05
                    + continuityEval * targetDecisionParams[5] // 0.1
                    + hitRatioEval * targetDecisionParams[6] // 0.25
                    + gotMeKilledEval * targetDecisionParams[7] // 0.1
                    + damageDealtEval * targetDecisionParams[8] // 0.125
                    + damageTakenEval * targetDecisionParams[9]; // 0.25

            Logger.info(botName + " eval: " /* + totalEval + " / "
                    + distanceEval + ", "
                    + energyEval + ", " + angleEval + ", " */
                    + attacksMeEval + ", " + alreadyAttackedEval + ", "
                    + continuityEval /*+ ", "
                    + hitRatioEval + ", " + gotMeKilledEval
                    + ", " + damageDealtEval
                    + ", " + damageTakenEval */);

            if (totalEval > maxEval) {
                maxEval = totalEval;
                gunTarget = botName;
            }

        }
    }

    public double compute1on1Firepower(double distance, MovableData target) {
        double power = 600.0 / distance;

        // compute relative movement angle
        double angle = Math.abs(BotMath.normalizeRelativeAngle(
                target.bearing - getHeadingRadians()));
        if (angle > Math.PI / 2.0) {
            angle = Math.PI - angle;
        }
        double angleFactor = (Math.PI / 2.0 - angle) / Math.PI + 0.75; // from 0.75 to 1.25
        power *= angleFactor;
//        double speedFactor = (8.0 - target.velocity) / 32.0 + 0.875; // from 0.875 to 1.125
//        power *= speedFactor;

        return Math.min(3.0, Math.max(Math.min(power, target.energy / 4.0), 0.1));
    }

    public double computeFirepower(double distance, MovableData target) {
        double power = 600.0 / distance;

        // compute relative movement angle
        double angle = Math.abs(BotMath.normalizeRelativeAngle(
                target.bearing - getHeadingRadians()));
        if (angle > Math.PI / 2.0) {
            angle = Math.PI - angle;
        }
        double angleFactor = (Math.PI / 2.0 - angle) / Math.PI + 0.75; // from 0.75 to 1.25
        power *= angleFactor;
        double speedFactor = (8.0 - target.velocity) / 32.0 + 0.875; // from 0.75 to 1.25
        power *= speedFactor;
        StaticBotData stats = (StaticBotData)statistics.get(target.name);
        double successFactor = 1.0;
        if (stats != null && stats.numFired[firingMode] >= 5) {
            successFactor = (double)stats.numHit[firingMode] / stats.numFired[firingMode];
            successFactor = successFactor + 0.75;
        }
        power *= successFactor;
//        double numEnemiesFactor = (double)(maxOthers - getOthers())
//                / (maxOthers - 1) / 4 + 0.875;
//        power *= numEnemiesFactor;

        return Math.min(3.0, Math.max(Math.min(power, target.energy / 4.0), 0.1));
    }

    public static double getBulletVelocity(double firePower) {
        return 20.0 - 3.0 * firePower;
    }

    public static double getDamageToFirepower(double power) {
        double damage = 4 * power;
        if (power > 1) {
            damage += 2 * (power-1);
        }
        return damage;
    }

    public void onScannedRobot(ScannedRobotEvent e) {
        MovableData previous = (MovableData)robots.get(e.getName());
        MovableData current = new MovableData(e, getX(), getY(),
                   getHeadingRadians(), previous);
        robots.put(e.getName(), current);

        if (getOthers() == 1) {
            targetBot = current;
            targetHistory.add(current);
            has1on1Target = true;
            target1on1 = e.getName();

            // tribute to Grafi's unbelievable Geek family: his original Geek radar
            setTurnRadarRightRadians(BotMath.normalizeRelativeAngle(current.bearing - getRadarHeadingRadians()) * 1.9);

            // check for bullet to dodge
            dEnergy = lastEnergy - current.energy;
            if (dEnergy > 0.0001) {
                // enemy has lost some energy, even taking bulletHit and hitByBullet
                // into account => she must have shot at us (not taking border crashing
                // or ramming into account).
                lastFireTime = getTime();
                // how long at minimum till next firing?
                // gunHeat * 0.1 !
                nextFireTime = getTime()
                        + (long)Math.ceil((1.0 + dEnergy / 5.0) / 0.1);
                num1on1GotFiredAt++;

                // add virtual bullets to list
                LinearEnemyBullet bullet = new LinearEnemyBullet(
                        dEnergy, targetBot.coords, getTime(),
                        new Coords(getX(), getY()), // we are the target of this bullet
                        getHeadingRadians(), /*avgVelocity*/getVelocity());
                enemyBullets.add(bullet);
                if (patternAnalyzerSelf.getNumPatterns() > 200 && myHistory.size() > 16) {
                    System.out.println("enough self patterns, adding PATTERN_BULLETS");
                    PatternEnemyBullet pBullet = new PatternEnemyBullet(
                            patternAnalyzerSelf, dEnergy, targetBot.coords, getTime(),
                            new Coords(getX(), getY()), // we are the target of this bullet
                            getHeadingRadians(), /*avgVelocity*/getVelocity());
                    enemyBullets.add(pBullet);
                }
            }
            // prepare for next tick
            lastEnergy = current.energy;
        } else {
            // decide initial evasion target
            if (e.getDistance() * 1.5 < gunTargetDistance) {
                if (evadeTarget == null) {
                    evadeTarget = gunTarget;
                }
            }
        }
    }

    public void onBulletHit(BulletHitEvent e) {
//        System.out.println("REAL HIT! (" + e.getTime() + ")");
        if (getOthers() > 1) {
            String hitBotName = e.getName();
            StaticBotData targetStats = (StaticBotData)statistics.get(hitBotName);
            if (targetStats == null) {
                targetStats = new StaticBotData();
            }
            targetStats.numHit[firingMode]++;
            numHit[firingMode]++;
            targetStats.damageTaken += getDamageToFirepower(e.getBullet().getPower());
            statistics.put(hitBotName, targetStats);
            Logger.info("onBulletHit: " + hitBotName + targetStats.numHit[firingMode]
                    + ", " + targetStats.damageTaken);

            // remember that we have attacked this bot this round.
            // for trying not to attack too many bots at once
            MovableData bot = (MovableData)robots.get(hitBotName);
            if (bot != null) {
                Logger.info("attacked " + hitBotName);
                bot.wasAlreadyAttacked = true;
                robots.put(hitBotName, bot);
            }
        } else if (getOthers() == 1) {
            // adapt 1-on-1 last energy value of opponent
            lastEnergy = e.getEnergy();
            num1on1BulletHit++;
        }
    }

    public void onHitByBullet(HitByBulletEvent e) {
        evadeTarget = e.getName();

        if (getOthers() > 1) {
            // remember when I was last hit by this bot
            MovableData bot = (MovableData)robots.get(evadeTarget);
            if (bot != null) {
                bot.lastAttackTime = e.getTime();
            }
            robots.put(evadeTarget, bot);

            StaticBotData targetStats = (StaticBotData)statistics.get(evadeTarget);
            if (targetStats == null) {
                targetStats = new StaticBotData();
            }
            targetStats.damageDealt += getDamageToFirepower(e.getPower());
            statistics.put(evadeTarget, targetStats);

            wiggleStep = BotMath.random.nextInt(100) + 50;

        } else if (getOthers() == 1) {
            // adapt 1-on-1 last energy value of opponent
            // she gets energy back for hitting us.
            lastEnergy += 3.0 * e.getPower();

            // adapt virtual enemy bullet lead factor

            // first, find the correct bullet(s)
            Iterator iter = enemyBullets.iterator();
            long timeDiff = Long.MAX_VALUE;
            HashSet hitBullets = new HashSet();
            while (iter.hasNext()) {
                VirtualBullet bullet = (VirtualBullet)iter.next();
                if (Math.abs(bullet.getHitTime() - getTime()) <= timeDiff) {
                    timeDiff = Math.abs(bullet.getHitTime() - getTime());
                    hitBullets.add(bullet);
                }
            }

            // now, finally adapt
            iter = hitBullets.iterator();
            while (iter.hasNext()) {
                VirtualBullet hitBullet = (VirtualBullet)iter.next();
                hitBullet.adaptReliability(e.getHeadingRadians());

                // and remove from set of bullets
                enemyBullets.remove(hitBullet);
            }

            // keep track of bullet dodging statistics
            numStopTimeHit[stopTime]++;
            if (targetBot != null) {
                if (targetBot.distance < 350.0) {
                    if (haveTurned) {
                        turnStats.hit(0);
                    } else {
                        turnStats.hit(1);
                    }
                } else {
                    if (haveTurned) {
                        turnStatsFar.hit(0);
                    } else {
                        turnStatsFar.hit(1);
                    }
                }
                if (longTurn) {
                    turnLengthStats.hit(0);
                } else {
                    turnLengthStats.hit(1);
                }
            }
            num1on1GotHit++;
        }
    }

    public void onRobotDeath(RobotDeathEvent e) {
        if (e.getName().equals(gunTarget)) {
            gunTarget = null;
            gunTargetDistance = Double.MAX_VALUE;
        }
        if (e.getName().equals(evadeTarget)) {
            evadeTarget = null;
        }
        MovableData bot = (MovableData)robots.get(e.getName());
        if (bot != null) {
            bot.isAlive = false;
        }
        robots.put(e.getName(), bot);

        // switched to 1-on-1 mode?
        if (getOthers() == 1) {
            // initialize 1-on-1!
            targetBot = null;
            updateFiringMode();
            patternAnalyzer.addHistory(targetHistory);
            maxVelocity = 0.1;
            setMaxVelocity(maxVelocity);
        }
    }

    public void onDeath(DeathEvent e) {
        // we got killed - by whom?
        if (evadeTarget == null) {
            return;
        }

        StaticBotData targetStats = (StaticBotData)statistics.get(evadeTarget);
        if (targetStats == null) {
            targetStats = new StaticBotData();
        }
        targetStats.numKilledMe++;
        statistics.put(evadeTarget, targetStats);

        storeStatistics();
    }

    public void onWin(WinEvent e) {
        maxVelocity = 0.0;
        setMaxVelocity(maxVelocity);
        ahead = 0;
        setAhead(ahead);
        storeStatistics();
        while (true) {
            execute();
        }
    }

    public void onSkippedTurn(SkippedTurnEvent e) {
        System.out.println("SKIPPED TURN!");
        Logger.warn("SKIPPED TURN!");
    }

    private void storeStatistics() {
        System.out.println("opponent's hit ratio: "
                + Math.round((double)num1on1GotHit / num1on1GotFiredAt * 100.0));
        System.out.println("our hit ratio: "
                + Math.round((double)num1on1BulletHit / num1on1Fired * 100.0));

        System.out.println("Firing statistics: ");
        for (int i = 0; i < VirtualBlotBullet.NUM_FIRING_MODES; i++) {
            System.out.println("FM " + i + ": " + Math.round(100.0 * num1on1Hit[i] / num1on1Fired));
        }
        System.out.println();

        File f = getDataFile("stats.log");
        try {
            PrintWriter pw = new PrintWriter(new RobocodeFileWriter(f));
            Iterator iter = statistics.keySet().iterator();
            while (iter.hasNext()) {
                String botName = (String)iter.next();
                String strLine = botName + "|";
                StaticBotData stats = (StaticBotData)statistics.get(botName);
                if (stats == null) {
                    continue;
                }
                strLine += stats.getHitRatioString();
                pw.println(strLine);
                pw.flush();
            }
            pw.close();
        } catch (IOException e) {
            e.printStackTrace(System.out);
        }
    }

    private void loadStatistics() {
        File f = getDataFile("stats.log");
        try {
            BufferedReader br = new BufferedReader(new FileReader(f));
            String strLine = br.readLine();
            while (strLine != null) {
                int firstSpace = strLine.indexOf("|");
                String botName = strLine.substring(0, firstSpace);
                StaticBotData stats = new StaticBotData(
                        strLine.substring(firstSpace + 1));
                statistics.put(botName, stats);

                strLine = br.readLine();
            }
            br.close();
        } catch (IOException e) {
            e.printStackTrace(System.out);
        }
    }

    private double checkBorder(double forward) {
        double newX = getX() + forward * Math.sin(getHeadingRadians());
        double newY = getY() + forward * Math.cos(getHeadingRadians());
        if (newX < 75.0 || newX > getBattleFieldWidth() - 75.0
                || newY < 75.0 || newY >= getBattleFieldHeight() - 75.0) {
            return -forward;
        }
        return forward;
    }

    private double getPerpendicularCorrection(MovableData targetBot, double wishedAngle) {
        if (targetBot == null) {
            return 0.0;
        }
        double angle = 0.0;
        if (meleeAhead > 0) {
            // normal: moving forward
            angle = BotMath.normalizeRelativeAngle(
                    targetBot.bearing - getHeadingRadians());

        } else {
            // moving backwards
            angle = BotMath.normalizeRelativeAngle(
                    targetBot.bearing - (Math.PI + getHeadingRadians()));
        }

        if (angle > 0.0) {
            return angle - wishedAngle;
        } else {
            return angle + wishedAngle;
        }
    }

    private double getPerpendicularCorrection1on1(MovableData targetBot, double wishedAngle) {
        if (targetBot == null) {
            return 0.0;
        }
        double angle = 0.0;
        if (ahead > 0) {
            // normal: moving forward
            angle = BotMath.normalizeRelativeAngle(
                    targetBot.bearing - getHeadingRadians());

        } else {
            // moving backwards
            angle = BotMath.normalizeRelativeAngle(
                    targetBot.bearing - (Math.PI + getHeadingRadians()));
        }

        if (angle > 0.0) {
            return angle - wishedAngle;
        } else {
            return angle + wishedAngle;
        }
    }
}
