package tjk.universe;

import robocode.*;
import tjk.universe.BracketHist.HistType;
import tjk.utils.FastTrig;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Iterator;

// Testing MovementPredictor
import tjk.utils.MovementPredictor;
import tjk.utils.MovementPredictor.*;

/**
 * Bot - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 */

public class Bot
{
	
	private String name = "";
	
	private Point2D.Double location,oldLocation;
	
	private double v,oldV,heading,energy,eDrop,heat,coolingRate,eCW;
	
	private boolean named = false;
	
	private boolean firedShot = false;
	
	private long zeroTime,vChangeTime;
	
	private ArrayList<Wave> waves = new ArrayList<Wave>(35);
	
	private Bot enemy;
	
	private WaveStats stats;
	
	private long tempZtime = 0;
	
	private long tempVCtime = 0;
	
	private Bot meOneAgo = this;
	
	private Bot meTwoAgo = this;
	
	public Bot(Point2D.Double L, Point2D.Double OL, double vee, double ov, double h, double e, double eD, 
					double ht, double cR, double ecw, boolean fS, long zT, long vCT) {
		this.location = new Point2D.Double(L.getX(),L.getY());
		this.oldLocation = new Point2D.Double(OL.getX(),OL.getY());
		this.v = vee;
		this.oldV = ov;
		this.heading = h;
		this.energy = e;
		this.eDrop = eD;
		this.heat = ht;
		this.coolingRate = cR;
		this.eCW = ecw;
		this.firedShot = fS;
		this.zeroTime = zT;
		this.vChangeTime = vCT;
	}
	
	public Bot(Universe u) {
		location = new Point2D.Double(0.0,0.0);
		oldLocation = new Point2D.Double(0.0,0.0);
		v = 0.0;
		oldV = 0.0;
		heading = 0.0;
		energy = 100.0;
		eDrop = 0.0;
		heat = 3.0;
		coolingRate = 0.1;
		eCW = 1.0;
		stats = new WaveStats();
		zeroTime = Universe.getTime();
		vChangeTime = Universe.getTime();
	}
	
	public Bot clone()
	{
		return new Bot(this.location, this.oldLocation, this.v, this.oldV, this.heading, 
					this.energy, this.eDrop, this.heat, this.coolingRate, this.eCW, 
					this.firedShot, this.zeroTime, this.vChangeTime);		
	}
	
	public void update(AdvancedRobot r, ScannedRobotEvent e) {
		if ( !named )
		{
			coolingRate = r.getGunCoolingRate();
			name = e.getName();
			named = true;
		}
	
		meTwoAgo = meOneAgo;
		meOneAgo = clone();
	
		double x = r.getX() + e.getDistance()*FastTrig.sin(r.getHeadingRadians() + e.getBearingRadians());
		double y = r.getY() + e.getDistance()*FastTrig.cos(r.getHeadingRadians() + e.getBearingRadians());
		oldLocation.setLocation(location);
		location = new Point2D.Double(x,y);
		heading = e.getHeadingRadians();
		eDrop = energy-e.getEnergy();
		firedShot = checkFired( eDrop , v);
		if (firedShot)
		{
			heat = Rules.getGunHeat(eDrop);
		}
		if ( heat > 0.0 ) 
		{
			heat = Math.max(heat-coolingRate,0.0);
		}
		energy = e.getEnergy();
		oldV = v;
		v = e.getVelocity();
		if ( Math.abs(v - 0.0) <= 0.0001  )
		{
			tempZtime = getTime();
		}
		zeroTime = getTime() - tempZtime;
		if ( Math.abs(v - oldV) >= 0.001 )
		{
			tempVCtime = getTime();
		}
		vChangeTime = getTime() - tempVCtime;
		eCW = calcEnemyCW();
	}
	
	public void update(AdvancedRobot r) {
		if ( !named )
		{
			coolingRate = r.getGunCoolingRate();
			name = r.getName();
			named = true;
		}
	
		meTwoAgo = meOneAgo;
		meOneAgo = clone();
	
		oldLocation.setLocation(location);
		location = new Point2D.Double(r.getX(),r.getY());
		heading = r.getHeadingRadians();
		eDrop = energy-r.getEnergy();
		firedShot = checkFired( eDrop , v );
		if (firedShot)
		{
			heat = Rules.getGunHeat(eDrop);
		}
		if ( heat > 0.0 ) 
		{
			heat = Math.max(heat-coolingRate,0.0);
		}
		energy = r.getEnergy();
		oldV = v;
		v = r.getVelocity();
		if ( Math.abs(v - 0.0) <= 0.0001  )
		{
			tempZtime = getTime();
		}
		zeroTime = getTime() - tempZtime;
		if ( Math.abs(v - oldV) >= 0.001 )
		{
			tempVCtime = getTime();
		}
		vChangeTime = getTime() - tempVCtime;
		eCW = calcEnemyCW();
		/*
		 * Heat sim is off until after first bullet of the round is fired. 
		 * Acceptable problem.
		 *
		if ( Math.abs(heat-r.getGunHeat()) > 0.05 )
		{
			System.out.println("Heat sim needs improvement! " + heat + " vs " + r.getGunHeat() );
		}
		*/
	}
	
	public void updateWaves()
	{	
		
		for ( Iterator<Wave> itr = waves.iterator(); itr.hasNext(); )
		{
			Wave w = itr.next();
			w.update();
			if ( w.getState() == Wave.DEAD )
			{
				// Load stale waves into WaveStats
				stats.addWave(w);
				
				// Kill stale waves.
				itr.remove();
			}
		}
	
	}
	
	public void clearWaves()
	{
		waves.clear();
	}
	
	public void removeWave(Point2D.Double hitLocation)
	{
		double minDistance = Double.POSITIVE_INFINITY;
		Wave removeWave = null;
		for ( Wave w: waves)
		{
			if ( w.real )
			{
				double testD = Math.abs( w.getRadius() - hitLocation.distance(w.getLocation()) );
				if ( testD < minDistance )
				{
					minDistance = testD;
					removeWave = w;
				}
			}
		}
	
		waves.remove(removeWave);
	
	}
	
	public BracketHist getBracketHist(HistType type, Wave w, int closestK, double realWeight, boolean drop)
	{
		BracketHist mainBrackets = w.getMainBracketHist();
		if ( mainBrackets == null )
		{
			BracketHist bh = stats.getBracketHist(type, w,closestK,realWeight,drop);
			w.setMainBracketHist(bh);
			return bh;
		}
		return mainBrackets;
	}
	
	public Wave fireVWave(double pow, long time)
	{
		return fireWave(false,pow,time);
	}

	public Wave fireWave(double pow, long time)
	{
		return fireWave(true,pow,time);
	}
	
	public Wave fireOldWave(boolean real, double power)
	{
		Wave w = getOldWave(real,power);
		fireWave(w);
		return w;
	}
	
	// get a wave from one tick ago with data from two ticks ago.
	public Wave getOldWave(boolean real, double power)
	{
		Bot twoOldMe = getMeTwoAgo();
		Bot twoOldE = enemy.getMeTwoAgo();
		//Bot oneOldMe = getMeOneAgo();
		// MCM and stat segments from two ticks ago...
		double[] mcm = enemyMCM(twoOldMe, twoOldE, Rules.getBulletSpeed(eDrop));
		double[] situation = situation(twoOldMe, twoOldE, true, eDrop);
		double cw = getMeTwoAgo().enemyCW();
		
		// Wave fired one tick ago.
		long firetime = Universe.getTime()-1;
		
		//Make and fire wave from myself one tick ago.
		Wave newwave = new Wave(real, getMeOneAgo(), enemy, firetime, power, cw, mcm, situation);
		return newwave;
	}
	
	// Firingwave for enemy... two ticks ago.. old data.
	public Wave checkFireWave()
	{
		if ( firedShot )
		{
			return fireOldWave(true,eDrop);
		}
		return null;
	}
	
	public Wave getWave(boolean real, double pow, long time)
	{
		return new Wave(real, this, enemy, time, pow, eCW, enemyMCM(Rules.getBulletSpeed(pow)), situation(real,pow));
	}
	
	private Wave fireWave(boolean real, double pow, long time)
	{
		Wave w = getWave(real,pow,time);
		fireWave( w );
		return w;
	}
	
	public void fireWave(Wave w)
	{
		if ( !waves.contains(w) )
		{
			waves.add(w);
		}
	}
	
	public Iterator<Wave> getWavesIterator()
	{
		return waves.iterator();
	}
	
	public boolean getFiredShot()
	{
		return firedShot;
	}
	
	public double bfH()
	{
		return Universe.bfH();
	}

	public double bfW()
	{
		return Universe.bfW();
	}
			
	public boolean checkFired(double eDrop, double V) {
		if ( eDrop >= 0.01 && eDrop <= 3.0 )
		{
			if ( Math.abs(Rules.getWallHitDamage(v)-eDrop) < 0.01 && ( location.getX() < 19 || location.getY() < 19 || location.getX() > Universe.bfW()-19 || location.getY() > Universe.bfH()-19 ) )
			{
				return false;
			}
			if ( Math.abs(Rules.ROBOT_HIT_DAMAGE-eDrop) < 0.01 && location.distanceSq(enemy.getLocation()) <= 2592  )
			{
				return false;
			}
			return true;
		}	
		return false;
	}
	
	public void setHeat(double h)
	{
		heat = h;
	}
	
	public double getHeat()
	{
		return heat;
	}
	
	public void setEnemy(Bot e)
	{
		enemy = e;
	}
	
	public Bot getEnemy()
	{
		return enemy;
	}
	
	public String getName() {
		return name;
	}
	
	public Point2D.Double nextLocation() {
		double newX = location.getX() + v*FastTrig.sin(heading);
		double newY = location.getY() + v*FastTrig.cos(heading);
		return new Point2D.Double(newX,newY);
	}

	public double headingToEnemy()
	{
		return headingFromTo(this,enemy);
	}

	public static double headingFromTo(Bot a, Bot b)
	{
		return FastTrig.atan2( b.getX()-a.getX() , b.getY()-a.getY() );
	}

	private double calcEnemyCW()
	{
		double v2 = enemy.getV();
		if ( Double.compare(v2,0.0) == 0.0 )
		{
			return 1.0;
		}
		double a = v2*FastTrig.cos( enemy.getHeading() - headingToEnemy() - FastTrig.HALF_PI );
		if ( Double.compare(a,0.0) >= 0 )
		{
			return 1.0;
		}
		else
		{
			return -1.0;
		}
	}
	
	public double closestLateralWallD()
	{
		return location.distance(closestLateralWallPoint());
	}
	
	public Point2D.Double closestLateralWallPoint()
	{
		double h = heading + FastTrig.HALF_PI;
		double x0 = location.getX();
		double y0 = location.getY();
		
		// Check left wall.. x = 18
		double T = (17.5 - x0)/FastTrig.sin(h);
		Point2D.Double answer = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
		double MinDist = location.distance(answer);
		
		// Check right wall.. x = universe.bfW()-18
		T = (Universe.bfW() - 17.5 - x0)/FastTrig.sin(h);
		Point2D.Double test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
		double testD = location.distance(test);
		if ( testD < MinDist )
		{
			MinDist = testD;
			answer = test;
		}
		
		// Check bottom wall.. y = 18
		T = (17.5 - y0)/FastTrig.cos(h);
		test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
		testD = location.distance(test);
		if ( testD < MinDist )
		{
			MinDist = testD;
			answer = test;
		}
	
		// Check top wall.. y = universe.bfH() - 18
		T = (Universe.bfH() - 17.5 - y0)/FastTrig.cos(h);
		test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
		testD = location.distance(test);
		if ( testD < MinDist )
		{
			MinDist = testD;
			answer = test;
		}
		
		return answer;
	}
	
	public double closestForwardWallD()
	{
		return location.distance(closestForwardWallPoint());
	}
	
	public Point2D.Double closestForwardWallPoint()
	{
		return closestWallPointInDirection(this.heading);
	}
	
	public double closestBackwardWallD()
	{
		return location.distance(closestBackwardWallPoint());
	}
	
	public Point2D.Double closestBackwardWallPoint()
	{
		return closestWallPointInDirection(this.heading+Math.PI);
	}
	
	public Point2D.Double closestWallPointInDirection(double h2)
	{
		double h = h2;
		if ( getV() < 0.0 )
		{
			h += Math.PI;
		}
		double x0 = location.getX();
		double y0 = location.getY();
		
		double T = 0.0;
		Point2D.Double answer = new Point2D.Double(-5000.0, -5000.0);
		double MinDist = Double.POSITIVE_INFINITY;
		Point2D.Double test = new Point2D.Double(-5000.0, -5000.0);
		double testD = Double.POSITIVE_INFINITY;
		
		// Check left wall.. x = 18
		T = (17.5 - x0)/FastTrig.sin(h);
		if ( T >= 0 )
		{
			test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
			testD = location.distance(test);
			if ( testD < MinDist )
			{
				answer = test;
				MinDist = testD;
			}
		}
		
		// Check right wall.. x = universe.bfW()-18
		T = (Universe.bfW() - 17.5 - x0)/FastTrig.sin(h);
		if ( T >= 0 )
		{
			test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
			testD = location.distance(test);
			if ( testD < MinDist )
			{
				answer = test;
				MinDist = testD;
			}
		}
		
		// Check bottom wall.. y = 18
		T = (17.5 - y0)/FastTrig.cos(h);
		if ( T >= 0 )
		{
			test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
			testD = location.distance(test);
			if ( testD < MinDist )
			{
				answer = test;
				MinDist = testD;
			}
		}
	
		// Check top wall.. y = universe.bfH() - 18
		T = (Universe.bfH() - 17.5 - y0)/FastTrig.cos(h);
		if ( T >= 0 )
		{
			test = new Point2D.Double(x0+T*FastTrig.sin(h), y0+T*FastTrig.cos(h));
			testD = location.distance(test);
			if ( testD < MinDist )
			{
				answer = test;
				MinDist = testD;
			}
		}
		
		return answer;
	}
	
	public double[] enemyMCM(double vb)
	{
		return enemyMCM(this,enemy,vb);
	}
	
	public double[] enemyMCM(Bot source, Bot target, double vb)
	{
		//TODO: This is (slightly less) naive maxescapeangle.  Make it better with Sim.
		/*
		double heading = headingToEnemy();
		final double d0 = getLocation().distance(getEnemy().getLocation());
		final int vBot0 = (int)Math.round(Math.abs(getEnemy().getV()));
		
		int t = 0;
		double botD = 0.0;
		while ( vb*t*vb*t < d0*d0 + botD*botD )
		{
			botD = tjkUtil.distArrayForward[vBot0][t];
			t++;
		}
		double max = Utils.normalRelativeAngle(FastTrig.atan(Math.abs(botD)/d0));
		
		t = 0;
		botD = 0.0;
		while ( vb*t*vb*t < d0*d0 + botD*botD )
		{
			botD = tjkUtil.distArrayBackward[vBot0][t];
			t++;
		}
		double min = Utils.normalRelativeAngle(FastTrig.atan(Math.abs(botD)/d0));
		
		double[] r = {min,heading,max};
		return r;
		*/
		Point2D.Double location = source.getLocation();
		WaveEndPredictionCondition ender = new WaveEndPredictionCondition(location, 0.0, vb);
		PredictionStatus enemyPred = new PredictionStatus(target.getX(), target.getY(), target.getHeading(), target.getV(), 0);
		
		double[] MEAangles = MovementPredictor.predictMEAangles(enemyPred, ender, location);
		
		MEAangles[0] = Math.abs(MEAangles[0]);
		MEAangles[2] = Math.abs(MEAangles[2]);
		
		// deBroglie uses ccw,headon,cw
		if ( eCW >= 0 ) 
		{
			double temp = MEAangles[2];
			MEAangles[2] = MEAangles[0];
			MEAangles[0] = temp;
		}
		
		return MEAangles;
		
	}

	public double[] situation(boolean real, double pow)
	{
		return Segmentation.situation(this,enemy,real,pow);
	}
	
	public double[] situation(Bot source, Bot e, boolean real, double pow)
	{
		return Segmentation.situation(source,e,real,pow);
	}
	
	public long timeSinceVZero()
	{
		return zeroTime;
	}

	public long timeSinceVChange()
	{
		return vChangeTime;
	}
	
	public long getTime()
	{
		return Universe.getTime();
	}
	
	public double getEnergy()
	{
		return energy;
	}
	
	public double enemyCW()
	{
		return eCW;
	}
	
	public Point2D.Double getLocation() {
		return location;
	}

	public Point2D.Double getOldLocation() {
		return oldLocation;
	}
				
	public double getX() {
		return location.getX();
	}
	
	public double getY() {
		return location.getY();
	}
	
	public double getV() {
		return v;
	}
	
	public double getOldV() {
		return oldV;
	}
	
	public double getA() {
		return v-oldV;
	}
	
	public int getStatSize()
	{
		return stats.getSize();
	}
	
	public WaveStats getStats()
	{
		return stats;
	}
	
	public double getHeading() {
		return heading;
	}
	
	public Bot getMeOneAgo()
	{
		return meOneAgo;
	}
	
	public Bot getMeTwoAgo()
	{
		return meTwoAgo;
	}
	
	public ArrayList<Wave> getWaves()
	{
		return waves;
	}

}