package sch;

import robocode.*;

/**
 * Functions used to calculate where the robot should be at a certain moment.<br>
 * This intercept method uses an approximate linear intercept.
 */
public class Intercept implements Constants {
	
	public Intercept () {}
	
	/**
	 * Return the long name of the intercept method
	 * @return		name of method
	 */
	public String getName() {return "Simple Approximized Linear Intercept";}
	
	/**
	 * Return the key of the intercept method
	 * @return		key of method
	 */
	public String getKey() {return "LinearApprox";}
	public int getComplexity() {return 2;}

	/**
	 * It's an helper function that calculates when a thing travelling at a certain speed will take to go from a point to another
	 * @param		startPos	starting position
	 * @param		endPos		end position
	 * @param		speed		the speed of the thing
	 * @param		now			start time
	 * @return					end time
	 */	
	protected double calcTime(Point startPos, Point endPos, double speed, double now) {
		double time = now + BotMath.getRange(startPos,endPos)/speed;
		return time;
	}

	/**
	 * Calculate the position where the bot will be at a future time.<br>
	 * This method should be overwritten in order to provide more powerfull targeting.<br>
	 * This use an approximate linear targeting.
	 * @param		myBotPos				position of my bot
	 * @param		targetPos				current position of the target
	 * @param		now						current time
	 * @param		targetPositionTiming	time of the target position
	 * @param		targetHeading			heading of the target
	 * @param		targetSpeed				speed of the target
	 * @param		bulletSpeed				speed of the bullet fired
	 * @param		targetAngSpeed			angular speed of the target
	 * @return								estimated position of the target
	 */
	 
	public Point calculate(	Point myBotPos,
							Point targetPos,
							double now,
							double targetPositionTiming,
							double targetHeading,
							double targetSpeed,
							double bulletSpeed,
							double targetAngSpeed) {

		double supposedImpactTime;

		supposedImpactTime=calcTime(myBotPos,targetPos,bulletSpeed,now);
		Point newPos=guessPosition(targetPos,targetHeading,targetSpeed,0,targetPositionTiming,supposedImpactTime);
		return newPos;
	}

	/**
	 * Helper function that calculates where an object will be at a certain time.<br>
	 * This one calculate the position based on the current heading and speed.
	 * @param		targetPosition			current position of the target
	 * @param		targetHeading			heading of the target
	 * @param		targetSpeed				speed of the target
	 * @param		targetPositionTiming	time of the target position
	 * @param		estimatedTime			time for which you want the position
	 * @return								future position
	 */
	protected Point guessPosition(Point targetPosition,
								double targetHeading,
								double targetSpeed,
								double targetAngSpeed,
								double targetPositionTiming,
								double estimatedTime) {
		
		double diffTime = estimatedTime - targetPositionTiming;
		
		double x = targetPosition.x+Math.sin(targetHeading)*targetSpeed*diffTime;
		double y = targetPosition.y+Math.cos(targetHeading)*targetSpeed*diffTime;
		
		return new Point(x,y);
	}
	
}