//--------------------------------------------------------------------------
// $Id: LindadaBot.java,v 1.1 2003/08/29 09:18:58 erning Exp $
//--------------------------------------------------------------------------
// Copyright (c) 2000-2003 Dragon Software Corp. All rights reserved.
//
// Please refer to COPYRIGHT for notices regarding the confidential
// and proprietary nature of this source code.
//--------------------------------------------------------------------------

package net.dragonsoft.robocoding.lindada;

import robocode.*;
import net.dragonsoft.robocoding.Enemy;
import net.dragonsoft.robocoding.EnemyState;
import net.dragonsoft.robocoding.util.Position;
import net.dragonsoft.robocoding.AbstractBot;

import java.util.LinkedList;
import java.util.Iterator;
import java.util.List;
import java.util.Hashtable;
import java.util.zip.GZIPOutputStream;
import java.util.zip.GZIPInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.FileInputStream;

public class LindadaBot extends AbstractBot
{
    public LindadaBot()
    {
        //_statsTable = new Hashtable();
    }

    public void run()
    {
        _isHitEnemy = false;
        _isEnemyFired = false;

        _waveList = new LinkedList();

        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
        setAdjustRadarForRobotTurn(true);

        do
        {
            setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
            execute();
        }
        while (_enemy == null);

        // starting fire
        aimAt(_enemy.getCurrentState().angle);
        shoot(0.1);
        execute(); // waste one tick?

        while (true)
        {
            EnemyState es = _enemy.getCurrentState();
            if (es.time != getTime())
            {   // lose the enemy, search again.
                setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
                _isHitEnemy = false;
                _isEnemyFired = false;
                execute();
                continue;
            }

            // radar focus on the enemy
            focus(es.angle);

            // To hit the enemy by vehicle
            if (es.energy < 0.6d && es.myEnergy > 1.2d)
            {
                move(es.x, es.y);
                _isHitEnemy = false;
                _isEnemyFired = false;
                execute();
                continue;
            }

            // the enemy fired?
            if (es.energyChange >= -3.0d && es.energyChange < 0d && !_isHitEnemy)
            {
                _isEnemyFired = true;
            }

            // select a best power and a wave
            double power = Math.min(Math.min(es.energy - 0.6d, es.myEnergy - 1.2d) / 4d + 0.1d, 3d);
            _waveList.add(new Wave(_enemy, power));
            // update wave that hit enemy
            updateStats();

            // smoke moving schema
            if (Math.abs(getDistanceRemaining()) <= _moveRemaining && _isEnemyFired)
            {
                _moveRemaining = Math.random() * 50d + 25d;
                double distance = es.distance;
                double x;
                double y;
                do
                {
                    double turnAngle = (Math.random() * 1.6d - 0.8d);
                    if (turnAngle > 0 && turnAngle < 0.1) turnAngle += 0.1;
                    else if (turnAngle < 0 && turnAngle > -0.1) turnAngle -= 0.1;

                    double moveAngle = es.angle + turnAngle;
                    double radius = distance;
                    if (radius < 600)
                    {
                        // increace the radius if too close to the enemy
                        radius /= Math.cos(turnAngle);
                    }

                    x = es.x - Math.sin(moveAngle) * radius;
                    y = es.y - Math.cos(moveAngle) * radius;

                    // reduce the around radius if the selected position out of range
                    distance -= 10d;

                    // loop until target position selected
                }
                while (isOutOfField(x, y));
                move(x, y);
            }
            setMaxVelocity(Math.abs(getTurnRemaining()) > 45d ? 0d : 8d);

            // too close to the enemy fire directly
            if (es.distance < 180d)
            {
                aimAt(es.angle);
                shoot(3.0d);
                _power = 0d;
                _isHitEnemy = false;
                _isEnemyFired = false;
                execute();
                continue;
            }

            // fire if aimed
            if (_power > 0d && Math.abs(getGunTurnRemaining()) < 0.1d)
            {
                shoot(_power);
                _power = 0d;
            }

            // select a shoot position
            if (getGunHeat() < (getGunCoolingRate() * 4) && Math.abs(_power) < 0.1d && Math.abs(getGunTurnRemaining()) < 0.1d)
            {
                short bestValue = 0;
                double bestAngle = 0d;
                String statsKey = getStatsKey(es);
                short[] stats = (short[])_statsTable.get(statsKey);
                if (stats != null)
                {
                    for (int i = 0; i < 90; i++)
                    {
                        if (stats[i] > bestValue)
                        {
                            double angle = Math.toRadians(i - 45d);
                            if (!isOutOfField(Position.calcPosition(es.myX, es.myY, es.angle + angle, es.distance)))
                            {
                                bestValue = stats[i];
                                bestAngle = angle;
                            }
                        }
                    }
                }
                aimAt(es.angle + bestAngle);
                if (bestValue <= 0)
                    shoot(Math.min(1.1d, power));
                else
                    _power = power;
                //
            }


            // clean flags
            _isHitEnemy = false;
            _isEnemyFired = false;

            execute();
        }
    }

    protected void updateStats()
    {
        for (Iterator i = _waveList.iterator(); i.hasNext();)
        {
            Wave wave = (Wave)i.next();
            wave.update(_enemy);
            int waveStatus = wave.getStatus();
            if (waveStatus == Wave.STATUS_TRAVELING) continue;
            i.remove();
            if (waveStatus != Wave.STATUS_HIT) continue;

            EnemyState ses = wave.getSentEnemyState();
            //EnemyState hes = wave.getHitEnemyState();

            //double power = wave.getPower();

            //double maxAngle = Math.asin(8d / (20 - 3 * power));
            double turnAngle = wave.getTurnAngle();
            // turnAngleIndex
            int tai = (int)Math.round(Math.toDegrees(turnAngle)) + 45;
            if (tai < 0 || tai >= 90) return;

            String statsKey = getStatsKey(ses);
            short[] stats = (short[])_statsTable.get(statsKey);
            if (stats == null)
            {
                stats = new short[90];
                _statsTable.put(statsKey, stats);
            }

            //stats[pi][tai] += (power > 1 ? 4 * power + 2 * (power - 1) : 4 * power);
            stats[tai]++;
        }
    }

    protected void saveStats()
    {
        try
        {
            ObjectOutputStream out = new ObjectOutputStream(new GZIPOutputStream(new RobocodeFileOutputStream(getDataFile(_enemy.getName() + ".gz"))));
            out.writeObject(_statsTable);
            out.close();
            System.out.println("Save data [" + _statsTable.size() +"] to file successfully");
        }
        catch (Exception ex)
        {
            System.out.println("Error while save data to file");
            //ex.printStackTrace(out);
        }
    }

    protected void loadStats()
    {
        try
        {
            ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new FileInputStream(getDataFile(_enemy.getName() + ".gz"))));
            _statsTable = (Hashtable)in.readObject();
            in.close();
            System.out.println("Load data [" + _statsTable.size() +"] from file successfully");
        }
        catch (Exception ex)
        {
            if (_statsTable == null)
            {
                _statsTable = new Hashtable();
                System.out.println("Use fresh data");
            }
        }
    }

    protected String getStatsKey(EnemyState es)
    {
        // headingChangeIndex
        int hci = Math.round(es.headingChange) == 0 ? 0 : es.headingChange < 0 ? 1 : 2;
        // velocityChangeIndex
        int vci = Math.round(es.velocityChange) == 0 ? 0 : es.velocity < 0 ? 1 : 2;
        // velocityIndex
        int vi = Math.round(es.velocity) == 0 ? 0 : es.velocity < 0 ? 1 : 2;
        // distanceIndex
        int di = (int)Math.round(es.distance / 50);
        // distanceToCenterIndex
        int d2ci = (int)Math.round(distanceToCenter(es.x, es.y) / 100);

        StringBuffer sb = new StringBuffer();
        sb.append((char)hci);
        sb.append((char)vci);
        sb.append((char)vi);
        sb.append((char)di);
        sb.append((char)d2ci);

        return sb.toString();
    }

    public void onScannedRobot(ScannedRobotEvent event)
    {
        if (_enemy == null)
        {
            _enemy = new SimpleEnemy(this, event.getName());
        }
        _enemy.update(event);
        if (_statsTable == null) loadStats();
    }

    public void onBulletHit(BulletHitEvent event)
    {
        // My bullet hitting the enemy
        _isHitEnemy = true;
    }

    public void onBulletHitBullet(BulletHitBulletEvent event)
    {
    }

    public void onBulletMissed(BulletMissedEvent event)
    {
    }

    public void onHitByBullet(HitByBulletEvent event)
    {
        // I'm hitting by a bullet
    }

    public void onHitRobot(HitRobotEvent event)
    {
    }

    public void onHitWall(HitWallEvent event)
    {
    }

    public void onRobotDeath(RobotDeathEvent event)
    {
    }

    public void onWin(WinEvent event)
    {
        saveStats();
        setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
        setTurnGunLeftRadians(Double.POSITIVE_INFINITY);
        setTurnRightRadians(Double.POSITIVE_INFINITY);
    }

    public void onDeath(DeathEvent event)
    {
        saveStats();
    }

    public void onSkippedTurn(SkippedTurnEvent event)
    {
        System.out.println("Skipped turn: " + getTime());
    }

    public void onCustomEvent(CustomEvent event)
    {
    }

    protected static Enemy _enemy = null;
    protected static Hashtable _statsTable = null;

    protected double _power = 0d;
    protected double _moveRemaining = 53d;

    protected boolean _isHitEnemy;
    protected boolean _isEnemyFired;

    protected List _waveList;

    //protected Position _nextPos;

}