/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* this class is responsible for scanning an enemy
 * in a one-on-one battle;
 * it locks on any scanned robot and uses
 * enemy object from parent robot for prediction
 */
public class Scanner
{
   /* angle that is taken as a margin around which the enemy is
    * scanned; it should be small enough to allow the radar compensate
    * tight body and gun turns and large enough to prevent loosing enemy
    */
   static final double scanAngle = Math.toRadians( 7.5 );

   // the robot for which this body logic is applicable
   AdvancedRobot robot;
   // the enemy that this scanner should track
   Enemy enemy;
   
   /* on creation provide a robot, for which this logic
    * is applicable and enemy robot to track;
    */
   public Scanner( AdvancedRobot robot, Enemy enemy )
   {
      this.robot = robot;
      this.enemy = enemy;
   }  
   
   /* rotates the scanner to sweep around enemy;
    * if the enemy data is not up to date the scanner is rotated around;
    * \param nextRobotPosition provide robot position predicted for the next frame;
    *        it is crucial for precise aiming
    */
   public void run( Vector2d nextRobotPosition )
   {
      if ( !enemy.isUpToDate() ) {
         // rotate radar as fast as you can if it is not turning
         if ( robot.getRadarTurnRemainingRadians() == 0 )
            robot.setTurnRadarRight( 360 );
      } else { // enemy data is up to date; track enemy
         // calculate relative enemy position
         Vector2d positionRelative = enemy.getData( 0 ).position;
         positionRelative.sub( nextRobotPosition );
         
         // calculate relative angle to enemy
         double angleRelative = positionRelative.angle();
         
         // compute where to the radar should turn
         double radarTurnAngle;
         double radarAngleToEnemy = Util.normalRelativeAngle( 
                                    angleRelative - robot.getRadarHeadingRadians() );
                                   
         if ( radarAngleToEnemy > 0 ) // radar is on the left of enemy
            radarTurnAngle = radarAngleToEnemy + scanAngle;
         else   // radar is on the right side of enemy
            radarTurnAngle = radarAngleToEnemy - scanAngle;
            
         // turn the radar            
         robot.setTurnRadarRightRadians( radarTurnAngle );
      }
   }
}
