package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;






/**
 * MadHatter - a robot by Simon Parker
 */
public class MadHatter extends AdvancedRobot
{

	boolean isLocked = false;
	int unsuccessfulScanCounter = 0;
	String targetName;

	double currentTargetPosX = 200;
	double currentTargetPosY = 200;
	double currentTargetHeading = 0;
	double currentTargetVelocity = 0;

	double currentVelocity = 0;

	double lastTargetHeading_deg = 0;
	double currentTargetAnglularVelocity = 0;

	private int bulletHitCount = 0;
	private int bulletMissCount = 0;

	double MOVE_DISTANCE = 150;
	final double MAX_MOVE_DISTANCE = 180;
	final double MIN_MOVE_DISTANCE = 120;
	final int MAX_VELOCITY = 8;
	final double RADIANS_PER_DEGREE = Math.PI / 180.0;
	final double DEGREES_PER_RADIAN = 180.0 / Math.PI;
	final double MIN_BULLET_POWER = 0.5;
	final double MAX_BULLET_POWER = 3.0;
	final int MAX_SCAN_FAIL_COUNT = 2;
	public static final double ROBOT_RADIUS = 45.0/2;
	final double ANGULAR_VELOCITY_DEG_PER_TURN = 10.0;
	final double OPTIMUM_TARGET_DISTANCE = 200.0;
	final double WALL_DEFLECTION_DISTANCE = 100.0;
	final double RADAR_TURN_ANGLE = 40;

	private JiggleAnalyser jiggleAnalyser = new JiggleAnalyser(this);

	/**
	 * run: MadHatter's default behavior
	 */
	public void run()
	{
		setAdjustGunForRobotTurn(true);
		setColors(Color.red,Color.red,Color.red);
		while (true)
		{
			if (unsuccessfulScanCounter<0) unsuccessfulScanCounter = 0;
			SetMovement();
			SetRotation();
			SetScan();
			SetGunRotation();
			execute();
		}
	}

	public void onSkippedTurn(SkippedTurnEvent event)
	{
		out.println("skipping turn");
	}

	private void SetMovement()
	{
		if ( getDistanceRemaining() == 0 )
		{
			MOVE_DISTANCE = Math.random()*(MAX_MOVE_DISTANCE - MIN_MOVE_DISTANCE) + MIN_MOVE_DISTANCE;

			if (currentVelocity <= 0)
			{
				setAhead(MOVE_DISTANCE);
				currentVelocity = MAX_VELOCITY;
			}
			else
			{
				setBack(MOVE_DISTANCE);
				currentVelocity = -MAX_VELOCITY;
			}
		}
	}

	private void SetRotation()
	{
		Coordinate targetPosition = new Coordinate(currentTargetPosX, currentTargetPosY);

		double predictedPosX = getX() + currentVelocity * Math.sin(Math.toRadians(getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN));
		double predictedPosY = getY() + currentVelocity * Math.cos(Math.toRadians(getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN));
		Coordinate turnRightPosition = new Coordinate(predictedPosX, predictedPosY);

		predictedPosX = getX() + currentVelocity * Math.sin(Math.toRadians(getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN));
		predictedPosY = getY() + currentVelocity * Math.cos(Math.toRadians(getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN));
		Coordinate turnLeftPosition = new Coordinate(predictedPosX, predictedPosY);

		double leftEval = evaluatePosition(turnLeftPosition, targetPosition, (getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN), currentTargetHeading);
		double rightEval = evaluatePosition(turnRightPosition, targetPosition, (getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN), currentTargetHeading);

		if (leftEval >= rightEval)
		{
			setTurnLeft(360);
		}
		else
		{
			setTurnRight(360);
		}
	}

	private void SetScan()
	{
		if (isLocked)
		{
			if (unsuccessfulScanCounter > MAX_SCAN_FAIL_COUNT)
			{
				unsuccessfulScanCounter = 0;
				isLocked = false;

				out.println("Lost Lock");
				return;
			}

			unsuccessfulScanCounter++;
			scan();

			//move radar to where the target should be next turn
			double predictedTargetPosX = currentTargetPosX
				+ currentTargetVelocity * Math.sin(currentTargetHeading * RADIANS_PER_DEGREE);

			double predictedTargetPosY = currentTargetPosY
				+ currentTargetVelocity * Math.cos(currentTargetHeading * RADIANS_PER_DEGREE);

			double predictedPosX = getX() + currentVelocity*Math.sin(RADIANS_PER_DEGREE * getHeading());
			double predictedPosY = getY() + currentVelocity*Math.cos(RADIANS_PER_DEGREE * getHeading());

			//calculate the scan angle required to maintain lock for next turn
			double targetRadarAngle = DEGREES_PER_RADIAN * Math.atan2(predictedTargetPosX-predictedPosX, predictedTargetPosY-predictedPosY);


			setTurnRadarRight(normalRelativeAngle(targetRadarAngle - getRadarHeading()));

		}
		else
		{
			// look for new robots
			setTurnRadarRight(720);
		}
	}

	private void SetGunRotation()
	{
		Intercept intercept;

		if (jiggleAnalyser.isUpdated)
		{
			//out.println("Jiggle " + jiggleAnalyser.isJiggling + ", Osc " + jiggleAnalyser.isOscilating);
		}

		if (jiggleAnalyser.isJiggling)
		{
			intercept = new JiggleIntercept(jiggleAnalyser);
		}
		else
		{
			intercept = new CircularIntercept();
		}

		// calculate target angle
		double posX = getX();
		double posY = getY();
		double dX = currentTargetPosX - posX;
		double dY = currentTargetPosY - posY;


		double bulletPower = calculateBulletPower
		(
			posX, posY, getGunHeading(),
			currentTargetPosX, currentTargetPosY, currentTargetHeading,
			currentTargetVelocity
		);

		intercept.calculate
		(
			posX, posY, currentTargetPosX, currentTargetPosY,
			currentTargetHeading, currentTargetVelocity, bulletPower, currentTargetAnglularVelocity
		);

		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - getGunHeading());
		// move gun to target angle
		setTurnGunRight(turnAngle);
		if 	(Math.abs(turnAngle) <= intercept.angleThreshold)
		{
			if 	(
					(intercept.impactPoint.x < 0) ||
					(intercept.impactPoint.x > getBattleFieldWidth()) ||
					(intercept.impactPoint.y < 0) ||
					(intercept.impactPoint.y > getBattleFieldHeight())
				)
			{
				//impact point off the field
				//out.println("T " + intercept.impactTime + ", x " + intercept.impactPoint.x + ", y " + intercept.impactPoint.y);
			}
			else
			{
				if (isLocked)
				{
					fire(bulletPower);
				}
			}
		}
		else
		{
			// gun not in position
			//out.println("ta " + turnAngle + ", th " + intercept.angleThreshold);
		}

	}

	private double calculateBulletPower
	(
		double xb,
		double yb,
		double bulletHeading,
		double xt,
		double yt,
		double targetHeading,
		double vt
	)
	{
		double dX = xt - xb;
		double dY = yt - yb;
		double distance = Math.sqrt(dX*dX+dY*dY);

		double power = MAX_BULLET_POWER - (distance-2*ROBOT_RADIUS)/300.0;
		power = Math.max(MIN_BULLET_POWER, power);
		power = Math.min(MAX_BULLET_POWER, power);
		//return 0.1;
		return power;
	}

	public void onHitWall(HitWallEvent e)
	{
		if (Math.abs(e.getBearing()) >= 90)
		{
			setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		else
		{
			setAhead(-MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
	}

	public void onHitRobot(HitRobotEvent e) {
		if (e.getBearing() > -90 && e.getBearing() < 90)
		{
			setBack(MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
		else
		{
			setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		execute();
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e)
	{
		unsuccessfulScanCounter--;

		double dX = currentTargetPosX - getX();
		double dY = currentTargetPosY - getY();
		double currentDistance = Math.sqrt(dX*dX+dY*dY);


		double distance = e.getDistance();
		if ((targetName != e.getName()) && ( distance > currentDistance) && isLocked)
		{
			return;
		}
		else
		{

			if (targetName != e.getName())
			{
				targetName = e.getName();
				lastTargetHeading_deg = e.getHeading(); //reset angular velocity
				jiggleAnalyser.reset();

				out.println("Locked onto " + targetName);
			}

			isLocked = true;
			targetName = e.getName();
			double angle_rad = getHeadingRadians() + e.getBearingRadians();

			currentTargetPosX = getX() + distance * Math.sin(angle_rad);
			currentTargetPosY = getY() + distance * Math.cos(angle_rad);
			currentTargetHeading = e.getHeading();
			currentTargetVelocity = e.getVelocity();
			currentTargetAnglularVelocity = currentTargetHeading - lastTargetHeading_deg;
			lastTargetHeading_deg = currentTargetHeading;

			Coordinate currentTargetPosition = new Coordinate(currentTargetPosX, currentTargetPosY);
			jiggleAnalyser.update(currentTargetPosition, currentTargetVelocity, currentTargetHeading);
		}

	}

	// Helper function
	public double normalRelativeAngle(double angle)
	{
		if (angle > -180 && angle <= 180)
			return angle;
		double fixedAngle = angle;
		while (fixedAngle <= -180)
			fixedAngle += 360;
		while (fixedAngle > 180)
			fixedAngle -= 360;
		return fixedAngle;
	}

	public void onRobotDeath(RobotDeathEvent e)
	{
		//Victory dance
		if (getOthers() == 0)
		{
			double bulletHitPercentage = 0;
			if ((bulletHitCount + bulletMissCount) != 0)
			{
				bulletHitPercentage = (double)bulletHitCount / (double)(bulletHitCount + bulletMissCount) * 100.0;
			}

			out.println("Hit " + bulletHitCount + ", Miss " + bulletMissCount + " (" + bulletHitPercentage + "%)");

			setAhead(0);
			turnRight(36000);

		}
	}

	public void onDeath(DeathEvent event)
	{
		double bulletHitPercentage = 0;
		if ((bulletHitCount + bulletMissCount) != 0)
		{
			bulletHitPercentage = (double)bulletHitCount / (double)(bulletHitCount + bulletMissCount) * 100.0;
		}

		out.println("Hit " + bulletHitCount + ", Miss " + bulletMissCount + " (" + bulletHitPercentage + "%)");
	}

	private double evaluatePosition(Coordinate robotPosition, Coordinate targetPosition, double robotHeading, double targetHeading)
	{
		final double WALL_DISTANCE_WEIGHT = 3.0;
		final double TARGET_BEARING_WEIGHT_MELEE = 0.0;
		final double TARGET_BEARING_WEIGHT_1_ON_1 = 0.1;
		final double CENTRE_DISTANCE_WEIGHT_MELEE = 1.0;
		final double CENTRE_DISTANCE_WEIGHT_1_ON_1 = 0.0;

		double dX = targetPosition.x - robotPosition.x;
		double dY = targetPosition.y - robotPosition.y;
		double distance = targetPosition.distanceFrom(robotPosition);

		double targetGoodness = - Math.abs(distance - OPTIMUM_TARGET_DISTANCE);

		double wallGoodnessX = 0;
		double wallGoodnessY = 0;
		if (robotPosition.x < WALL_DEFLECTION_DISTANCE)
		{
			wallGoodnessX = -WALL_DISTANCE_WEIGHT * (WALL_DEFLECTION_DISTANCE - robotPosition.x);
		}

		if (robotPosition.y < WALL_DEFLECTION_DISTANCE)
		{
			wallGoodnessY = -WALL_DISTANCE_WEIGHT * (WALL_DEFLECTION_DISTANCE - robotPosition.y);
		}

		if (robotPosition.x > (getBattleFieldWidth()-WALL_DEFLECTION_DISTANCE))
		{
			wallGoodnessX = WALL_DISTANCE_WEIGHT * (getBattleFieldWidth()-WALL_DEFLECTION_DISTANCE - robotPosition.x);
		}

		if (robotPosition.y > (getBattleFieldHeight()-WALL_DEFLECTION_DISTANCE))
		{
			wallGoodnessY = WALL_DISTANCE_WEIGHT * (getBattleFieldHeight()-WALL_DEFLECTION_DISTANCE - robotPosition.y);
		}

		double bearing_deg = normalRelativeAngle(robotHeading-Math.toDegrees(Math.atan2(dX,dY)));

		double bearingWeight = TARGET_BEARING_WEIGHT_MELEE;
		if (getOthers() == 1) bearingWeight = TARGET_BEARING_WEIGHT_1_ON_1;

		double bearingGoodness = - bearingWeight * Math.abs(Math.abs(Math.abs(bearing_deg) - 90) -90);

		Coordinate centrePosition = new Coordinate(getBattleFieldWidth()/2, getBattleFieldHeight()/2);
		double centreDistanceWeight = CENTRE_DISTANCE_WEIGHT_MELEE;
		if (getOthers() == 1) centreDistanceWeight = CENTRE_DISTANCE_WEIGHT_1_ON_1;
		double centreGoodness = robotPosition.distanceFrom(centrePosition) * centreDistanceWeight;


		return targetGoodness + wallGoodnessX + wallGoodnessY + bearingGoodness + centreGoodness;
	}

	public void onBulletHit(BulletHitEvent event)
	{
		bulletHitCount++;
	}

	public void onBulletHitBullet(BulletHitBulletEvent event)
	{
		bulletHitCount++;
	}

	public void onBulletMissed(BulletMissedEvent event)
	{
		bulletMissCount++;
	}

	/******************************************************************************/


}


