package ags.muse.movement;

import static robocode.util.Utils.normalRelativeAngle;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ags.muse.base.Rules;
import ags.muse.physics.RobotSimV2;
import ags.muse.recon.*;
import ags.muse.recon.waves.EWave;
import ags.muse.recon.waves.EWaveDetector;
import ags.muse.recon.waves.EnemyTargetingWatcher;
import ags.muse.base.actors.MovementActor;
import ags.util.PreciseIntersect;
import ags.util.Range;
import ags.util.points.AbsolutePoint;
import ags.util.points.RelativePoint;

public class CloakedEngine {
    private Rules rules;
    private RobotList robots;
    private EWaveDetector ewaves;
    private EnemyTargetingWatcher etargeting;

    public CloakedEngine(Rules rules, RobotList robots, EWaveDetector ewaves, EnemyTargetingWatcher etargeting) {
        this.rules = rules;
        this.robots = robots;
        this.ewaves = ewaves;
        this.etargeting = etargeting;
    }

    private double testDanger(TestPoint p, double low_danger) {
        AbsolutePoint point = p.point;
        double danger = 0;

        Collection<EnemyRobot> enemies = robots.getEnemies();
        if (enemies.size() > 1)
        {
            for (EnemyRobot e : enemies) {
                AbsolutePoint epoint = e.getLocation();
                double dist = epoint.distance(point);
                double cdist = e.getClosestDist();
                if (cdist + 100 < dist) {
                    continue;
                }

                double distratio = (dist/cdist);

                danger += 1.0/(distratio * dist);
                
                if (danger > low_danger) return danger;
            }
        }
        else if (enemies.size() == 1)
        {
            EnemyRobot e = enemies.iterator().next();
            AbsolutePoint epoint = e.getLocation();
            double dist = epoint.distance(point);

            danger += 10.0*Math.max(0, Math.abs(dist-400)-20);

            if (danger > low_danger) return danger;
        }

        if (robots.getEnemies().size() > 0)
        {
            danger += 0.01/Math.max(0, point.getX()-18);
            danger += 0.01/Math.max(0, point.getY()-18);
            danger += 0.01/Math.max(0, (rules.BATTLEFIELD_WIDTH-point.x)-18);
            danger += 0.01/Math.max(0, (rules.BATTLEFIELD_HEIGHT-point.y)-18);
        }
        if (danger > low_danger) return danger;
        
        danger += getPathDanger(p, low_danger - danger);
        if (danger > low_danger) return danger;

        return danger;
    }

    private static class SimWave {
        public final AbsolutePoint origin;
        public double radius;
        public PreciseIntersect intersect;
        public final double speed;
        public final EWave wave;
        
        public SimWave(EWave wave) {
            this.origin = wave.getOrigin();
            this.radius = wave.getRadius();
            this.speed = wave.getVelocity();
            this.wave = wave;
            this.intersect = new PreciseIntersect(origin.x, origin.y, wave.getZeroAngle(), wave.getMaximumEscapeAngle(), wave.getGFDirection());
        }
        
        public void update(double tx, double ty) {
            double lastradius = radius;
            radius += speed;
            intersect.processTick(tx, ty, radius, lastradius);
        }
    }
    
    private double getPathDanger(TestPoint p, double low_danger) {
        double danger = 0;
        RobotSimV2 sim = new RobotSimV2(robots.getSelf().getLocation(), robots.getSelf().getVelocity());
        AbsolutePoint endPoint = p.point;

        // Set up enemy locations
        ArrayList<Rectangle2D> enemyBounds = new ArrayList<Rectangle2D>();
        for (EnemyRobot e : robots.getEnemies()) {
            AbsolutePoint epoint = e.getLocation();
            enemyBounds.add(new Rectangle2D.Double(epoint.x-18, epoint.y-18, 36, 36));
        }

        // Set up waves for simulation
        List<SimWave> waves = new ArrayList<SimWave>();
        for (EWave wave : ewaves.getWaves()) {
            SimWave sWave = new SimWave(wave);
            waves.add(sWave);
            sWave.update(sim.getX(), sim.getY());
        }
        
        for (int i=0; true; i++) {
            // Simulate movement
            GotoResult gt = new GotoResult(sim.getX(), sim.getY(), sim.getHeading(), endPoint.x, endPoint.y);
            sim.simTick(gt.velocity, gt.turn);

            // Simulate waves
            for (SimWave sWave : waves) {
                sWave.update(sim.getX(), sim.getY());
            }
            // End the loop
            if (i >= 30) {
                break;
            }

            // Check out the danger
            Rectangle2D bounds = new Rectangle2D.Double(sim.getX()-22, sim.getY()-22, 44, 44);
            for (Rectangle2D eBound : enemyBounds) {
                if (eBound.intersects(bounds)) {
                    return Double.POSITIVE_INFINITY;
                }
            }
        }
        
        for (SimWave sWave : waves) {
            Range gfRange = sWave.intersect.getGFRange();
            if (gfRange != null) {
                double waveProbability = sWave.wave.getProbability();
                double waveDanger = etargeting.getDanger(sWave.wave, gfRange);

                danger += 1000 * waveProbability * waveDanger;
            }
        }

        return danger;
    }

    private class TestPoint {
        private final AbsolutePoint point;
        private final int dir;
        private TestPoint(AbsolutePoint point, int dir) {
            this.point = point;
            this.dir = dir;
        }
    }
    MovementPoly fwd_poly, rev_poly;
    TestPoint goal_point;
    public void run(MovementActor movementActor) {
        SelfRobot self = robots.getSelf();
        fwd_poly = new MovementPoly(rules, self.getLocation(), self.getVelocity(), 1, 30);
        rev_poly = new MovementPoly(rules, self.getLocation(), self.getVelocity(), -1, 30);

        // Create the test points
        List<TestPoint> test_points = new ArrayList<TestPoint>();
        test_points.add(new TestPoint(self.getLocation(), 0));
        if (goal_point != null) {
            if (goal_point.dir == 1 && !fwd_poly.contains(goal_point.point)) {
                // No longer reachable
            } else if (goal_point.dir == -1 && !rev_poly.contains(goal_point.point)) {
                // No longer reachable
            } else {
                test_points.add(goal_point);
            }
        }
        for (int i=0; i<30; i++) {
            AbsolutePoint p = fwd_poly.getRandomPoint();
            if (p != null) {
                test_points.add(new TestPoint(p, 1));
            }
        }
        for (int i=0; i<30; i++) {
            AbsolutePoint p = rev_poly.getRandomPoint();
            if (p != null) {
                test_points.add(new TestPoint(p, -1));
            }
        }

        // Test the points and set the goal point
        double low_danger = Double.POSITIVE_INFINITY;
        TestPoint best_point = goal_point;
        for (TestPoint point : test_points) {
            double danger = testDanger(point, low_danger);
            if (danger < low_danger) {
                best_point = point;
                low_danger = danger;
            }
        }
        goal_point = best_point;

        // If all points had infinite danger somehow, use 'state still' point
        if (goal_point == null)
        {
            goal_point = test_points.get(0);
        }

        // Go to the goal point
        goToGoal(movementActor);
    }
    
    private static final double HALFPI = Math.PI/2;
    private static class GotoResult  {
        public final double turn;
        public final double velocity;
        public GotoResult(double lx, double ly, double heading, double gx, double gy) {
            double velocity = getMaxVelocity(Point2D.distance(lx, ly, gx, gy));
            //double velocity = Point2D.distance(lx, ly, gx, gy);
            double turn = normalRelativeAngle(Math.atan2(gx - lx, gy - ly) - heading);
            if (gx == lx && gy == ly) {
                turn = 0;
            }
            if (Math.abs(turn) > HALFPI) {
                velocity *= -1;
                if(turn > 0) {
                    turn -= Math.PI;
                } else {
                    turn += Math.PI;
                }
            }
            this.velocity = velocity;
            this.turn = turn;
        }
        
        private static double getMaxVelocity(double distance) {
            final double decelTime =  Math.max(1,Math.ceil(
                //sum of 0... decelTime, solving for decelTime using quadratic formula
                (Math.sqrt((4*2/robocode.Rules.DECELERATION)*distance + 1) - 1)/2));
     
            final double decelDist = (decelTime / 2.0) * (decelTime-1) // sum of 0..(decelTime-1)
                * robocode.Rules.DECELERATION;
     
            return ((decelTime - 1) * robocode.Rules.DECELERATION) +
                ((distance - decelDist) / decelTime);
        }
    }

    private void goToGoal(MovementActor movementActor) {
        AbsolutePoint l = robots.getSelf().getLocation();
        RelativePoint v = robots.getSelf().getVelocity();
        GotoResult gotoResult = new GotoResult(l.x, l.y, v.direction,
                goal_point.point.x, goal_point.point.y);
        
        movementActor.setMove(gotoResult.velocity);
        movementActor.setTurnBody(gotoResult.turn);
        robots.getSelf().setIntention(rules, gotoResult.velocity, gotoResult.turn);
    }

    public void paint(Graphics2D g) {
        /*
        g.setColor(Color.RED);
        for (EWave wave : ewaves.getWaves()) {
            AbsolutePoint origin = wave.getOrigin();
            AbsolutePoint p = robots.getSelf().getLocation();
            PreciseIntersect intersect = new PreciseIntersect(origin.x, origin.y, wave.getZeroAngle(), wave.getMaximumEscapeAngle(), wave.getGFDirection());
            intersect.processTick(p.x, p.y, wave.getRadius(), wave.getRadius()-wave.getVelocity());
            Range range = intersect.getGFRange();
            if (range == null) continue;
            wave.drawSegment(g, range);
        }
        */
        
        /*
        if (fwd_poly != null && rev_poly != null) {
            g.setColor(new Color(255, 64, 64, 48));
            fwd_poly.paint(g);
            g.setColor(new Color(64, 255, 64, 48));
            rev_poly.paint(g);
        }
        */

        if (goal_point != null) {
            g.setColor(new Color(0, 255, 255, 192));
            g.drawOval((int)(goal_point.point.x+0.5), (int)(goal_point.point.y+0.5), 5, 5);
        }
        /*
        g.setColor(new Color(255, 64, 64, 128));
        for (EnemyRobot r : robots.getEnemies()) {
            RobotHistory h = r.getHistory();
            do {
                AbsolutePoint l = h.getLocation();
                g.drawOval((int)(l.x+0.5), (int)(l.y+0.5), 3, 3);
                h = h.prev();
            } while(h != null);
        }*/
    }
}
