package bvh.mini;

import robocode.*;
import robocode.util.Utils;
import java.awt.geom.Point2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.RoundRectangle2D;
import java.awt.Color;
import java.util.*;


/**
 * <h2>Freya</h2> <h3>- a robot by Bart "Loki" van Hest</h3>
 *
 * <p>Freya was the goddess of love, marriage, and fertility. Her identity and
 *   attributes were often confused with those of the goddess Frigg. As a deity
 *   of the dead, Freya was entitled to half the warriors killed in battle,
 *   the other half going to Odin.
 *   She was the daughter of Njord and the sister of the god Frey and originally
 *   one of the Vanir. She was frequently represented as riding in a chariot
 *   drawn by cats."</p>
 *
 * @version 0.55
 * @author Bart van Hest
 */
public class Freya extends AdvancedRobot {
   static final double             TWEEPI = 2D * Math.PI; // 360 graden draaien
   static final double             PI = Math.PI; // 180 graden draaien
   static final double             DRIEKWARTPI = 3D * Math.PI / 4D; // 135 graden draaien
   static final double             HALFPI = Math.PI / 2D; //  90 graden draaien
   static final double             DRIEACHTSTEPI = 3D * Math.PI / 8D; //  67.5 graden draaien
   static final double             KWARTPI = Math.PI / 4D; //  45 graden draaien
   static final double             EENACHTSTEPI = Math.PI / 8D; //  22.5 graden draaien
   static final double             EENZESTIENDEPI = Math.PI / 16D; //  11.25 graden draaien
   static final double             BOTGROOTTE = 36D; // de afmetingen van een robot
   static final double             WANDAFSTAND = 44D; // minimale afstand tot wand: anders bewegingsrichting omkeren
   static final double             RONDING = 80D; // ronding van slagveld
   static final double             DIAMETERBINNENCIRKEL = 300D;     // straal binnen cirkel
   static final double             MAXIMUM_SNELHEID = 8;
   static final double             VUURKRACHTFACTOR = 0.06D; // stapjes waarmee vuurkracht wordt vergroot (was 0.4)
   static final double             MAXIMUMVUURKRACHT = 3.0D;
   static final double             MINIMUMVUURKRACHT = 0.3D;
   static final double             STANDAARDVUURKRACHT = 2.1D;
   static final int                GF_MIDDEN = 13; // hoeksegmenten = 2*GF_MIDDEN+1
   static final int                POSITIESEGMENTEN = 3;
   static final int                AFSTANDSEGMENTEN = 6;
   static final int                LATERALESNELHEIDSSEGMENTEN = 4;
   static final int                TRANSVERSALESNELHEIDSSEGMENTEN = 7;

   // kanon
   private static RoundRectangle2D bewegingsveld;
   private static Hashtable doelen;
   private Doel             doel;

   // beweging
   private Point2D.Double   positie;
   private Point2D.Double   bestemming;

   /**
    * run: Freya's hoofdroutine:
    */
   public void run() {
      Color p = Color.pink;
      setColors(p, p, p);

      setAdjustGunForRobotTurn(true);
      setAdjustRadarForGunTurn(true);

      // initialiseer omhullenden die bepalend zijn voor inschatten positie van doel op het slagveld:
      bewegingsveld = new RoundRectangle2D.Double(WANDAFSTAND, WANDAFSTAND, getBattleFieldWidth() - 2D * WANDAFSTAND,
            getBattleFieldHeight() - 2D * WANDAFSTAND, RONDING, RONDING);

      if (doelen == null) {
         doelen = new Hashtable();
      }

      do {
         positie = new Point2D.Double(getX(), getY());

         if (bestemming == null) {
            bestemming = positie;
         }

         if (doel != null) {
            stuurRadar(doel);
            navigatie();
            vuurKanon();
         }
         else {
            setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
         }
         
         execute();
      } while (true);
   }

   /**
    * onScannedRobot: What to do when you see another robot
    * let op: van alle scanned bots tijdens een radar-sweep wordt de dichtstbijzijnde bot doorgegegeven!
    */
   public void onScannedRobot(ScannedRobotEvent evt) {
      String naam = evt.getName();
      Doel   d = null;
      d = (Doel) doelen.get(naam);

      if (d == null) {
         d = new Doel();
         doelen.put(naam, d);
      }

      // bijwerken globale data
      d.actief = true;
      d.setLocation(projecteerPositie(positie,
            d.richting = Utils.normalRelativeAngle(getHeadingRadians() + evt.getBearingRadians()), evt.getDistance()));
      d.lateraleSnelheid = (d.snelheid = evt.getVelocity()) * Math.sin(evt.getHeadingRadians() - d.richting);
      d.transversaleSnelheid = d.snelheid * Math.cos(evt.getHeadingRadians() - d.richting);
      d.energie = evt.getEnergy();

      if (doel == null || !doel.actief
            || ((getGunHeat() / getGunCoolingRate()) > 5D
            && (positie.distance(d) < 0.85*positie.distance(doel))
               )
         ) {
         doel = d;
      }

      // vuur test-golf:
      MiniGolf mg = new MiniGolf(d, positie, d.richting, berekenKogelSnelheid(bepaalKanonVuurkracht(d)),
            (d.lateraleSnelheid < 0 ? -1D : 1D ), geefGFStats(d));

      addCustomEvent(mg);

      stuurKanon(mg);
   }

   /**
    * Wanneer een tegenstander wordt uitgeschakeld moet deze tegenstander op non-actief
    * worden gezet in de hash-tabel met tegenstanders om te voorkomen dat data van deze
    * tegenstander nog wordt gebruikt bij bijv. navigatie.
    * Omdat het huidige doel een referentie is naar een Doel-object uit de hash-table
    * wordt deze automatisch bijgewerkt.
    *
    * @param evt Het RobotDeathEvent dat naar de robot wordt gestuurd als een andere
    * robot wordt uitgeschakeld.
    */
   public void onRobotDeath(RobotDeathEvent evt) {
      String naam = evt.getName();

      if (doelen.containsKey(naam)) {
         Doel d = (Doel) doelen.get(naam);
         d.actief = false;
      }
   }

   /**
    * Alle doelen worden op inactief gezet als initiatie voor de volgende ronde. Verder
    * wordt de tijdens het gevecht verzamelde statistiek opgeslagen.
    *
    * @param evt Het DeathEvent dat naar de robot wordt gestuurd als de robot wordt
    * uitgeschakeld.
    */
   public void onDeath(DeathEvent evt) {
      for (Enumeration en = doelen.elements(); en.hasMoreElements();) {
         ((Doel) en.nextElement()).actief = false;
      }
   }

   /**
    * De navigatie is gebaseerd op een anti-gravity/ minimum risk benadering (n.b. deze werd
    * door mij oorspronkelijk maximale entropie/bewegingsvrijheid methode genoemd).
    * De robot beweegd loodrecht op de richting naar de geselecteerde tegenstander, waarbij voor
    * een aantal afstanden en random richtingen in wordt bepaald wat het
    * risico is om naar deze positie te gaan. De positie met het laagste risico wordt gekozen.
    *
    * De maximale afstand is afhankelijk van de maximale ontsnappingshoek o.b.v. de afstand tot
    * de tegenstander en de laatst afgeschoten kogel vuurkracht van de tegenstander.
    *
    */
   private void navigatie() {
      Point2D.Double testPositie;
      double         afstotingLaatstePositie;
      double         pogingen = 0;

      // beweging op basis van resterende afstand om botsingen, en daardoor stilstand,
      // te voorkomen. Dit treedt wel op bij bepaling van afstand tot (onbereikbare) 
      // bestemming.
      if (Math.abs(getDistanceRemaining())<18D) {
         afstotingLaatstePositie = Math.random()>Math.max((10-getOthers())*0.1,0.25) ? 1 : 0;

         double risicoBestemming = weeg(bestemming, afstotingLaatstePositie);
         double risicoTestPositie;

         do {
            testPositie = projecteerPositie(positie, TWEEPI * Math.random(),
                  0.5 * Math.min(positie.distance(doel), 3D*BOTGROOTTE+Math.random()*6D*BOTGROOTTE));
            risicoTestPositie = weeg(testPositie, afstotingLaatstePositie);

            if (bewegingsveld.contains(testPositie) && risicoTestPositie < risicoBestemming) {
               bestemming = testPositie;
               risicoBestemming = risicoTestPositie;
            }
         } while (pogingen++ < 100);
      }

      // controleer of draaihoek gelijk is aan minimale draaihoek:
      double hoek = Utils.normalRelativeAngle(bepaalRichting(positie, bestemming) - getHeadingRadians());
      double draaiHoek = Math.atan(Math.tan(hoek));

      // draai en verplaats robot:
      setAhead(positie.distance(bestemming) * (hoek == draaiHoek ? 1 : -1));
      setTurnRightRadians(draaiHoek);

      // eventueel snelheid minderen om sneller te kunnen draaien:
      setMaxVelocity(Math.abs(getTurnRemainingRadians()) > KWARTPI ? (90D - Math.abs(getTurnRemaining())) / 7.5D
                                                                   : MAXIMUM_SNELHEID);
   }

   /**
    * Deze functie bepaalt het risico van de opgegeven positie. Bij de bepaling van het
    * risico worden de volgende afwegingen meegenomen:
    * <ul>
    * <li>afstand tot de huidige positie (verder weg is gunstiger)</li>
    * <li>de verhouding van de eigen energie tot die van de tegenstander (lager is gunstiger)</li>
    * <li>de hoek van de positie t.ov. de eigen positie en die van de tegenstander (de ontsnappingsrichting)</li>
    * <li>afstand tot de positie van de tegenstander (verder weg is gunstiger)</li>
    * </ul>
    *
    * @param p de positie waarvoo het risico moet worden bepaald
    * @param r een parameter die aangeeft of de huidige positie moet worden afgestoten.
    *
    * @return het risico van de opgegeven positie
    */
   public double weeg(Point2D.Double p, double r) {
      double eval = r / (12D*p.distanceSq(positie));

      for (Enumeration en = doelen.elements(); en.hasMoreElements();) {
         Doel d = (Doel) en.nextElement();

         if (d.actief) {
            eval += (Math.min(d.energie / getEnergy(), 2) * (1.0D
            + Math.abs(Math.cos(bepaalRichting(positie, p) - bepaalRichting(d, p))))) / p.distanceSq(d);
         }
      }

      return eval;
   }

   public int[] geefGFStats(Doel d) {
      return (d.guessFactors[(new Ellipse2D.Double(0, 0, getBattleFieldWidth(), getBattleFieldHeight())).contains(d) ? ((new Ellipse2D.Double( (getBattleFieldWidth()-DIAMETERBINNENCIRKEL)/2D, (getBattleFieldHeight()+DIAMETERBINNENCIRKEL)/2D, DIAMETERBINNENCIRKEL, DIAMETERBINNENCIRKEL)).contains(d) ? 1 : 2) : 0]
                            [geefAfstandIndex(d)]
                            [(int) Math.abs(d.lateraleSnelheid/3D)]
                            [(int) (3D-d.transversaleSnelheid/3D)]);
   }

   private int geefAfstandIndex(Doel d) {
      return(Math.min((int)Math.floor(positie.distance(d)/200D), AFSTANDSEGMENTEN));
   }

   private void stuurKanon(MiniGolf mg) {
      int schattingGF = GF_MIDDEN;

      // schiet direct op doel indien dit stationair is doordat het bijna is uitgeschakeld
      if (!(doel.energie == 0 && doel.snelheid == 0)) {
         // anders bepaal meest waarschijnlijke schietrichting
         for (int gf = (2 * GF_MIDDEN); gf >= 0; gf--)
            if (mg.stats[gf] > mg.stats[schattingGF]) {
               schattingGF = gf;
            }
      }

      setTurnGunRightRadians(Utils.normalRelativeAngle(doel.richting - getGunHeadingRadians()
            + mg.cirkelRichtingBinBreedte * (schattingGF - GF_MIDDEN)));
   }

   /**
    * Vuurt het kanon af als en de resterende energie voldoende is om niet te worden
    * uitgeschakeld en het kanon is uitgelijnd met de (verwachte) richting naar het doel.
    */
   private void vuurKanon() {
      if (Math.abs(getGunTurnRemainingRadians()) < Math.atan2(18D, doel.distance(positie)) && getEnergy() > STANDAARDVUURKRACHT) {
         setFire(bepaalKanonVuurkracht(doel));
      }
   }

   private double bepaalKanonVuurkracht(Doel d) {
      return (Math.max(MAXIMUMVUURKRACHT - VUURKRACHTFACTOR*(Math.exp(geefAfstandIndex(d)/1.5D)), MINIMUMVUURKRACHT));
   }

   /**
    * berekend de snelheid van de kogel op basis van de kracht van de kogel.
    *
    * @param kanonVuurkracht de kracht van de kogel.
    *
    * @return de snelheid van de kogel
    */
   private double berekenKogelSnelheid(double kanonVuurkracht) {
      return (20D - 3D * kanonVuurkracht);
   }

   /**
    * In melee-mode draait de radar rond, tenzij het kanon bijna klaar is om te schieten:
    * in dat geval wordt de radar gelocked op de tegenstander.
    * In 1-vs-1 gevechten wordt de radar altijd gelocked op de tegenstander, tenzij het
    * doel niet langer actief is (dus is vernietig, waarna een lock niet meer van belang
    * is).
    */
   private void stuurRadar(Doel d) {
      if (d.actief && (getOthers() == 1 || (getGunHeat() / getGunCoolingRate()) < 5)) {
         setTurnRadarRightRadians(2.0D * Utils.normalRelativeAngle(bepaalRichting(positie, d)
               - getRadarHeadingRadians()));
      } else {
         setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
      }
   }

   /**
    * bepaal richting van bot tot doel-positie
    *
    * @param pos de start positie
    * @param richting de richting in radialen naar de te berekenen positie.
    * @param afstand de afstand tot de te berekenen positie.
    *
    * @return de berekende positie
    */
   private static Point2D.Double projecteerPositie(Point2D.Double pos, double richting, double afstand) {
      return new Point2D.Double(pos.x + afstand * Math.sin(richting), pos.y + afstand * Math.cos(richting));
   }

   /**
    * bepaal richting van bot tot doel-positie
    *
    * @param oorsprong de positie waar vandaan de hoek moet worden berekend
    * @param doel de positie waar naar de hoek moet worden berekend.
    *
    * @return de berekende hoek in radialen
    */
   private static double bepaalRichting(Point2D.Double oorsprong, Point2D.Double doel) {
      return Math.atan2((doel.x - oorsprong.x), (doel.y - oorsprong.y));
   }

   /**
    * innerclass met de Doel.
    */
   private static class Doel extends Point2D.Double {
      static int[][][][][] guessFactors;
      boolean                 actief;
      double                  richting;
      double                  energie;
      double                  snelheid;
      double                  lateraleSnelheid;
      double                  transversaleSnelheid;

      public Doel() { 
         this.guessFactors = new int[POSITIESEGMENTEN][AFSTANDSEGMENTEN + 1][LATERALESNELHEIDSSEGMENTEN][TRANSVERSALESNELHEIDSSEGMENTEN][2 * GF_MIDDEN
            + 1];
      }
   }

   /**
    * innerclass met de golf.
    */
   private class MiniGolf extends Condition {
      Point2D.Double startPositie;
      double         afstand;
      double         richting;
      double         cirkelRichtingBinBreedte;
      double         kogelSnelheid;
      int[]          stats;
      Doel           doel;

      public MiniGolf(Doel doel, Point2D.Double positie, double doelRichting, double kogelSnelheid,
         double cirkelRichting, int[] stats) {
         this.doel = doel;
         this.startPositie = positie;
         this.richting = doelRichting;
         this.kogelSnelheid = kogelSnelheid;
         this.cirkelRichtingBinBreedte = 1.1*cirkelRichting*Math.asin(MAXIMUM_SNELHEID/kogelSnelheid)/(double)GF_MIDDEN;
         this.stats = stats;
      }

      /**
       * Method bepaalt of de wave de tegenstander heeft bereikt. Indien dit het geval is
       * wordt de hoek berekend waaronder een kogel afgevuurd had moeten worden om de tegenstander
       * te raken en wordt de corresponderende bin in de GF-statistiek opgehoogd.
       *
       * @return boolean waarde die aangeeft of de wave kan worden verwijdert of niet.
       */
      public boolean test() {
         double nabijheid = (afstand += kogelSnelheid) - doel.distance(startPositie);

         if (!doel.actief || nabijheid >= 20) {
            removeCustomEvent(this);
         }

         if (Math.abs(nabijheid) <= 20) {
            try {
               stats[(int) Math.round(Utils.normalRelativeAngle(bepaalRichting(startPositie, doel) - richting) / cirkelRichtingBinBreedte)
               + GF_MIDDEN]++;
            } catch (ArrayIndexOutOfBoundsException e) {
            }
         }
         return false;
      }

   } // einde innerclass met de golf.
}
