package jeremyreeder.collective;

import java.util.*; import java.lang.*; import java.awt.*; import java.io.*;
import robocode.*; import robocode.util.*;

/*
_________        .__  .__                 __  .__            __________        ___.           __
\_   ___ \  ____ |  | |  |   ____   _____/  |_|__|__  __ ____\______   \ ____  \_ |__   _____/  |_
/    \  \/ /  _ \|  | |  | _/ __ \_/ ___\   __\  \  \/ // __ \|       _//  _ \  | __ \ /  _ \   __\
\     \___(  <_> )  |_|  |_\  ___/\  \___|  | |  |\   /\  ___/|    |   (  <_> ) | \_\ (  <_> )  |
 \______  /\____/|____/____/\___  >\___  >__| |__| \_/  \___  >____|_  /\____/  |___  /\____/|__|
        \/                      \/     \/                   \/       \/             \/
*/
public abstract class CollectiveRobot extends RateControlRobot {
// An abstract, data-sharing robot by Jeremy Reeder, with useful Droid functionality
	protected int maxBogeyAge() { return 12; }

/*	 _________                   __               __  .__.__
	\______   \_______  ____    |__| ____   _____/  |_|__|  |   ____
	 |     ___/\_  __ \/  _ \   |  |/ __ \_/ ___\   __\  |  | _/ __ \
	 |    |     |  | \(  <_> )  |  \  ___/\  \___|  | |  |  |_\  ___/
	 |____|     |__|   \____/\__|  |\___  >\___  >__| |__|____/\___  >
		                    \______|    \/     \/                  \/
*/	abstract class Projectile {
		private double initialX, initialY, initialTime, headingRadians, velocity;
		protected double getRadius() { return 0; }
		public Double getTimeToImpact() {
			final double now = getTime(); double distance = getDistance(now);
			final double closingRate = distance - getDistance(now + 1);
			final double timeToImpact = distance / closingRate;
			if (timeToImpact < 0)
				return Double.POSITIVE_INFINITY;
			else
				return timeToImpact;
		}
		public boolean willHitMe() {
			return (getDistance(getTime() + getTimeToImpact()) <= 0);
		}
		public double getInitialX() { return initialX; }
		public void setInitialX(double initialX) { this.initialX = initialX; }
		public double getInitialY() { return initialY; }
		public void setInitialY(double initialY) { this.initialY = initialY; }
		public double getHeadingRadians() { return headingRadians; }
		public void setHeadingRadians(double headingRadians) { this.headingRadians = headingRadians; }
		public double getInitialTime() { return initialTime; }
		public void setInitialTime(double initialTime) { this.initialTime = initialTime; }
		public double getVelocity() { return velocity; }
		public void setVelocity(double velocity) { this.velocity = velocity; }
		public double getX() { final double now = getTime(); return getX(now); }
		public double getX(double time) {
			final double elapsedTime = time - getInitialTime();
			final double distanceTraveled = getVelocity() * elapsedTime;
			final double deltaX = distanceTraveled * Math.sin(getHeadingRadians());
			return getInitialX() + deltaX;
		}
		public double getY() { final double now = getTime(); return getY(now); }
		public double getY(double time) {
			final double elapsedTime = time - getInitialTime();
			final double distanceTraveled = getVelocity() * elapsedTime;
			final double deltaY = distanceTraveled * Math.cos(getHeadingRadians());
			return getInitialY() + deltaY;
		}
		public boolean isWithinBattleField() {
			final double x = getX();
			final double y = getY();
			return (x >= 0 && x <= getBattleFieldWidth() && y >= 0 && y <= getBattleFieldHeight());
		}
		public double getBearingRadians() {
			final double now = getTime();
			return getBearingRadians(now);
		}
		public double getBearingRadians(double time) {
			final double deltaX = getX(time) - CollectiveRobot.this.getX(time);
			final double deltaY = getY(time) - CollectiveRobot.this.getY(time);
			return Utils.normalRelativeAngle(
				Math.atan2(deltaX, deltaY) - CollectiveRobot.this.getHeadingRadians()
			);
		}
		public double getDistance() {
			final double now = getTime();
			return getDistance(now);
		}
		public double getDistance(double time) {
			final double deltaX = getX(time) - CollectiveRobot.this.getX(time);
			final double deltaY = getY(time) - CollectiveRobot.this.getY(time);
			final double distance = Math.sqrt(
				Math.pow(deltaX, 2) + Math.pow(deltaY, 2)
			) - getRadius() - CollectiveRobot.this.getRadius();
			return distance;
		}
		private String reporterName;
		public String getReporterName() {
			return reporterName;
		}
		protected void setReporterName(String reporterName) {
			this.reporterName = reporterName;
		}
		public boolean equals(Projectile other) {
			return hashCode() == other.hashCode();
		}
		public Projectile(double initialX, double initialY, double initialTime, double headingRadians, double velocity, String reporterName) {
			setInitialX(initialX);
			setInitialY(initialY);
			setInitialTime(initialTime);
			setHeadingRadians(headingRadians);
			setVelocity(velocity);
			setReporterName(reporterName);
		}
	}
/*	__________
	\______   \ ____   ____   ____ ___.__.
	 |    |  _//  _ \ / ___\_/ __ <   |  |
	 |    |   (  <_> ) /_/  >  ___/\___  |
	 |______  /\____/\___  / \___  > ____|
		    \/      /_____/      \/\/
*/	class Bogey extends Projectile implements Comparable<Bogey> {
		public boolean isMe() {
			final String name = getName();
			final String myName = CollectiveRobot.this.getName();
			return name.equals(myName);
		}
		public boolean isHostile() {
			return !isTeammate(getName());
		}
		public boolean isWithinRadarRange() {
			return (getDistance() + getRadius() + CollectiveRobot.this.getRadius() < Rules.RADAR_SCAN_RADIUS);
		}
		private Double previousInitialTime = null, previousEnergy = null;
		public Double getPreviousInitialTime() { return previousInitialTime; }
		public void setPreviousInitialTime(Double previousInitialTime) { this.previousInitialTime = previousInitialTime; }
		public Double getPreviousEnergy() { return previousEnergy; }
		public void setPreviousEnergy(Double previousEnergy) { this.previousEnergy = previousEnergy; }
		protected double getRadius() { return CollectiveRobot.this.getRadius(); }
		public int compareTo(Bogey other) {
			if (this.equals(other))
				return 0;
			else {
				if (this.getName().equals(other.getName()))
					return (int)Math.ceil(
						1000 * (
							other.getInitialTime() - this.getInitialTime() // if 2 copies, most recent first
						)
					);
				else if (this.isMe() && !other.isMe())
					return 1;
				else if (!this.isMe() && other.isMe())
					return -1;
				else if (this.isHostile() && !other.isHostile())
					return -1;
				else if (!this.isHostile() && other.isHostile())
					return 1;
				else {
					final double now = getTime();
					return (int)Math.ceil(getDistance(now + 2) - other.getDistance(now + 2));
				}
			}
		}
		private String name;
		private double energy;
		public void setName(String name) {
			this.name = name;
		}
		public String getName() {
			return name;
		}
		public void setEnergy(double energy) {
			this.energy = energy;
		}
		public double getEnergy() {
			return energy;
		}
		public Double getEnergyDifferential() {
			if (getPreviousEnergy() == null)
				return null;
			else
				return getEnergy() - getPreviousEnergy();
		}
		public boolean hasJustFired() {
			if (getInitialTime() >= getTime() - 1) // only return true if the data is fresh
				return (getFirePower() != null);
			else return false;
			// It is impossible to be 100% certain whether the Bogey has fired, especially when
			// there are many Bogies on the battlefield.  But this is my best guess, based on
			// the Bogey's energy drop, adjusted to cancel out some other energy factors.
		}
		public Double getFirePower() {
			Double energyDifferential = getEnergyDifferential();
			if (energyDifferential != null && energyDifferential <= -0.1 && energyDifferential >= -3)
				return -energyDifferential;
			else
				return null;
		}
		public Bogey(
			double initialX, double initialY, double initialTime, double headingRadians, double velocity, String reporterName,
			double energy, String name
		) {
			super(initialX, initialY, initialTime, headingRadians, velocity, reporterName);
			setEnergy(energy);
			setName(name);
		}
		public Bogey(ScannedRobotEvent e) {
			this(
				CollectiveRobot.this.getX() + e.getDistance() * Math.sin(
					Utils.normalAbsoluteAngle(
						CollectiveRobot.this.getHeadingRadians() + e.getBearingRadians()
					)
				),
				CollectiveRobot.this.getY() + e.getDistance() * Math.cos(
					Utils.normalAbsoluteAngle(
						CollectiveRobot.this.getHeadingRadians() + e.getBearingRadians()
					)
				),
				(double)e.getTime(), e.getHeadingRadians(), e.getVelocity(), CollectiveRobot.this.getName(),
				e.getEnergy(), e.getName()
			);
			System.out.println("Detected fresh bogey " + getName() + " by scanning");
		}
		public Bogey(BulletHitEvent e) {
			this(
				e.getBullet().getX(), e.getBullet().getY(),
				(double)(e.getTime() - 0.001), 0, 0, CollectiveRobot.this.getName(),
				0, e.getName() // e.getBullet().getVictim() is sometimes null; this should be more reliable
			); // initialTime is set slightly old in order to avoid overwriting data from scans, which are more informative
			System.out.println("Detected pseudo-stale bogey " + getName() + " by shooting");
		}
		public Bogey(HitRobotEvent e) {
			this(
				CollectiveRobot.this.getX() + 2 * CollectiveRobot.this.getRadius() * Math.sin(
					Utils.normalAbsoluteAngle(
						CollectiveRobot.this.getHeadingRadians() + e.getBearingRadians()
					)
				),
				CollectiveRobot.this.getY() + 2 * CollectiveRobot.this.getRadius() * Math.cos(
					Utils.normalAbsoluteAngle(
						CollectiveRobot.this.getHeadingRadians() + e.getBearingRadians()
					)
				),
				(double)e.getTime() - 0.001, 0, 0, CollectiveRobot.this.getName(),
				e.getEnergy(), e.getName()
			);
			System.out.println("Detected pseudo-stale bogey " + getName() + " by collision");
		}
		public Bogey(HitByBulletEvent e) {
			this(
				CollectiveRobot.this.getX() + (Rules.RADAR_SCAN_RADIUS / 4 - 2 * CollectiveRobot.this.getRadius()) * Math.sin(
					Utils.normalAbsoluteAngle(e.getHeadingRadians() + Math.PI)
				),
				CollectiveRobot.this.getY() + (Rules.RADAR_SCAN_RADIUS / 4 - 2 * CollectiveRobot.this.getRadius()) * Math.cos(
					Utils.normalAbsoluteAngle(e.getHeadingRadians() + Math.PI)
				),
				(double)(e.getTime() - 4), 0, 0, CollectiveRobot.this.getName(),
				0, e.getName()
			);
			System.out.println("Detected stale bogey " + getName() + " by getting shot");
		}
		public Bogey() { // self-reported
			this(
				CollectiveRobot.this.getX(), CollectiveRobot.this.getY(),
				(double)getTime(), CollectiveRobot.this.getHeadingRadians(), CollectiveRobot.this.getVelocity(), CollectiveRobot.this.getName(),
				CollectiveRobot.this.getEnergy(), CollectiveRobot.this.getName()
			);
		}
		public void broadcast() {
			if (
				isMe() // report myself
			||
				(
					isHostile() // assume teammates will report themselves
				&&
					getReporterName() == CollectiveRobot.this.getName() // report only first-hand data
				)
			) {
				Map<String, String> map = new HashMap<String, String>();
				map.put("messageType", "Projectile");
				map.put("projectileType", "Bogey");
				map.put("initialX", Double.toString(getInitialX()));
				map.put("initialY", Double.toString(getInitialY()));
				map.put("initialTime", Double.toString(getInitialTime()));
				map.put("velocity", Double.toString(getVelocity()));
				map.put("headingRadians", Double.toString(getHeadingRadians()));
				map.put("energy", Double.toString(getEnergy()));
				map.put("name", getName());
				try {
					broadcastMessage((Serializable)map);
				} catch (IOException e) {
					System.out.println(e.getMessage());
					e.printStackTrace();
				}
			}
		}
		public double getX(double time) { // keep x within the boundaries of the battlefield
			return Math.max(
				getRadius(),
				Math.min(super.getX(time), CollectiveRobot.this.getBattleFieldWidth() - getRadius())
			);
		}
		public double getY(double time) { // keep y within the boundaries of the battlefield
			return Math.max(
				getRadius(),
				Math.min(super.getY(time), CollectiveRobot.this.getBattleFieldHeight() - getRadius())
			);
		}
	}
/*	__________                             _________       __
	\______   \ ____   ____   ____ ___.__./   _____/ _____/  |
	 |    |  _//  _ \ / ___\_/ __ <   |  |\_____  \_/ __ \   __\
	 |    |   (  <_> ) /_/  >  ___/\___  |/        \  ___/|  |
	 |______  /\____/\___  / \___  > ____/_______  /\___  >__|
		    \/      /_____/      \/\/            \/     \/
*/	class BogeySet extends TreeSet<Bogey> {
		public Bogey get(String name) {
			for (Bogey bogey : this) {
				if (bogey.getName().equals(name))
					return bogey;
			}
			return null;
		}
		public boolean add(Bogey newBogey) {
			Iterator i = this.iterator();
			while (i.hasNext()) {
				Bogey existingBogey = (Bogey)i.next();
				if (existingBogey.getName().equals(newBogey.getName())) {
					if (newBogey.getInitialTime() > existingBogey.getInitialTime()) { // only add if newer
						newBogey.setPreviousInitialTime(existingBogey.getInitialTime());
						newBogey.setPreviousEnergy(existingBogey.getEnergy());
						i.remove();
						if ( // detect enemy Missiles
							newBogey.isHostile() // assume that teammates will announce their Missiles
						&&
							newBogey.hasJustFired()
						&&
							newBogey.getDistance() <= ( // ignore Bogeys too distant to see me
								Rules.RADAR_SCAN_RADIUS - newBogey.getRadius() - getRadius() +
								Math.abs(newBogey.getVelocity()) + Math.abs(getVelocity())
							)
						) {
							System.out.println("Detected missile fired by " + newBogey.getName());
							final double firePower = newBogey.getFirePower();
							missileSet.add(new NonLeadingMissile(newBogey, getName()));
							missileSet.add(new LeadingMissile(newBogey, getName()));
						}
						newBogey.broadcast();
						return super.add(newBogey);
					} else return false;
				}
			}
			newBogey.broadcast();
			return super.add(newBogey);
		}
		public boolean remove(String name) {
			for (Bogey bogey : this) {
				if (bogey.getName().equals(name)) {
					return remove(bogey);
				}
			}
			return false;
		}
		public void paint() {
			final double now = getTime();
			Graphics2D graphics = getGraphics();
			final int myX = (int)getX(now); final int myY = (int)getY(now);
			final int radius = (int)getRadius(); final int diameter = 2 * radius + 1;
			graphics.setColor(Color.white);
			graphics.drawOval(myX - (radius + 5), myY - (radius + 5), diameter + 10, diameter + 10); // draw a big circle around me
			graphics.setColor(Color.orange); // highlight first (most threatening) bogey
			for (Bogey bogey : bogeySet) {
				final int xNow = (int)bogey.getX(now);
				final int yNow = (int)bogey.getY(now);
				final int xFuture = (int)bogey.getX(now + 5);
				final int yFuture = (int)bogey.getY(now + 5);
				graphics.drawOval(xNow - radius, yNow - radius, diameter, diameter);
				graphics.drawLine(xNow, yNow, xFuture, yFuture);
				graphics.setColor(Color.white);
			}
		}
		public void cleanAndSort() {
			Iterator bogeyIterator = this.iterator();
			BogeySet newBogeySet = new BogeySet();
			while (bogeyIterator.hasNext()) {
				Bogey bogey = (Bogey)bogeyIterator.next();
				bogeyIterator.remove();
				if (bogey.getInitialTime() >= getTime() - maxBogeyAge())
					newBogeySet.add(bogey);
			}
			bogeyIterator = newBogeySet.iterator();
			for (Bogey bogey : newBogeySet)
				bogeySet.add(bogey);
		}
	}
/*	   _____  .__              .__.__
	  /     \ |__| ______ _____|__|  |   ____
	 /  \ /  \|  |/  ___//  ___/  |  | _/ __ \
	/    Y    \  |\___ \ \___ \|  |  |_\  ___/
	\____|__  /__/____  >____  >__|____/\___  >
		    \/        \/     \/             \/
*/	class Missile extends Projectile implements Comparable<Missile> {
		private String shooterName;
		public String getShooterName() { return shooterName; }
		public void setShooterName(String shooterName) { this.shooterName = shooterName; }
		public Missile(double initialX, double initialY, double initialTime, double headingRadians, double firePower, String reporterName, String shooterName) {
			super(
				initialX, initialY,
				initialTime,
				headingRadians,
				Rules.getBulletSpeed(firePower),
				reporterName
			);
			setShooterName(shooterName);
			// broadcasting only my own bullets because they're the only ones I'm sure of
		}
		public Missile(double firePower) { // reported by me, the shooter
			super(
				CollectiveRobot.this.getX(), CollectiveRobot.this.getY(),
				getTime(),
				CollectiveRobot.this.getGunHeadingRadians(),
				Rules.getBulletSpeed(firePower),
				CollectiveRobot.this.getName()
			);
			setShooterName(CollectiveRobot.this.getName());
			broadcast();
		}
		public int compareTo(Missile other) { // most threating first
			if (this.equals(other))
				return 0;
			else {
				final boolean thisWillHitMe = willHitMe();
				final boolean otherWillHitMe = other.willHitMe();
				if (
					(thisWillHitMe && otherWillHitMe) ||
					!(thisWillHitMe || otherWillHitMe)
				) return (int)Math.ceil(getTimeToImpact() - other.getTimeToImpact()); // first to hit comes first
				else if (thisWillHitMe && !otherWillHitMe)
					return -1; // missile that will hit comes first
				else // if (otherWillHitMe && !thisWillHitMe) // (implied)
					return 1;
			}
		}
		public void broadcast() {
			Map<String, String> map = new HashMap<String, String>();
			map.put("messageType", "Projectile");
			map.put("projectileType", "Missile");
			map.put("initialX", Double.toString(getInitialX()));
			map.put("initialY", Double.toString(getInitialY()));
			map.put("initialTime", Double.toString(getInitialTime()));
			map.put("firePower", Double.toString((20 - getVelocity()) / 3));
			map.put("headingRadians", Double.toString(getHeadingRadians()));
			map.put("energy", Double.toString(getEnergy()));
			map.put("name", getName());
			try {
				broadcastMessage((Serializable)map);
			} catch (IOException e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			}
		}

	}
/*	 _______                .____                      .___.__                   _____  .__              .__.__
	 \      \   ____   ____ |    |    ____ _____     __| _/|__| ____    ____    /     \ |__| ______ _____|__|  |   ____
	 /   |   \ /  _ \ /    \|    |  _/ __ \\__  \   / __ | |  |/    \  / ___\  /  \ /  \|  |/  ___//  ___/  |  | _/ __ \
	/    |    (  <_> )   |  \    |__\  ___/ / __ \_/ /_/ | |  |   |  \/ /_/  >/    Y    \  |\___ \ \___ \|  |  |_\  ___/
	\____|__  /\____/|___|  /_______ \___  >____  /\____ | |__|___|  /\___  / \____|__  /__/____  >____  >__|____/\___  >
		    \/            \/        \/   \/     \/      \/         \//_____/          \/        \/     \/             \/
*/	class NonLeadingMissile extends Missile { // assume the enemy fires directly at me
		public NonLeadingMissile(Bogey bogey, String reporterName) {
			super(0, 0,
				bogey.getInitialTime() - 1, // enemy fired at least one tick before his last known position
				0, bogey.getFirePower(), reporterName, bogey.getName()
			);
			final double fireTime = getTime();
			setInitialX(bogey.getX(fireTime));
			setInitialY(bogey.getY(fireTime));
			final double deltaX = CollectiveRobot.this.getX(fireTime - 1) - getInitialX(); // enemy last saw me one tick before he fired
			final double deltaY = CollectiveRobot.this.getY(fireTime - 1) - getInitialY();
			setHeadingRadians(
				Utils.normalAbsoluteAngle(
					Math.atan2(deltaX, deltaY)
				)
			);
		}
	}
/*	.____                      .___.__                  _____  .__               .__.__
	|    |    ____ _____     __| _/|__| ____    ____   /     \ |__| ______ ______|__|  |   ____
	|    |  _/ __ \\__  \   / __ | |  |/    \  / ___\ /  \ /  \|  |/  ___//  ___/|  |  | _/ __ \
	|    |__\  ___/ / __ \_/ /_/ | |  |   |  \/ /_/  >    Y    \  |\___ \ \___ \ |  |  |_\  ___/
	|_______ \___  >____  /\____ | |__|___|  /\___  /\____|__  /__/____  >____  >|__|____/\___  >
		    \/   \/     \/      \/         \//_____/         \/        \/     \/              \/
*/	class LeadingMissile extends Missile { // assume the enemy uses my linear targeting algorithm
		public LeadingMissile(Bogey bogey, String reporterName) {
			super(0, 0,
				bogey.getInitialTime() - 1, // enemy fired at least one tick before his last known position
				0, bogey.getFirePower(), reporterName, bogey.getName()
			);
			final double firePower = bogey.getFirePower();
			final double fireTime = getTime();
			setInitialX(bogey.getX(fireTime));
			setInitialY(bogey.getY(fireTime));
			final double deltaX = CollectiveRobot.this.getX(fireTime - 1) - getInitialX(); // enemy last saw me one tick before he fired
			final double deltaY = CollectiveRobot.this.getY(fireTime - 1) - getInitialY();
			final double absoluteBearingToMe = Utils.normalAbsoluteAngle(
				Math.atan2(deltaX, deltaY)
			);
			final double myVelocity = CollectiveRobot.this.getVelocity();
			final double myHeading = CollectiveRobot.this.getHeadingRadians();
			setHeadingRadians(
				Utils.normalAbsoluteAngle(
					absoluteBearingToMe + (
						Math.asin(
							myVelocity * Math.sin(myHeading - absoluteBearingToMe) / Rules.getBulletSpeed(firePower)
						) + 2 * (
							myVelocity * Math.sin(myHeading - absoluteBearingToMe) / 13
						)
					) / 3
				)
			);
		}
	}
/*	   _____  .__              .__.__           _________       __  
	  /     \ |__| ______ _____|__|  |   ____  /   _____/ _____/  |_
	 /  \ /  \|  |/  ___//  ___/  |  | _/ __ \ \_____  \_/ __ \   __\
	/    Y    \  |\___ \ \___ \|  |  |_\  ___/ /        \  ___/|  |
	\____|__  /__/____  >____  >__|____/\___  >_______  /\___  >__|
		    \/        \/     \/             \/        \/     \/
*/	class MissileSet extends TreeSet<Missile> {
		public Missile first() {
			for (Missile missile : this)
				return missile;
			return null;
		}
		public boolean add(Missile missile) {
			return super.add(missile);
		}
		public void paint() {
			Graphics2D graphics = getGraphics();
			graphics.setColor(Color.orange); // highlight first (most threatening) missile
			for (Missile missile : missileSet) {
				final double now = getTime();
				final int xPast = (int)missile.getX(now - 2);
				final int yPast = (int)missile.getY(now - 2);
				final int xNow = (int)missile.getX(now);
				final int yNow = (int)missile.getY(now);
				graphics.drawLine(xPast, yPast, xNow, yNow);
				graphics.setColor(Color.white);
			}
		}
		public void cleanAndSort() {
			// TODO: figure out why some Missiles seem to disappear before they should
			Iterator missileIterator = this.iterator();
			MissileSet newMissileSet = new MissileSet();
			while (missileIterator.hasNext()) {
				Missile missile = (Missile)missileIterator.next();
				missileIterator.remove();
				if (missile.isWithinBattleField())
					newMissileSet.add(missile);
			}
			missileIterator = newMissileSet.iterator();
			for (Missile missile : newMissileSet)
				missileSet.add(missile);
		}
	}
/*
_________               .__            
\_   ___ \_____    ____ |  |__   ____  
/    \  \/\__  \ _/ ___\|  |  \_/ __ \ 
\     \____/ __ \\  \___|   Y  \  ___/ 
 \______  (____  /\___  >___|  /\___  >
        \/     \/     \/     \/     \/ 
*/	class Cache { // avoids calling the getters of the Robocode API too often, which could disable me
		// caching only instantaneous data (not *rates*), which can not change until the next tick
		private double x, y, headingRadians, gunHeadingRadians, radarHeadingRadians, velocity;
		private long time;
		public double getX() { return x; }
		public double getY() { return y; }
		public double getHeadingRadians() { return headingRadians; }
		public double getGunHeadingRadians() { return gunHeadingRadians; }
		public double getRadarHeadingRadians() { return radarHeadingRadians; }
		public double getVelocity() { return velocity; }
		public long getTime() { return time; }
		public Cache(double x, double y, double headingRadians, double gunHeadingRadians, double radarHeadingRadians, double velocity, long time) {
			this.x = x; this.y = y; this.headingRadians = headingRadians;
			this.gunHeadingRadians = gunHeadingRadians;
			this.radarHeadingRadians = radarHeadingRadians;
			this.velocity = velocity; this.time = time;
		}
	}




	private boolean PAINT_MODE = true;
	Cache cache = null;
	private Double battleFieldWidth = null;
	public double getBattleFieldWidth() { // access Robocode API for this only once per battle
		if (battleFieldWidth == null)
			return (battleFieldWidth = super.getBattleFieldWidth());
		else return battleFieldWidth;
	}
	private Double battleFieldHeight = null;
	public double getBattleFieldHeight() { // access Robocode API for this only once per battle
		if (battleFieldHeight == null)
			return (battleFieldHeight = super.getBattleFieldHeight());
		else return battleFieldHeight;
	}
	protected double radiansToDegrees(double radians) { return radians * 180 / Math.PI; }
	public double getHeadingRadians() { return cache.headingRadians; }
	public double getGunHeadingRadians() { return cache.gunHeadingRadians; }
	public double getRadarHeadingRadians() { return cache.radarHeadingRadians; }
	public double getHeading() { return radiansToDegrees(getHeadingRadians()); }
	public double getGunHeading() { return radiansToDegrees(getGunHeadingRadians()); }
	public double getRadarHeading() { return radiansToDegrees(getRadarHeadingRadians()); }
	public long getTime() { return cache.time; }
	public double getX() { return cache.x; }
	public double getY() { return cache.y; }
	private Double radius = null;
	protected double getRadius() {
		if (radius == null) {
			final double safetyMargin = 1.05;
			radius = safetyMargin * Math.sqrt(
				Math.pow(
					getWidth() / 2, 2
				) + Math.pow(
					getHeight() / 2, 2
				)
			);
		}
		return radius;
	}
	protected BogeySet bogeySet = new BogeySet();
	protected MissileSet missileSet = new MissileSet();
	public void onRobotDeath(String robotName) {
		bogeySet.remove(robotName);
	}
	public void onBattleEnded(BattleEndedEvent e) {
		bogeySet.clear();
		missileSet.clear();
	}
	public double getX(double time) {
		final double elapsedTime = time - getTime();
		final double distanceTraveled = elapsedTime * getVelocity();
		return getX() + distanceTraveled * Math.sin(getHeadingRadians());
	}
	public double getY(double time) {
		final double elapsedTime = time - getTime();
		final double distanceTraveled = elapsedTime * getVelocity();
		return getY() + distanceTraveled * Math.cos(getHeadingRadians());
	}
	public void setFire(double firePower) {
		final double energy = getEnergy();
		if (energy > 0 && getGunHeat() == 0) {
			firePower = Math.min(firePower, energy / 3); // limit energy consumption when weak
			missileSet.add(new Missile(firePower));
			super.setFire(firePower);
		}
	}
	public void fire(double firePower) { setFire(firePower); stop(); }
	public int nonZeroSign(double number) {
		if (number >= 0) return 1; else return -1;
	}
	protected Bogey aquireTarget() {
		Bogey firstBogey = bogeySet.first();
		if (firstBogey.isHostile()) return firstBogey;
		else return null;
	}
	protected void scanTargetArea(Bogey target) { // scan a wide arc that contains the target's next position
		final double radarBearingRadians = Utils.normalRelativeAngle(
			target.getBearingRadians(getTime() + 1) - (
				getRadarHeadingRadians() - getHeadingRadians()
			)
		);
		final double radarOffsetRadians = ((Rules.RADAR_TURN_RATE_RADIANS + Rules.GUN_TURN_RATE_RADIANS) / 2) * nonZeroSign(radarBearingRadians);
		setRadarRotationRateRadians(radarBearingRadians + radarOffsetRadians);
	}
	protected boolean aimAtTarget(Bogey target, double firePower) {
		final double now = getTime();
		firePower = Math.min(firePower, getEnergy() / 3); // match overridden setFire(), where energy consumption is limited when weak
		double gunOffsetRadians = 0;
		final double absoluteBearingRadians = getHeadingRadians() + target.getBearingRadians(now + 1);
		final double targetHeadingRadians = target.getHeadingRadians();
		final double gunBearingRadians = ( // imprecise linear targeting
			Utils.normalRelativeAngle(
				absoluteBearingRadians - getGunHeadingRadians() + (
					Math.asin(
						target.getVelocity() * Math.sin( // precise linear targeting, ignorant of walls
							targetHeadingRadians - absoluteBearingRadians
						) / Rules.getBulletSpeed(firePower)
					) + 2 * ( // simple imprecise linear targeting
						target.getVelocity() * Math.sin(
							targetHeadingRadians - absoluteBearingRadians
						) / 13
					)
				) / 3 // weighted average of two algorithms
			)
		);
		if (
			getGunHeat() > 0.1 // if barrel is too hot to fire now or on the next tick
		||
			(!(this instanceof Droid) && target.getInitialTime() < now - 2) // or if I have radar & haven't seen the target lately
		) { // wave barrel to widen scan arc
			gunOffsetRadians = (-Rules.GUN_TURN_RATE_RADIANS / 2) * nonZeroSign(gunBearingRadians);
		} // otherwise aim directly at target's next position
		setGunRotationRateRadians(gunBearingRadians + gunOffsetRadians);
		return (
			gunOffsetRadians != 0 // Am I wagging the barrel, or am I aiming at my target?
		|| // Is my barrel currently aimed at the target's vital area (the middle third)?
			Math.abs(gunBearingRadians) < Math.tan((target.getRadius() / 3) / target.getDistance(now))
		); // If so, I'm ready to fire.
	}
	protected void searchForBogeys() { // spin the radar scanner as quickly as possible in the direction in which it's already moving
		final double turnRateRadians = Rules.MAX_TURN_RATE_RADIANS;
		final double gunRotationRateRadians = turnRateRadians + Rules.GUN_TURN_RATE_RADIANS;
		final double radarRotationRateRadians = gunRotationRateRadians + Rules.RADAR_TURN_RATE_RADIANS;
		final int sign = nonZeroSign(getTurnRateRadians());
		// setTurnRateRadians(sign * turnRateRadians);
		setGunRotationRateRadians(sign * gunRotationRateRadians);
		setRadarRotationRateRadians(sign * radarRotationRateRadians);
		if (this instanceof Droid) setFire(0.1); // if necessary, use small bullets in lieu of radar
	}
	protected void refreshData() {
		cache = new Cache( // must be called first, or we'll have Robocode's data from the previous tick
			super.getX(), super.getY(), super.getHeadingRadians(), super.getGunHeadingRadians(),
			super.getRadarHeadingRadians(), super.getVelocity(), super.getTime()
		);
		bogeySet.add(new Bogey()); // announce my position to teammates
		final long now = getTime();
		bogeySet.cleanAndSort();
		missileSet.cleanAndSort();
		if (PAINT_MODE) { bogeySet.paint(); missileSet.paint(); }
	}
	public void onScannedRobot(ScannedRobotEvent e) { bogeySet.add(new Bogey(e)); }
	public void onHitByBullet(HitByBulletEvent e) {
		bogeySet.add(new Bogey(e)); // stale data
		Bogey bogey = bogeySet.get(e.getName()); // may be newer
		Double previousEnergy = bogey.getPreviousEnergy();
		if (previousEnergy != null)
			bogey.setPreviousEnergy( // cancel enemy's energy gain to increase accuracy of bullet detection
				previousEnergy + Rules.getBulletHitBonus(e.getBullet().getPower())
			);
	}
	public void onBulletHit(BulletHitEvent e) {
		bogeySet.add(new Bogey(e)); // pseudo-stale (incomplete) data
		Bogey bogey = bogeySet.get(e.getName()); // may be more complete
		Double previousEnergy = bogey.getPreviousEnergy();
		if (previousEnergy != null)
			bogey.setPreviousEnergy( // cancel enemy's energy loss to increase accuracy of bullet detection
				previousEnergy - Rules.getBulletDamage(e.getBullet().getPower())
			);
	}
	public void onHitRobot(HitRobotEvent e) { bogeySet.add(new Bogey(e)); }
	public void onMessageReceived(MessageEvent e) {
		final String sender = e.getSender();
		Serializable message = e.getMessage();
		if (message instanceof Map) {
			try {
				@SuppressWarnings("unchecked") Map<String, String> map = (Map<String, String>)message;
				String messageType = map.get("messageType");
				if (messageType.equals("Projectile")) {
					final String projectileType = map.get("projectileType");
					final double initialX = Double.parseDouble(map.get("initialX"));
					final double initialY = Double.parseDouble(map.get("initialY"));
					final double initialTime = Double.parseDouble(map.get("initialTime"));
					final double headingRadians = Double.parseDouble(map.get("headingRadians"));
					if (projectileType.equals("Bogey")) {
						final double velocity = Double.parseDouble(map.get("velocity"));
						final String name = map.get("name");
						System.out.println("Bogey " + name + " reported by " + sender);
						bogeySet.add(
							new Bogey(initialX, initialY, initialTime, headingRadians, velocity, sender,
								Double.parseDouble(map.get("energy")), name
							)
						);
					} else if (projectileType.equals("Missile")) {
						final double firePower = Double.parseDouble(map.get("firePower"));
						System.out.println("Missile annouced by " + sender);
						missileSet.add(
							new Missile(initialX, initialY, initialTime, headingRadians, firePower, sender,
								map.get("shooterName")
							)
						);
					}
				}
			} catch (ClassCastException cce) {
				System.out.println("Error processing message from " + e.getSender()); // must be Map<String, String>
			}
		} else System.out.println("Error receiving message from " + e.getSender()); // must be Map
	}
}

