package rc.yoda.plugin;

import robocode.*;
import java.io.File;
import java.util.HashMap;
import java.awt.Graphics2D;
import java.awt.Color;

import rc.yoda.plugin.moves.Ataru;
import rc.yoda.utils.*;

/** 
 * MovementArray - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * MovementArray.java : v1.0 -- 2007/05/12
 */


/**
 * MovementArray is class that load and manages
 * all Movement classes and determines which Movement
 * is the best to use a what time. Acts as a 
 * dynamic MutiMode manger for a robots movements
 *
 * @author Robert Codd
 * @version v1.1
 */
public class MovementArray implements Plugin
{
	/**
	 * The name of the folder from which movements
	 * will be loaded from
	 */
	private static final String FOLDER = "moves";
	
	/**
	 * A map of all movements this Movement Array is utilizing 
	 */
	private static HashMap movements;
	
	/**
	 * A map of the stats that coorespond to to the loaded movements
	 */
	private static HashMap moveStats;
	
	/**
	 * The name of the movement currently in use
	 */
	private static String movementKey = "";
	
	/**
	 * A reference to the AdvancedRobot this MovementArray is working for
	 */
	private AdvancedRobot robot;
	
	/**
	 * The amount of times the enemy has fired this round
	 */
	private double bulletsFired = 0;
	
	/**
	 * The amount of times this robot has been hit by a bullet this round
	 */
	private double bulletsHit = 0;
	
	/**
	 * Class Constructor specifing the AdvancedRobot this MovementArray is working for
	 */
	public MovementArray(AdvancedRobot robot) {
		this.robot = robot;
		init();
	}

	/**
	 * Initialized needed aspects of the Movement Array
	 */
	private void init() {
		if (robot.getRoundNum() == 0) { movements = new HashMap(); moveStats = new HashMap(); }
		populateMap();
		movementKey = bestMovement();
	}

	/**
	 * Event method called by this robot when it detects
	 * that the enemy fired a bullet
	 *
	 * @param double deltaEnergy the power of the bullet fired
	 */
	public void onRobotFire(double deltaEnergy) {
		bulletsFired++;
		if (movements.size() == 0) { return; }
		getMovement().onRobotFire(deltaEnergy);

	} 
	
	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */	
	public void onScannedRobot(ScannedRobotEvent e) {
		if (movements.size() == 0) { return; }		
		getMovement().onScannedRobot(e);
	}
	
	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */	
	public void onHitByBullet(HitByBulletEvent e) {
		bulletsHit++;
		if (movements.size() == 0) { return; }		
		getMovement().onHitByBullet(e);
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hits another robot
	 * 
	 * @param BulletHitEvent information about the robot 
	 * that got hit by the bullet
	 */	
	public void onBulletHit(BulletHitEvent e) {}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hit a wall
	 *
	 * @param BulletMissedEvent information about the bullet
	 */	
	public void onBulletMissed(BulletMissedEvent e) {}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */	
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		if (movements.size() == 0) { return; }
		getMovement().onBulletHitBullet(e);
	}

	/**
	 * Event method called by Robocode when this robot hits a wall
	 * 
	 * @param HitWallEvent information about the wall
	 */	
	public void onHitWall(HitWallEvent e) {
		if (movements.size() == 0) { return; }
		getMovement().onHitWall(e);
	}
	
	/**
	 * Event method called by Robocode when this robot 
	 * collides with another robot
	 *
	 * @param HitRobotEvent information about the collision and 
	 * the other robot in the crash
	 */	
	public void onHitRobot(HitRobotEvent e) {
		if (movements.size() == 0) { return; }
		getMovement().onHitRobot(e);
	}
	
	/**
	 * Event method called by Robocode when a robot dies
	 *
	 * @param RobotDeathEvent name of decaesed robot
	 */	
	public void onRobotDeath(RobotDeathEvent e) {}
	
	/**
	 * Event method called by Robocode when this robot
	 * wins a round
	 *
	 * @param WinEvent
	 */	
	public void onWin(WinEvent e) {
		if (movements.size() == 0) { return; }
		moveStats.put(movementKey, new Double((bulletsFired == 0 ? 0.0 : bulletsHit / bulletsFired)));
		movementKey = bestMovement();
		getMovement().onWin(e);
	}

	/**
	 * Event method called by Robocode when this robot dies
	 *
	 * @param DeathEvent 
	 */
	public void onDeath(DeathEvent e) {
		if (movements.size() == 0) { return; }
		moveStats.put(movementKey, new Double((bulletsFired == 0 ? 0.0 : bulletsHit / bulletsFired)));	
		movementKey = bestMovement();		
		getMovement().onDeath(e);
	}
	
	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */		
	public void onPaint(Graphics2D g) {
		if (movements.size() == 0) { return; }
		getMovement().onPaint(g);
		g.setColor(Color.white);
		g.drawString(String.format("Movement in use : %s", movementKey), 5 , 586);
	}

	/**
	 * return a reference to the Movement class that is currently in use
	 *
	 * @return Movement the movments that is currently in use this round
	 */
	private Movement getMovement() {
		return (Movement)movements.get(movementKey);
	}

	/**
	 * return the key to the Movement with the lowest Hit ratio
	 *
	 * @return String the key to the Movement with the lowest Hit ratio
	 */
	private String bestMovement() {
		double lowestHR = Double.POSITIVE_INFINITY;
		String keys[] = (String[]) movements.keySet().toArray(new String[movements.size()]),
			key = "";
		
		for (int count = 0; count < keys.length; count++)
		{
			double value = ((Double)moveStats.get(keys[count])).doubleValue();
			lowestHR = (value < lowestHR && (key = keys[count]) == key ? value : lowestHR);
		}
		return key;
	}

	/**
	 * Retrives an array of Movement classes in the directory FOLDER from
	 * the Factory Class and stores them in the HashMap movements
	 */
	private void populateMap() {		
		Object[] parameters = { (Object)robot };
		String canonicalPath = "",
			packageName = "";
		
		/*{
			Class Claas = getClass(); 
			String dir = robot.getDataDirectory().getPath();
			
			packageName =  Claas.getName().replaceAll(Claas.getSimpleName(), FOLDER);
			canonicalPath = dir.split(robot.getName().split(" ")[0].concat(".data"))[0]
				+ packageName.replace(".", File.separator); 
		}*/
	
		Factory factory = new Factory((new Movement(robot)).getClass(), parameters, new Class[]{
				Ataru.class
		});
		Movement[] temp = (Movement[]) factory.getClasses();
		movements.clear();
								
		for (int count = 0; count < temp.length; count++)
		{
			String key = temp[count].getClass().getSimpleName();
			if (!moveStats.containsKey(key)) { moveStats.put(key, new Double(0)); }
			movements.put(key, temp[count]);
		}
	}
}