package djc;
import robocode.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * DefaultTacticalStrategy
 *   Move around randomly.
 *
 * All code created by Dan Cieslak, 2002.
 *
 * All code that is mine is available for anyone to use for
 * any purpose, so long as I am listed as a source.
 *
 */
public class DefaultTacticalStrategy extends TacticalStrategy
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static final double MIN_DISTANCE = 120.0;
    public static final double MAX_DISTANCE_RANGE = 60.0;
    public static final double SCAN_TIME_THRESHHOLD = 20.0;
    public static final double RADAR_WOBBLE = 22.5;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    double radarOffset = MathHelper.TWO_PI;
    boolean hitRobot = false;

    // Constructor; set up the DefaultTacticalStrategy class.
    public DefaultTacticalStrategy (StrategyBot theRobot, TacticalStrategyManager theStrategyManager)
    {
	super(theRobot, theStrategyManager);
	name = TacticalStrategyManager.DEFAULT;
    }

    // All the do-nothing methods...
    public void endTurn(){}        // any cleanup at the end of the turn
    public void onHitWall(HitWallEvent e){}           // Event handler

    /**
     * reset - Initialize the strategy
     */
    public void reset()
    {
	self.setAdjustGunForRobotTurn(true);
    }

    /**
     * startTurn
     *
     * 1. Set myPos to current location
     * 2. Clear any off-screen bullets
     */
    public void startTurn()
    {
	try {
	    self.myPos.set(self.getX(), self.getY());
	    self.myselfAsTarget.position.set(self.getX(), self.getY());
	    double velocity = self.getVelocity();
	    double heading_rad = self.getHeadingRadians();
	    double currentTime = self.getTime();
	    self.myselfAsTarget.velocity = velocity;
	    self.myselfAsTarget.heading_rad = heading_rad;
	    
	    self.myselfAsTarget.ja.update(self.myPos, velocity, heading_rad, currentTime);
	    self.myselfAsTarget.pa.update(self.myPos, velocity, heading_rad, (int)currentTime);
	    
	    Enumeration bL = self.inboundBulletList.elements();
	    IncomingBullet ib;
	    
	    while(bL.hasMoreElements()) {
		ib = (IncomingBullet)bL.nextElement();
		ib.updatePosition(self.getTime());
		if(ib.offField()) {
		    ib = (IncomingBullet)self.inboundBulletList.remove(ib.name);
		}
	    }
	    
	    Target t;
	    AdvancedBullet ab;
	    int sz = self.activeBulletsFired.size()-1;
	    for(int i=0;i<sz;i++) {
		ab = (AdvancedBullet)self.activeBulletsFired.get(i);
		//self.out.println("ab -> " + ab.targetName + " " + ab.targetMode
		//		     + " " + (int)ab.b.getX() + "," + (int)ab.b.getY());
		if(ab.expectedImpactTime == 0 && ab.fireRespPos != -1) {
		    if(self.targetList.containsKey(ab.targetName)) {  // Existing target
			t = (Target)self.targetList.get(ab.targetName);
		    } else {
			t = new Target(ab.targetName, self);
			self.targetList.put(ab.targetName, t);
		    }
		    if(t.fireResponse[ab.fireRespPos][Target.ACTUAL_LOCATION] == null) {
			Position p = new Position();
			p.x = t.position.x;
			p.y = t.position.y;
			t.fireResponse[ab.fireRespPos][Target.ACTUAL_LOCATION] = p;
			t.currentFireRespPos = Math.max(t.currentFireRespPos, ab.fireRespPos);
		    }
		    ab.expectedImpactTime--;
		}
		
		if(!ab.b.isActive()) { // hit another bot or left the field
		    AdvancedBullet ab1 = (AdvancedBullet)self.activeBulletsFired.remove(i);
		    
		    if(self.targetList.containsKey(ab1.targetName)) {  // Existing target
			t = (Target)self.targetList.get(ab1.targetName);
		    } else {
			t = new Target(ab1.targetName, self);
			self.targetList.put(ab1.targetName, t);
		    }
		    t.updateCompletedBullet(ab1);
		}
	    }
	} catch (Exception exception) {}
    }

    /**
     * setMovement - Move around randomly.
     */
    public void setMovement()
    {
	if(self.getTime() >= lastShotPassesAt)
	    myMaxVelocity = 8;

	if(this instanceof TronTacticalStrategy)
	    self.setMaxVelocity(Math.abs(self.getTurnRemaining()) > 22.5 ? myTurnVelocity : myMaxVelocity);
	else
	    self.setMaxVelocity(myMaxVelocity);

	if(self.getDistanceRemaining() == 0)
	    self.goTo(pickNewLocation(MIN_DISTANCE, MAX_DISTANCE_RANGE));
    }

    /**
     * pick a random point at least minRadius away and at most maxRadius
     * and make sure it is in the field.
     *
     * @param minRadius - smallest distance away to pick
     * @param maxRadius - largest distance away to pick
     * @return the new random coordinate
     */
    public Coordinate pickNewLocation(double minRadius, double maxRadius)
    {
	// how much x to move
	double dx = self.random.nextDouble() * (maxRadius - minRadius) + minRadius;
	// how much y to move
	double dy = self.random.nextDouble() * (maxRadius - minRadius) + minRadius;
	double width = self.getBattleFieldWidth();
	double height = self.getBattleFieldHeight();
	double myWidth = self.getWidth() / 2;

	if(self.random.nextBoolean())
	    dx = -dx;
	if(self.random.nextBoolean())
	    dy = -dy;

	Coordinate newLoc = new Coordinate(self.getX() + dx, self.getY() + dy);

	// Sanity check for wall avoidance
	if(newLoc.offField(width, height, myWidth))
	    newLoc.x -= 2 * dx;
	if(newLoc.offField(width, height, myWidth))
	    newLoc.y -= 2 * dy;
	if(newLoc.offField(width, height, myWidth))
	    newLoc.x -= 2 * dx;

	return newLoc;
    }

    /**
     * What to do when shot at.
     *
     * @param shooter - who shot at me
     */
    public void enemyFired(Target shooter, double bulletPower)
    {
	self.enemyFiredEvents++;
	//self.out.println(shooter.name + " fired " + bulletPower);
	IncomingBullet directFire = new IncomingBullet(IncomingBullet.DIRECT + 
						       shooter.name + "_" +
						       (int)self.getTime());
	directFire.set(shooter.position.x, shooter.position.y);
	directFire.velocity = MathHelper.getShotVelocity(bulletPower);
	directFire.heading_rad = shooter.position.headingTo(self.myPos);
	directFire.lastTimePosition = self.getTime();
	self.inboundBulletList.put(directFire.name, directFire);

	// if far enough away to matter...
	if(self.myPos.distanceFrom(shooter.position) > 3) { // was 300
	    IncomingBullet predictedFire = new IncomingBullet(IncomingBullet.PREDICTIVE +
							      shooter.name + "_" +
							      (int)self.getTime());
	    predictedFire.set(shooter.position.x, shooter.position.y);
	    predictedFire.velocity = MathHelper.getShotVelocity(bulletPower);
	    //predictedFire.heading_rad = shooter.position.headingTo(self.myPos);
	    try {
		double trgtDist = self.myPos.distanceFrom(shooter.position);
		predictedFire.heading_rad = self.myselfAsTarget.aimHere(shooter.position,
									bulletPower,
									trgtDist,
									self.getTime(),
									self.getBattleFieldWidth(),
									self.getBattleFieldHeight());
	    } catch (Exception e) {
		predictedFire.heading_rad = self.getHeadingRadians();
	    }
	    predictedFire.lastTimePosition = self.getTime();
	    self.inboundBulletList.put(predictedFire.name, predictedFire);
	    // What about Pattern Matching and Jiggling?
	}

    }

    /**
     * This used to setTurnRadarRightRadians(Double.MAX_VALUE)
     * but this really is not a good idea in 1v1 and probably
     * contributes to bad targeting information.
     * <br>
     * Thus, I have improved the scanning algorithm a bit, based
     * on SnippetBot code.
     */
    public void setScan()
    {
	if(!hitRobot) {
	    if(self.currentTarget == null || self.getOthers() > self.targetList.size()) {  // Full spin if no target
		radarOffset = MathHelper.TWO_PI;
	    } else {  // We have a target.
		Target t;
		
		if(self.getOthers() > 1) {  // There are other possible targets
		    t = findOldestTarget();
		} else {
		    t = self.currentTarget;
		}
		
		if(self.getTime() - t.lastTimePosition < SCAN_TIME_THRESHHOLD) {
		    t = self.currentTarget;
		    radarOffset = MathHelper.normalizeBearing(self.myPos.headingTo(t.position) - 
							      self.getRadarHeadingRadians());
		    if(radarOffset < 0) {
			radarOffset -= Math.toRadians(RADAR_WOBBLE);
		    } else {
			radarOffset += Math.toRadians(RADAR_WOBBLE);
		    }
		    radarOffset = MathHelper.normalizeBearing(radarOffset);
		} else {
		    radarOffset = MathHelper.TWO_PI;
		}
	    } // we have a currentTarget
	}
	self.setTurnRadarRightRadians(radarOffset);
	hitRobot = false;
    }

    /**
     * onScannedRobot - Peek-a-boo, I see you!
     */
    public void onScannedRobot(ScannedRobotEvent e)
    {
	Target t;
	if(self.targetList.containsKey(e.getName())) {  // Existing target
	    t = (Target)self.targetList.get(e.getName());
	    // self.out.println("Old target: " + e.getName());
	} else {
	    t = new Target(e.getName(), self);
	    self.targetList.put(e.getName(), t);
	    // self.out.println("New target: " + e.getName());
	}
	
	// Just in case this is a stale reference to a previous "self"
	t.self = self;

	double t_heading = e.getHeadingRadians();
	double t_bearing = (self.getHeadingRadians() + e.getBearingRadians()) % MathHelper.TWO_PI;
	double t_dist = e.getDistance();
	double t_x = self.getX() + Math.sin(t_bearing) * t_dist;  // target x coord
	double t_y = self.getY() + Math.cos(t_bearing) * t_dist;  // target y coord
	Coordinate t_coord = new Coordinate(t_x, t_y);
	double t_velocity = e.getVelocity();
	double t_energy = e.getEnergy();
	boolean bAntiGrav = false;
	boolean bPosHist = true;

	if(this instanceof AntiGravTacticalStrategy) {
	    bAntiGrav = true;
	}

	if((t.energy - t_energy <= MathHelper.MAX_SHOT_ENERGY) && 
	   (t.energy - t_energy >= MathHelper.MIN_SHOT_ENERGY - 0.02)) {  // .098 just to be sure...
	    enemyFired(t, t.energy - t_energy);
	}

	t.updateFromScan(e.getName() , t_coord, t_velocity , t_heading, t_energy, 
			 t_dist, t_bearing, self.getTime(), self.getEnergy(), 
			 bAntiGrav, bPosHist);
    }

    /**
     * onHitByBullet - Ouch! Quit it!
     */
    public void onHitByBullet(HitByBulletEvent e)
    {
	Target t;
	if(self.targetList.containsKey(e.getName())) {  // Existing target
	    t = (Target)self.targetList.get(e.getName());
	    // self.out.println("Old target: " + e.getName());
	} else {
	    t = new Target(e.getName(), self);
	    self.targetList.put(e.getName(), t);
	    // self.out.println("New target: " + e.getName());
	}
	// Just in case this is a stale reference to a previous "self"
	t.self = self;

	t.updateFromHitByBullet(e.getPower(), self.getTime() , e.getHeadingRadians(), 
				new Coordinate(self.getX(),self.getY()) );
	
    }

    /**
     * onHitRobot - Drove into somebody.
     */
    public void onHitRobot(HitRobotEvent e) 
    {
	Target t;
	if(self.targetList.containsKey(e.getName())) {  // Existing target
	    t = (Target)self.targetList.get(e.getName());
	    // self.out.println("Old target: " + e.getName());
	} else {
	    t = new Target(e.getName(), self);
	    self.targetList.put(e.getName(), t);
	    // self.out.println("New target: " + e.getName());
	}
	// Just in case this is a stale reference to a previous "self"
	t.self = self;
	double radarTurn = e.getBearingRadians() - MathHelper.normalizeBearing(self.getRadarHeadingRadians());
	radarOffset = MathHelper.normalizeBearing(radarTurn);
	hitRobot = true;
    }
    
    /**
     * onBulletHit - I shot the sherriff
     */
    public void onBulletHit(BulletHitEvent e)
    {
	Target t;
	if(self.targetList.containsKey(e.getName())) {  // Existing target
	    t = (Target)self.targetList.get(e.getName());
	    // self.out.println("Old target: " + e.getName());
	} else {
	    t = new Target(e.getName(), self);
	    self.targetList.put(e.getName(), t);
	    // self.out.println("New target: " + e.getName());
	}

	// Just in case this is a stale reference to a previous "self"
	t.self = self;

	Bullet b = e.getBullet();
	t.updateFromBulletHit(b.getPower() , self.getTime() ,
			      e.getEnergy(),
			      new Coordinate(b.getX() , b.getY()),
			      new Coordinate(self.getX(), self.getY()));

    }

    /**
     * onRobotDeath - Bring out yer dead
     */
    public void onRobotDeath(RobotDeathEvent e)
    {
	Target t;
	if(self.targetList.containsKey(e.getName())) {  // Existing target
	    t = (Target)self.targetList.get(e.getName());
	    // self.out.println("Old target: " + e.getName());
	} else {
	    t = new Target(e.getName(), self);
	    self.targetList.put(e.getName(), t);
	    // self.out.println("New target: " + e.getName());
	}
	t.isAlive = false;
	t.deselectTarget();
	t.energy = 0.0;
	t.deathCount++;
	t.outSurvived -= self.getOthers() + 1;
    }

    /**
     * onDeath - Not quite dead, yet.
     */
    public void onDeath(DeathEvent e)
    {
	// Update timeStrategyUsed.
	Double d = (Double)self.strategyManager.timeStrategyUsed.remove(self.currentStrategy.name);
	Double t1 = (Double)self.strategyManager.lastTimeStrategyStarted.get(self.currentStrategy.name);
	if(d != null && t1 != null) {
	    d = new Double(d.doubleValue() + (self.getTime() - t1.doubleValue()));
	    self.strategyManager.timeStrategyUsed.put(self.currentStrategy.name, d);
	}
	Enumeration en = self.targetList.elements();
	Target t;

	while(en.hasMoreElements()) {
	    t = (Target)en.nextElement();
	    if(t.isAlive) t.outSurvived += self.getOthers();
	}

    }

}
